#ifndef _RHEOLEF_S_CURL_H
#define _RHEOLEF_S_CURL_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/form_element.h"

namespace rheolef { 

/*Class:
NAME: @code{s_curl} -- curl-like operator for the Stokes stream function computation
@bfindex s_curl
@apindex P1
@apindex P2
@apindex P1d
SYNOPSIS:
    @example
      form(const space M, const space& V, "s_curl");
    @end example
DESCRIPTION:       
    @noindent
    Assembly the form associated to the
    s\_curl operator on a finite element space @code{V}:
@iftex  
    $$ 
      b(\xi,u) = \int_\Omega u. s\_curl \xi \, dx
    $$ 
@end iftex  
@ifnottex  
    @example
                /
                |
      b(xi,u) = |  u.s_curl(xi) dx
                |
                / Omega
    @end example
@end ifnottex  
    @noindent
    The M and V space may be a either @code{P1} or @code{P2} finite element space.
    The M space is scalar-valued while the V is vector-valued.
    See also @ref{form class} and @ref{space class}.

    For cartesian coordinate systems, this form coincide 
    with the usual "curl" one (see @ref{curl form}).
@cindex stream function
    In the axisymmetric case:
    @example       
                /
                | (d xi      d xi   )
      b(xi,u) = | (---- ur - ---- uz) r dr dz
                | (d z       d r    )
                / Omega
    @end example
    The b form is denoted as "s_curl", for Stokes stream function computation (see @ref{s_grad_grad form})
    as it is closely related to the "curl" operator (see @ref{curl form}), but differs
    by the r and 1/r factors, as:
    @example       
                   (       d (r xi)     d xi )
        curl(xi) = ( (1/r) -------- ; - -----)
                   (         d r        d z  )
    @end example
    while
    @example       
                   ( d xi       d xi )
      s_curl(xi) = ( ----  ;  - ---- )
                   ( d r        d z  )
    @end example
    Notice also that the differentiation is performed on the 
    xi variable here: b(xi,u)=(s_curl(xi),u) while the "curl" form brings the differentiation
    on the u vector-valued variable: (curl(u),xi), i.e. a transpose formulation.
ORIENTATION AND SIGN FIX:
    The (r,theta,z) coordinate system has positive orientation,
    thus (z,r,theta) and (z,r) are positive also.
    But (r,z,theta) and (r,z) are negative : the sign of s_curl
    is then inverted to obtain the same result as if (z,r) was used.
EXAMPLE:
    @noindent
    The following piece of code build the form
    associated to the @code{P1} approximation:
    @example
   	geo g("square");
   	space M(g, "P1");
   	space V(g, "P1", "vector");
   	form a(M, V, "s_curl");
    @end example
AUTHOR:
    LMC-IMAG, 38041 Grenoble cedex 9, France
    | Pierre.Saramito@imag.fr
DATE:
    29 february 2012
End:
*/
template<class T, class M>
class s_curl: public form_element_rep<T,M> {
public:
    typedef form_element_rep<T,M>       base;
    typedef typename base::size_type    size_type;
    typedef typename base::geo_type     geo_type;
    typedef typename base::space_type   space_type;

    s_curl (const space_type& X, const space_type& Y, const geo_type& omega, 
        const quadrature_option_type& qopt)
	: form_element_rep<T,M> (X,Y,omega,qopt) { base::initialize_all(); }

    void operator() (const geo_element& K, ublas::matrix<T>& m) const;
    void initialize () const;
    bool is_symmetric () const { return false; }
};

}// namespace rheolef
#endif // _RHEOLEF_S_CURL_H
