///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
// -----
// plane
// -----
//
//       [ 2D_00   2D_01   2D_02 ]
//  2D = [         2D_11   2D_12 ]
//       [                 2D_22 ]
//
// ordered by column: tau = 2D
//
//                  [    u0     u1     u2 ]
//
//  [   tau_00 ] =  [ 2*dx0               ]
//  [ 2*tau_01 ] =  [ 2*dx1  2*dx0        ]
//  [   tau_11 ] =  [        2*dx1        ]
//  [ 2*tau_02 ] =  [ 2*dx2         2*dx0 ]
//  [ 2*tau_12 ] =  [        2*dx2  2*dx1 ]
//  [   tau_22 ] =  [               2*dx2 ]
//
// since the tau_ij, i!=j components have a 2 factor in the
// mass matrix (see mass.cc and inv_mass.cc)
//
// ------------
// axisymmetric
// ------------
//       [ 2D_00   2D_01   0     ]
//  2D = [         2D_11   0     ]
//       [                 2D_22 ]
//
// ordered by column:
//
//                  [    u0     u1 ]
//
//  [   tau_00 ] =  [ 2*dx0        ]
//  [ 2*tau_01 ] =  [ 2*dx1  2*dx0 ]
//  [   tau_11 ] =  [        2*dx1 ]
//  [   tau_22 ] =  [  2/r         ]
//
// the weight is r*dr*dz => the 1/r will disappears for tau_22
//
#include "2D.h"
#include "rheolef/ublas_matrix_range.h"
namespace rheolef {

using namespace std;
using namespace ublas;

template<class T, class M>
void
_2D<T,M>::operator() (const geo_element& K, matrix<T>& dd) const
{
  check_macro (! base::is_on_band(), "unsupported banded level set extension");

  switch (K.dimension()) {
    case 1:
      base::build_d_dx (K, dd, 0);
      dd *= 2;
      break;
    case 2: {
      matrix<T> dx0, dx1;
      base::build_d_dx (K, dx0, 0);
      base::build_d_dx (K, dx1, 1);
      size_type ni = dx0.size1();
      size_type nj = dx0.size2();
      size_type n_comp = space_constant::n_component (	
			  space_constant::tensor, 
		          base::coordinate_dimension(),
			  base::coordinate_system());
      dd.resize (n_comp*ni, 2*nj);
      dd.clear();
      //  [   tau_00 ] =  [ 2*dx0        ]
      //  [ 2*tau_01 ] =  [ 2*dx1  2*dx0 ]
      //  [   tau_11 ] =  [        2*dx1 ]
      mr_set (dd, range(0,ni),     range(0,nj),       2.*dx0);
      mr_set (dd, range(ni,2*ni),  range(0,nj),       2.*dx1);
      mr_set (dd, range(ni,2*ni),  range(nj,2*nj),    2.*dx0);
      mr_set (dd, range(2*ni,3*ni),range(nj,2*nj),    2.*dx1);
      if (base::coordinate_system() == space_constant::cartesian) break;
      // otherwise, add the (2/r) ur vr r dr dz = 2 ur vr dr dz 
      ublas::matrix<T> m;
      base::set_use_coordinate_system_weight(false);
      base::build_scalar_mass (K, m);
      base::set_use_coordinate_system_weight(true);
      size_type k = 0;
      if (base::coordinate_system() == space_constant::axisymmetric_zr) k++;
      mr_set (dd, range(3*ni,4*ni),    range(k*nj,(k+1)*nj),	2.*m);
      break;
    }
    case 3: {
      matrix<T> dx0, dx1, dx2;
      base::build_d_dx (K, dx0, 0);
      base::build_d_dx (K, dx1, 1);
      base::build_d_dx (K, dx2, 2);
      size_type ni = dx0.size1();
      size_type nj = dx0.size2();
      dd.resize (6*ni, 3*nj);
      dd.clear();
      //  [   tau_00 ] =  [ 2*dx0               ]
      //  [ 2*tau_01 ] =  [ 2*dx1  2*dx0        ]
      //  [   tau_11 ] =  [        2*dx1        ]
      mr_set (dd, range(0,ni),     range(0,nj),       2.*dx0);
      mr_set (dd, range(ni,2*ni),  range(0,nj),       2.*dx1);
      mr_set (dd, range(ni,2*ni),  range(nj,2*nj),    2.*dx0);
      mr_set (dd, range(2*ni,3*ni),range(nj,2*nj),    2.*dx1);

      //  [ 2*tau_02 ] =  [ 2*dx2         2*dx0 ]
      //  [ 2*tau_12 ] =  [        2*dx2  2*dx1 ]
      //  [   tau_22 ] =  [               2*dx2 ]
      mr_set (dd, range(3*ni,4*ni),range(0,nj),       2.*dx2);
      mr_set (dd, range(3*ni,4*ni),range(2*nj,3*nj),  2.*dx0);
      mr_set (dd, range(4*ni,5*ni),range(nj,2*nj),    2.*dx2);
      mr_set (dd, range(4*ni,5*ni),range(2*nj,3*nj),  2.*dx1);
      mr_set (dd, range(5*ni,6*ni),range(2*nj,3*nj),  2.*dx2);
      break;
    }
    default: {
      fatal_macro ("form `2D' not supported in " << K.dimension() << "d geometry");
    }
  }
}
template<class T, class M>
void
_2D<T,M>::initialize () const
{
  base::set_n_derivative(1);

  check_macro (
	space_constant::vector == base::get_first_space().valued_tag(),
	"unsupported non-vectorial first space for `2D' form");
  check_macro (
	space_constant::tensor == base::get_second_space().valued_tag(),
	"unsupported non-tensorial second space for `2D' form");
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template class _2D<Float,sequential>;

#ifdef _RHEOLEF_HAVE_MPI
template class _2D<Float,distributed>;
#endif // _RHEOLEF_HAVE_MPI

} // namespace rheolef

