#ifndef _RHEO_VEC_H
#define _RHEO_VEC_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

# include "rheolef/array.h"
namespace rheolef {

/*Class:vec
NAME:  vec - vector in distributed environment (@PACKAGE@-@VERSION@)
SYNOPSYS:       
 STL-like vector container for a sequential or
 distributed memory machine model.
 Additional operation fom classical algebra.
EXAMPLE:
   A sample usage of the class is:
   @example
     int main(int argc, char**argv) @{
        environment distributed(argc, argv);
        vec<double> x(100, 3.14);
        dcout << x << endl;
     @}
   @end example
IMPLEMENTATION NOTE:
   Implementation use array<T,M>.
SEE ALSO: "array"(1)
AUTHORS: Pierre.Saramito@imag.fr
DATE:   19 november 1998
End:
*/

//<vec:
template <class T, class M = rheo_default_memory_model>
class vec : public array<T, M> {
public:

// typedef:

    typedef array<T, M> base;
    typedef typename base::size_type       size_type;
    typedef typename base::iterator        iterator;
    typedef typename base::const_iterator  const_iterator;

// allocator/deallocator:

    vec (const distributor& ownership,
	const T&  init_val = std::numeric_limits<T>::max());

    vec(size_type dis_size = 0,
	const T&  init_val = std::numeric_limits<T>::max());

    void resize (
        const distributor& ownership,
        const T&  init_val = std::numeric_limits<T>::max());

    void resize (
        size_type size = 0,
        const T&  init_val = std::numeric_limits<T>::max());

// expression template:

    template<typename Expr>
    vec (const Expr& expr);

    template<typename Expr>
    vec<T,M>& operator= (const Expr& expr);

    template<typename Expr>
    vec<T,M>& operator+= (const Expr& expr);

    template<typename Expr>
    vec<T,M>& operator-= (const Expr& expr);
};
//>vec:

// ----------------------------------------------------------------------------
// inlined
// ----------------------------------------------------------------------------
template <class T, class M>
inline
vec<T,M>::vec (
    const distributor& ownership,
    const T&  init_val)
  : array<T,M>(ownership,init_val)
{
}
template <class T, class M>
inline
vec<T,M>::vec (
    size_type dis_size,
    const T&  init_val)
  : array<T,M>(dis_size,init_val)
{
}
template <class T, class M>
inline
void
vec<T,M>::resize (
        const distributor& ownership,
        const T&  init_val)
{
    array<T,M>::resize (ownership, init_val);
}
template <class T, class M>
inline
void
vec<T,M>::resize (
        size_type dis_size,
        const T&  init_val)
{
    array<T,M>::resize (dis_size, init_val);
}
template <class T>
inline
idiststream&
operator >> (idiststream& ips,  vec<T,sequential>& x)
{ 
    return x.get_values(ips);
}
template <class T> 
inline
odiststream&
operator << (odiststream& ops, const vec<T,sequential>& x)
{
    return x.put_values(ops);
}
#ifdef _RHEOLEF_HAVE_MPI
template <class T>
inline
idiststream&
operator >> (idiststream& ips,  vec<T,distributed>& x)
{ 
    return x.get_values(ips); 
}
template <class T> 
inline
odiststream&
operator << (odiststream& ops, const vec<T,distributed>& x)
{
    return x.put_values(ops);
}
#endif // _RHEOLEF_HAVE_MPI

} // namespace rheolef
#endif // _RHEO_VEC_H
