///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "rheolef/piola_v1.h"
using namespace rheolef;

piola_on_quadrature::piola_on_quadrature()
 : basis_on_quadrature(),
  _Xh(),
  _use_coordinate_system_weight(true),
  _use_coordinate_system_dual_weight(false)
{
}
piola_on_quadrature::piola_on_quadrature (const quadrature& quad, const space& Xh)
 : basis_on_quadrature(quad, Xh.get_p1_transformation()),
  _Xh(Xh),
  _use_coordinate_system_weight(true),
  _use_coordinate_system_dual_weight(false)
{
}
void
piola_on_quadrature::set(const quadrature& quad, const space& Xh)
{
  basis_on_quadrature::set (quad, Xh.get_p1_transformation());
  _Xh = Xh;
}
piola_on_quadrature::size_type
piola_on_quadrature::coordinate_dimension() const
{
  return _Xh.get_global_geo().dimension();
}
void
piola_on_quadrature::set_use_coordinate_system_weight (bool use)
{
    _use_coordinate_system_weight = use;
}
bool
piola_on_quadrature::use_coordinate_system_weight () const
{
    return _use_coordinate_system_weight;
}
void
piola_on_quadrature::set_use_coordinate_system_dual_weight (bool use)
{
    _use_coordinate_system_dual_weight = use;
}
bool
piola_on_quadrature::use_coordinate_system_dual_weight () const
{
    return _use_coordinate_system_dual_weight;
}
Float
piola_on_quadrature::weight_coordinate_system (const geo_element& K, size_type q) const
{
    if (_Xh.get_geo().coordinate_system_type() == fem_helper::cartesian
        || ! use_coordinate_system_weight()) {
  	return 1;
}
    point xq = transformation (K, q);
    switch (_Xh.get_geo().coordinate_system_type()) {
      case fem_helper::axisymmetric_rz: 
	  return use_coordinate_system_dual_weight() ? 1/xq[0] : xq[0];
      case fem_helper::axisymmetric_zr:
	  return use_coordinate_system_dual_weight() ? 1/xq[1] : xq[1];
      default: {
	fatal_macro ("unsupported coordinate system `" 
	<<  fem_helper::coordinate_system_name (_Xh.get_global_geo().coordinate_system_type()) << "'");
	return 0;
      }
    }
}
// F (hat_xq)
point
piola_on_quadrature::transformation (const geo_element& K, size_type q) const
{
    size_type nt = _Xh.get_p1_transformation().size(K);
    check_macro (nt == K.size(), "unsupported transformation " << _Xh.get_p1_transformation().name());
    basis_on_quadrature::const_iterator tr = basis_on_quadrature::begin (K, q);
    point xq (0,0,0);
    size_type coord_d = coordinate_dimension();
    geo::const_iterator_node p = _Xh.get_geo().begin_vertex();
    for (size_type k = 0; k < nt; k++, tr++) {
      Float coeff = *tr;
      const point& ak = p[K[k]];
      for (size_type i = 0; i < coord_d; i++) {
        xq[i] += ak[i] * coeff;
      }
    }
    return xq;
}
// DF (hat_xq)
void
piola_on_quadrature::jacobian_transformation (const geo_element& K, size_type q, tensor& DF) const
{
    size_type nt = _Xh.get_p1_transformation().size(K);
    check_macro (nt == K.size(), "unsupported transformation " << basis_on_quadrature::get_basis().name());
    DF.fill (0);
    geo::const_iterator_node p = _Xh.get_geo().begin_vertex();
    basis_on_quadrature::const_iterator_grad grad_tr
     = basis_on_quadrature::begin_grad (K, q);
    size_type coord_d = coordinate_dimension();
    for (size_type k = 0; k < nt; k++, grad_tr++)
      ::cumul_otimes (DF, p[K[k]], *grad_tr, coord_d, K.dimension());
}
Float
piola_on_quadrature::det_jacobian_transformation (const tensor& DF, size_type map_dim) const
{
    if (coordinate_dimension() == map_dim) {
      return DF.determinant (map_dim);
    }
    /* surface jacobian: references:
     * Spectral/hp element methods for CFD
     * G. E. M. Karniadakis and S. J. Sherwin
     * Oxford university press
     * 1999
     * page 165
     */
    switch (map_dim) {
      case 0: return 1;
      case 1: return norm(DF.col(0));
      case 2: return norm(vect(DF.col(0), DF.col(1)));
      default:
        error_macro ("det_jacobian_piola_transformation: unsupported element dimension "
	    << map_dim << " in " << coordinate_dimension() << "D mesh.");
    }
}
Float
piola_on_quadrature::det_jacobian_transformation (const geo_element& K, size_type q) const
{
    tensor DF;
    jacobian_transformation (K, q, DF);
    return det_jacobian_transformation (DF, K.dimension());
} 

// piola without quadrature
piola::piola()
 : _Xh(),
  _use_coordinate_system_weight(true),
  _use_coordinate_system_dual_weight(false)
{
}
piola::piola (const space& Xh)
 : _Xh(Xh),
  _use_coordinate_system_weight(true),
  _use_coordinate_system_dual_weight(false)
{
}
void
piola::set(const space& Xh)
{
  _Xh = Xh;
}
piola::size_type
piola::coordinate_dimension() const
{
  return _Xh.get_global_geo().dimension();
}
void
piola::set_use_coordinate_system_weight (bool use)
{
    _use_coordinate_system_weight = use;
}
bool
piola::use_coordinate_system_weight () const
{
    return _use_coordinate_system_weight;
}
void
piola::set_use_coordinate_system_dual_weight (bool use)
{
    _use_coordinate_system_dual_weight = use;
}
bool
piola::use_coordinate_system_dual_weight () const
{
    return _use_coordinate_system_dual_weight;
}
Float
piola::weight_coordinate_system (const geo_element& K, const point& hat_x) const
{
    if (_Xh.get_geo().coordinate_system_type() == fem_helper::cartesian
        || ! use_coordinate_system_weight()) {
  	return 1;
	}
    point x = transformation (K, hat_x);
    switch (_Xh.get_geo().coordinate_system_type()) {
      case fem_helper::axisymmetric_rz: 
	  return use_coordinate_system_dual_weight() ? 1/x[0] : x[0];
      case fem_helper::axisymmetric_zr:
	  return use_coordinate_system_dual_weight() ? 1/x[1] : x[1];
      default: {
	fatal_macro ("unsupported coordinate system `" 
	<<  fem_helper::coordinate_system_name (_Xh.get_global_geo().coordinate_system_type()) << "'");
	return 0;
      }
    }
}
// F (hat_x)
point
piola::transformation (const geo_element& K, const point& hat_x) const
{
    basis b = _Xh.get_p1_transformation();
    size_type nt = b.size(K);
    check_macro (nt == K.size(), "unsupported transformation " << _Xh.get_p1_transformation().name());
    point x (0,0,0);
    size_type coord_d = coordinate_dimension();
    geo::const_iterator_node p = _Xh.get_geo().begin_vertex();
    std::vector<Float> p1;
    b.eval(K, hat_x, p1);
    std::vector<Float>::const_iterator tr=p1.begin();
    for (size_type k = 0; k < nt; k++, tr++) {
      Float coeff = *tr;
      const point& ak = p[K[k]];
      for (size_type i = 0; i < coord_d; i++) {
        x[i] += ak[i] * coeff;
      }
    }
    return x;
}
// DF (hat_x)
void
piola::jacobian_transformation (const geo_element& K, const point& hat_x, tensor& DF) const
{
    basis b = _Xh.get_p1_transformation();
    size_type nt = b.size(K);
    check_macro (nt == K.size(), "unsupported transformation " << b.name());
    DF.fill (0);
    geo::const_iterator_node p = _Xh.get_geo().begin_vertex();
    size_type coord_d = coordinate_dimension();
    for (size_type k = 0; k < nt; k++)
     {
      point grad = b.grad_eval(K, k, hat_x);
      ::cumul_otimes (DF, p[K[k]], grad, coord_d, K.dimension());
     }
}
Float
piola::det_jacobian_transformation (const tensor& DF, size_type map_dim) const
{
    if (coordinate_dimension() == map_dim) {
      return DF.determinant (map_dim);
    }
    /* surface jacobian: references:
     * Spectral/hp element methods for CFD
     * G. E. M. Karniadakis and S. J. Sherwin
     * Oxford university press
     * 1999
     * page 165
     */
    switch (map_dim) {
      case 0: return 1;
      case 1: return norm(DF.col(0));
      case 2: return norm(vect(DF.col(0), DF.col(1)));
      default:
        error_macro ("det_jacobian_piola_transformation: unsupported element dimension "
	    << map_dim << " in " << coordinate_dimension() << "D mesh.");
    }
}
Float
piola::det_jacobian_transformation (const geo_element& K, const point& hat_x) const
{
    tensor DF;
    jacobian_transformation (K, hat_x, DF);
    return det_jacobian_transformation (DF, K.dimension());
} 
/*
void
piola::piola_basis (const basis& b, const geo_element& K, const point& hat_x, vector<Float>& p) const
{
    std::vector<Float> hat_p(n);
    tensor DF;
    jacobian_transformation (K, hat_x, DF);
    p = piola_basis(K, hat_x, hat_p.begin(), hat_p.end(), p.begin());
}
*/
Float
piola::reference_eval (
    const geo_element& K,
    const tiny_vector<Float>& dof,
    const point& hat_x,
    size_type i_comp) const
{
    basis b= _Xh.get_basis();
    size_type n = b.size(K);
    tensor DF(0.);
    std::vector<Float> hat_p(n), p(n);
    b.eval (K, hat_x, hat_p);
    if (b.family()!=reference_element::Lagrange)
     {
       jacobian_transformation (K, hat_x, DF);
     }
    std::vector<Float>::const_iterator ip 
    	= piola_basis(_Xh, K, b, DF, hat_p.begin(), hat_p.end(), p.begin());
    Float value = 0;
    for (size_type i = 0; i < n; i++)
	value += dof[i]*ip[i];
    return value;
}
Float
piola::reference_d_dxi_eval (
    size_type i,
    const geo_element& K,
    const tiny_vector<Float>& dof,
    const point& hat_x,
    size_type i_comp) const
{
    basis b=_Xh.get_basis();
    size_type n = b.size(K);
    std::vector<point> hat_grad_p (n), grad_p(n);
    b.grad_eval (K, hat_x, hat_grad_p);
    tensor DF(0.);
    if (b.family()!=reference_element::Lagrange)
     {
       jacobian_transformation (K, hat_x, DF);
     }
    std::vector<point>::const_iterator igrad_p 
    	= piola_basis(_Xh, K, b, DF, hat_grad_p.begin(), hat_grad_p.end(), grad_p.begin());
    Float value = 0;
    for (size_type j = 0; j < n; j++)
	value += dof[i]*igrad_p[j][i];
    return value;
}
