///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// visualisation of geometries using gnuplot
//
// authors:
//    	Pierre.Saramito@imag.fr
//
// date: 12 may 1997
//
// content:
//      * graphic output (with options; see geoplot(1))
//            - gnuplot
//            - plotmtv
//            - atom
//            - x3d
//            - vtk
//      * get vtk polydata file (line or surface )
//        as for a cut in a mesh
//
#include "rheolef/georep.h"
#include "rheolef/geo-visu-aux.h"
#include "rheolef/iorheo.h"
#include "rheolef/rheostream.h" // i/o utility
using namespace rheolef;
using namespace std;

int
georep::gnuplot1d (const string& basename, bool execute, bool clean, bool verbose) const
{
  string filename = basename + ".plot";
  ofstream plot (filename.c_str());

  if (verbose) clog << "! file \"" << filename << "\" created.\n";
  plot << "#!gnuplot\n";
  plot << "set title \"" << basename << ": " 
       << size() << " elements, " << n_node() << " vertices\"\n";
  Float delta = 0.1*(_xmax[0] - _xmin[0]);
  plot << "plot [" << _xmin[0]-delta << ":" << _xmax[0]+delta << "][-1:1] "
       << "\"" << basename << ".gdat\" title \"" << basename << "\" with lp \\\n";

  filename = basename + ".gdat";
  ofstream gdat (filename.c_str());
  if (verbose) clog << "! file \"" << filename << "\" created.\n";
  int digits10 = numeric_limits<Float>::digits10;
  gdat << setprecision(digits10);
  //
  // plot elements
  //
  const_iterator iter_e = begin();
  const_iterator last_e = end();
  while (iter_e != last_e)
      gnuplot_element (gdat, *iter_e++, begin_node(), 2);
  gdat.close();
  //
  // plot domains, by decreasing dimension order
  //
  for (size_type dim = dimension(); dim+1 >= 1; dim--) {
    domlist_type::const_iterator last = _domlist.end();
    for (domlist_type::const_iterator iter = _domlist.begin(); iter != last; iter++) {
      const domain& dom = *iter;
      if (dom.dimension() != dim) continue;
      if (dom.size() == 0) continue;
      filename = basename + "." + dom.name() + ".gdat";
      gdat.open(filename.c_str());
      if (verbose) clog << "! file \"" << filename << "\" created.\n";
      string mode = (dim == 1) ? "lp" : "points pointsize 3";
      plot << ", \"" << filename << "\" title \"" << dom.name() << "\" with " << mode << " \\\n";
      for (domain::const_iterator p = dom.begin(); p != dom.end(); p++) {
	  gnuplot_element (gdat, *p, begin_node(), 2);
      }
      gdat.close();
    }
  }
  plot << endl;
  //
  // end of plot
  //
  plot << endl;
  plot << "pause -1 \"<return>\"\n";
  plot.close();
  //
  // run gnuplot
  //
  int status = 0;
  char command [1000];
  if (execute) {
      sprintf (command, "gnuplot -persist %s.plot", basename.c_str());
      if (verbose) clog << "! " << command << endl;
      cin.sync();
      status = system (command);
  }
  //
  // clear gnuplot data
  //
  if (clean) {
      sprintf (command, "/bin/rm -f %s.plot %s.gdat", basename.c_str(), basename.c_str());
      if (verbose) clog << "! " << command << endl;
      status = system (command);
      domlist_type::const_iterator last = _domlist.end();
      for (domlist_type::const_iterator iter = _domlist.begin(); iter != last; iter++) {
          sprintf (command, "/bin/rm -f %s.%s.gdat", basename.c_str(), (*iter).name().c_str());
          if (verbose) clog << "! " << command << endl;
          status = system (command);
      }
  }
  return status;
}
int
georep::gnuplot2d (const string& basename, bool execute, bool clean, bool verbose) const
{
  char filename [1000];
  sprintf (filename, "%s.plot", basename.c_str());
  ofstream plot (filename);
  string lines = (map_dimension() == 2) ? "lines" : "linespoints";
  if (verbose) clog << "! file \"" << filename << "\" created.\n";
  plot << "#!gnuplot" << endl
       << "set title \"" << basename << ": " 
       << size() << " elements, " << n_node() << " vertices\"" << endl
       << "set size ratio -1" << endl
       << "plot \"" << basename << ".gdat\" title \"" << basename
       << "\" with " << lines << " \\" << endl
       ;
  sprintf (filename, "%s.gdat", basename.c_str());
  ofstream gdat (filename);
  if (verbose) clog << "! file \"" << filename << "\" created.\n";
  int digits10 = numeric_limits<Float>::digits10;
  gdat << setprecision(digits10);
  //
  // plot elements
  //
    // isoline in 2D space: draw continuous path for each connex component
 
  if (map_dimension() == 2) {
    const_iterator iter_e = begin();
    const_iterator last_e = end();
    while (iter_e != last_e)
      gnuplot_element (gdat, *iter_e++, begin_node(), 2);
  } else if (map_dimension() == 1) {
    vector<Float> y;
    hazel_1d paths (*this);
    georep::const_iterator K = begin();
    georep::const_iterator_vertex x = begin_vertex();
    for (size_t i = 0; i < paths.size(); i++) {
      gdat << "# " << i << "-th connex component" << endl;
      size_t prev_idx = size_t(-1);
      for (size_t j = 0; j < paths[i].size(); j++) {
        const geo_element& Kj = K[paths[i][j]];
	// fix bad edge orientation:
        size_t iloc0;
        if (j != 0) {
          iloc0 = (Kj[0] == prev_idx) ? 0 : 1;
        } else {
          if (paths[i].size() == 1) {
            iloc0 = 0;
          } else {
            const geo_element& K1 = K[paths[i][1]];
            iloc0 = (Kj[1] == K1[0] || Kj[1] == K1[1]) ? 0 : 1;
          }
        }
        size_t iloc1 = (iloc0 +1) % 2;
        string sgn = (iloc0==0)?"+":"-";
        gdat << "# K" << Kj.index() << " = (" << Kj[iloc0]
             << "," << Kj[iloc1] << ")" << sgn << endl;

        gdat << x[Kj[iloc0]][0] << " " << x[Kj[iloc0]][1] << endl;
        if (j == paths[i].size()-1) {
          gdat << x[Kj[iloc1]][0] << " " << x[Kj[iloc1]][1] << endl;
        }
        prev_idx = Kj[iloc1];
      }
      gdat << endl << endl;
    }
  }
  gdat.close();
  //
  // plot domains
  //
  domlist_type::const_iterator last = _domlist.end();
  for (domlist_type::const_iterator iter = _domlist.begin(); iter != last; iter++) {
      const domain& dom = *iter;
      if (dom.size() == 0) continue;
      sprintf (filename, "%s.%s.gdat", basename.c_str(), dom.name().c_str());
      gdat.open(filename);
      if (verbose) clog << "! file \"" << filename << "\" created.\n";
      string mode;
      if (dom.dimension() == 0) {
	mode = "points pointsize 3";
      } else if (dom.dimension() != dimension()) {
	mode = "lines linewidth " + itos(dimension() - dom.dimension() + 1);
      } else {
	mode = "lines lw 3 lt 3";
      }
      plot << ", \"" << filename << "\" title \"" << dom.name() << "\" with " << mode << " \\\n";
      for (domain::const_iterator p = dom.begin(); p != dom.end(); p++) {
	  gnuplot_element (gdat, *p, begin_node(), 2);
      }
      gdat.close();
  }
  //
  // end of plot
  //
  plot << endl;
  plot << "pause -1 \"<return>\"\n";
  plot.close();
  //
  // run gnuplot
  //
  int status = 0;
  char command [1000];
  if (execute) {
      sprintf (command, "gnuplot -persist %s.plot", basename.c_str());
      if (verbose) clog << "! " << command << endl;
      status = system (command);
  }
  //
  // clear gnuplot data
  //
  if (clean) {
      sprintf (command, "/bin/rm -f %s.plot %s.gdat", basename.c_str(), basename.c_str());
      if (verbose) clog << "! " << command << endl;
      status = system (command);
      for (domlist_type::const_iterator iter = _domlist.begin(); iter != last; iter++) {
          sprintf (command, "/bin/rm -f %s.%s.gdat", basename.c_str(), (*iter).name().c_str());
          if (verbose) clog << "! " << command << endl;
          status = system (command);
      }
  }
  return status;
}
int
georep::gnuplot2d (const string& basename, plot_options& opt) const
{
  char filename [1000];
  sprintf (filename, "%s.plot", basename.c_str());
  ofstream plot (filename);

  if (opt.verbose) clog << "! file \"" << filename << "\" created.\n";
  plot << "#!gnuplot\n";
  if (!opt.bare)
  	plot << "set title \"" << basename << ": " 
    	   << size() << " elements, " << n_node() << " vertices\"\n";
  else
  	plot << "set nokey" << endl;
  if (opt.empty) 
   {
   	plot << "set noborder" << endl;
   	plot << "set noxtics" << endl;
   	plot << "set noytics" << endl;
   }
  if (opt.xmax==opt.xmin) 
   { 
   	opt.xmin=xmin(); 
	opt.xmax=xmax(); 
	if (opt.xmirror) opt.xmin[0]=-opt.xmax[0];
	if (opt.ymirror) opt.xmin[1]=-opt.xmax[1];
	if (opt.border != Float(0)) {
		point diag = opt.xmax-opt.xmin;
		Float w=opt.border*::sqrt(sqr(diag[0])+sqr(diag[1]));
		opt.xmax[0]+=w;
		opt.xmax[1]+=w;
		opt.xmin[0]-=w;
		opt.xmin[1]-=w;
	 }
   }
  if (opt.equal_scales)
   {
   	plot << "set size ratio -1" << endl;
   }

  if (opt.postscript && opt.noxplot)  
   {
	plot << "set output '" << basename << ".eps'" << endl;
	write_gnuplot_postscript_options(plot,opt);
   }
  ofstream gdat;
  string coma=",";
  plot << "set xrange ["<<opt.xmin[0]<<":"<<opt.xmax[0]<<"]" << endl; // to allow mouse-zoom
  plot << "set yrange ["<<opt.xmin[1]<<":"<<opt.xmax[1]<<"]" << endl;
  plot << "plot ";
  if (!opt.silhouette)
   {
	plot << "\"" << basename << ".gdat\" title \"" << basename << "\" with lines linetype 1 \\\n";
	if (opt.xmirror) plot << ", \"" << basename << ".gdat\" using (-$1):2 notitle with lines linetype 1 \\\n";
	if (opt.ymirror) plot << ", \"" << basename << ".gdat\" using 1:(-$2) notitle with lines linetype 1 \\\n";
	if (opt.xmirror && opt.ymirror) plot << ", \"" << basename << ".gdat\" using (-$1):(-$2) notitle with lines linetype 1 \\\n";

	sprintf (filename, "%s.gdat", basename.c_str());
	gdat.open (filename);
	if (opt.verbose) clog << "! file \"" << filename << "\" created.\n";
	int digits10 = numeric_limits<Float>::digits10;
	gdat << setprecision(digits10);
	//
	// plot elements
	//
	const_iterator iter_e = begin();
	const_iterator last_e = end();
	while (iter_e != last_e)
	    gnuplot_element (gdat, *iter_e++, begin_node(), 2);
	gdat.close();
   }
  else { plot << "\\\n"; coma=""; }
  //
  // plot domains
  //
  domlist_type::const_iterator last = _domlist.end();
  size_t i=1;
  if (!opt.meshonly)
  for (domlist_type::const_iterator iter = _domlist.begin(); iter != last; iter++) {
      const domain& dom = *iter;
      i++;
      if (dom.size() == 0) continue;
      sprintf (filename, "%s.%s.gdat", basename.c_str(), dom.name().c_str());
      gdat.open(filename);
      if (opt.verbose) clog << "! file \"" << filename << "\" created.\n";
      string mode;
      if (dom.dimension() == 0) {
	mode = "points linetype " + itos(i) + " pointsize 3";
      } else if (dom.dimension() != dimension()) {
	mode = "lines linewidth " + itos(dimension() - dom.dimension() + 1) + " linetype " + itos(i) ;
      } else {
	mode = "lines linetype " + itos(i);
      }
      plot << coma << " \"" << filename << "\" title \"" << dom.name() << "\" with " << mode << " \\\n";
      if (opt.xmirror) 
  	plot << ", \"" << filename << "\" using (-$1):2 notitle with " << mode << " \\\n";	
      if (opt.ymirror) 
  	plot << ", \"" << filename << "\" using 1:(-$2) notitle with " << mode << " \\\n";	
      if (opt.xmirror && opt.ymirror)
  	plot << ", \"" << filename << "\" using (-$1):(-$2) notitle with " << mode << " \\\n";	
      coma = ",";
      for (domain::const_iterator p = dom.begin(); p != dom.end(); p++) {
	  gnuplot_element (gdat, *p, begin_node(), 2);
      }
      gdat.close();
  }
  //
  // end of plot
  //
  plot << endl; // because of the trailing "\"
  plot << endl; 
  if (opt.execute) plot << "pause -1 \"<return>\"\n";
  if (opt.postscript && (!opt.noxplot)) 
   {
	plot << "set output '" << basename << ".eps'" << endl;
   	write_gnuplot_postscript_options(plot,opt);
	plot << "replot";
   }
  plot.close();
  //
  // run gnuplot
  //
  int status = 0;
  char command [1000];
  if (opt.execute) {
      sprintf (command, "gnuplot -persist %s.plot", basename.c_str());
      if (opt.verbose) clog << "! " << command << endl;
      status = system (command);
  }
  //
  // clear gnuplot data
  //
  /*
  if (opt.clean) {
      sprintf (command, "/bin/rm -f %s.plot %s.gdat", basename.c_str(), basename.c_str());
      if (opt.verbose) clog << "! " << command << endl;
      status = system (command);
      iter = _domlist.begin();
      while (iter != last) {
          sprintf (command, "/bin/rm -f %s.%s.gdat", basename.c_str(), (*iter).name().c_str());
          if (opt.verbose) clog << "! " << command << endl;
          status = system (command);
          ++iter;
      }
  }
  */
  return status;
}
int
georep::gnuplot3d (const string& basename, bool execute, bool clean, bool verbose) const
{
  char filename [1000];
  sprintf (filename, "%s.plot", basename.c_str());
  ofstream plot (filename);
  if (verbose) clog << "! file \"" << filename << "\" created.\n";
  plot << "#!gnuplot\n";
  plot << "set parametric\n";
  plot << "set nokey\n";
  plot << "xrot=50\n";
  plot << "yrot=10\n";
  plot << "zrot=10\n";
  plot << "set title \"" << basename << ": " 
       << size() << " elements, " << n_node() << " vertices\"\n";
  plot << "splot \"" << basename << ".gdat\" title \"" << basename << "\" with lines \\\n";

  sprintf (filename, "%s.gdat", basename.c_str());
  ofstream gdat (filename);
  if (verbose) clog << "! file \"" << filename << "\" created.\n";
  int digits10 = numeric_limits<Float>::digits10;
  gdat << setprecision(digits10);
  //
  // plot elements
  //
  const_iterator iter_e = begin();
  const_iterator last_e = end();
  while (iter_e != last_e)
      gnuplot_element (gdat, *iter_e++, begin_node(), 3);
  gdat.close();
  //
  // plot domains
  //
  domlist_type::const_iterator last = _domlist.end();
  for (domlist_type::const_iterator iter = _domlist.begin(); iter != last; iter++) {
      const domain& dom = *iter;
      if (dom.size() == 0) continue;
      sprintf (filename, "%s.%s.gdat", basename.c_str(), dom.name().c_str());
      gdat.open(filename);
      if (verbose) clog << "! file \"" << filename << "\" created.\n";
      string mode;
      if (dom.dimension() == 0) {
	mode = "points pointsize 3";
      } else if (dom.dimension() != dimension()) {
	mode = "lines linewidth " + itos(dimension() - dom.dimension() + 1);
      } else {
	mode = "lines";
      }
      plot << ", \"" << filename << "\" title \"" << dom.name() << "\" with " << mode << " \\\n";
      for (domain::const_iterator p = dom.begin(); p != dom.end(); p++) {
	  gnuplot_element (gdat, *p, begin_node(), 3);
      }
      gdat.close();
  }
  //
  // end of plot
  //
  plot << endl;
  plot << "load \"" << basename << ".rot\"\n";
  plot << "pause -1 \"<return>\"\n";
  plot.close();
  //
  // rotation
  //
  sprintf (filename, "%s.rot", basename.c_str());
  ofstream rot (filename);
  if (verbose) clog << "! file \"" << filename << "\" created.\n";
  rot << "yrot=(yrot+1)%360\n";
  rot << "set view xrot,yrot\n";
  rot << "replot\n";
  rot << "reread\n";
  rot.close();
  //
  // run gnuplot
  //
  int status = 0;
  char command [1000];
  if (execute) {
      sprintf (command, "gnuplot -persist %s.plot", basename.c_str());
      if (verbose) clog << "! " << command << endl;
      status = system (command);
  }
  //
  // clear gnuplot data
  //
  if (clean) {
      sprintf (command, "/bin/rm -f %s.plot %s.gdat %s.rot", 
			basename.c_str(), basename.c_str(), basename.c_str());
      if (verbose) clog << "! " << command << endl;
      status = system (command);
      for (domlist_type::const_iterator iter = _domlist.begin(); iter != last; iter++) {
          sprintf (command, "/bin/rm -f %s.%s.gdat", basename.c_str(), (*iter).name().c_str());
          if (verbose) clog << "! " << command << endl;
          status = system (command);
      }
  }
  return status;
}
