///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
//    point : set of coordinates
//
//    author: Pierre.Saramito@imag.fr
//
#include "rheolef/tensor.h"
#include "rheolef/tensor-eig3.h"
#include "rheolef/tensor-svd-algo.h"
using namespace rheolef;
using namespace std;
namespace rheolef { 


// output

ostream& 
tensor::put (ostream& out, size_type d) const
{
    switch (d) {
    case 0 : return out;
    case 1 : return out << _x[0][0];
    case 2 : return out << "[" << _x[0][0] << ", " << _x[0][1] << ";\n" 
	                << " " << _x[1][0] << ", " << _x[1][1] << "]";
    default: return out << "[" << _x[0][0] << ", " << _x[0][1] << ", " << _x[0][2] << ";\n" 
	                << " " << _x[1][0] << ", " << _x[1][1] << ", " << _x[1][2] << ";\n"
	                << " " << _x[2][0] << ", " << _x[2][1] << ", " << _x[2][2] << "]";
    }
}
istream&
tensor::get (istream& in)
{
    for (size_type i = 0; i < 3; i++) for (size_type j = 0; j < 3; j++)
        _x[i][j] = 0;
    char c;
    if (!in.good()) return in;
    in >> ws >> c;
    // no tensor specifier: 
    // 1d case: "3.14"
    if (c != '[') {
        in.unget();
        return in >> _x[0][0];
    }
    // has a tensor specifier, as:
    //  	[  23, 17; 25, 2]
    in >> _x[0][0] >> ws >> c;
    if (c == ']') return in; // 1d case
    if (c != ',') error_macro ("invalid tensor input: expect `,'");
    in >> _x[0][1];
    in >> ws >> c;
    size_type d = 3;
    if (c == ',') d = 3;
    else if (c == ';') d = 2;
    else error_macro ("invalid tensor input: expect `,' or `;'");
    if (d == 3) {
        in >> _x[0][2] >> ws >> c;
        if (c != ';') error_macro ("invalid tensor input: expect `;'");
    }
    in >> _x[1][0] >> ws >> c;
    if (c != ',') error_macro ("invalid tensor input: expect `,'");
    in >> _x[1][1] >> ws >> c;
    if (d == 2) {
        if (c != ']') error_macro ("invalid tensor input: expect `]'");
	return in;
    }
    // d == 3
    if (c != ',') error_macro ("invalid tensor input: expect `,'");
    in >> _x[1][2] >> ws >> c;
    if (c != ';') error_macro ("invalid tensor input: expect `;'");
    in >> _x[2][0] >> ws >> c;
    if (c != ',') error_macro ("invalid tensor input: expect `,'");
    in >> _x[2][1] >> ws >> c;
    if (c != ',') error_macro ("invalid tensor input: expect `,'");
    in >> _x[2][2] >> ws >> c;
    if (c != ']') error_macro ("invalid tensor input: expect `]'");
    return in;
}
bool operator == (const tensor& a, const tensor& b)
{
    typedef tensor::size_type size_type;
    for (size_type i = 0; i < 3; i++) for (size_type j = 0; j < 3; j++)
        if (a._x[i][j] != b._x[i][j]) return false;
    return true;
}
tensor operator - (const tensor& a)
{
    tensor b;
    typedef tensor::size_type size_type;
    for (size_type i = 0; i < 3; i++) for (size_type j = 0; j < 3; j++)
        b._x[i][j] = - a._x[i][j];
    return b;
}
tensor operator + (const tensor& a, const tensor& b)
{
    tensor c;
    typedef tensor::size_type size_type;
    for (size_type i = 0; i < 3; i++) for (size_type j = 0; j < 3; j++)
        c._x[i][j] = a._x[i][j] + b._x[i][j];
    return c;
}
tensor operator - (const tensor& a, const tensor& b)
{
    tensor c;
    typedef tensor::size_type size_type;
    for (size_type i = 0; i < 3; i++) for (size_type j = 0; j < 3; j++)
        c._x[i][j] = a._x[i][j] - b._x[i][j];
    return c;
}
tensor operator * (Float k, const tensor& a)
{
    tensor b;
    typedef tensor::size_type size_type;
    for (size_type i = 0; i < 3; i++) for (size_type j = 0; j < 3; j++)
        b._x[i][j] = k * a._x[i][j];
    return b;
}
point operator * (const tensor& a, const point& x)
{
    point y;
    typedef tensor::size_type size_type;
    for (size_type i = 0; i < 3; i++) for (size_type j = 0; j < 3; j++)
        y [i] += a._x[i][j] * x[j];
    return y;
}
point operator * (const point& x, const tensor& a)
{
    point y;
    typedef tensor::size_type size_type;
    for (size_type i = 0; i < 3; i++) for (size_type j = 0; j < 3; j++)
        y [i] += a._x[j][i] * x[j];
    return y;
}
tensor
trans (const tensor& a, tensor::size_type d) {
    tensor b;
    typedef tensor::size_type size_type;
    for (size_type i = 0; i < d; i++) for (size_type j = 0; j < d; j++)
        b._x[i][j] = a._x[j][i];
    return b;
}
tensor
inv (const tensor& a, tensor::size_type d) {
  tensor b;
  typedef tensor::size_type size_type;
  switch (d) {
    case 0: 
      break;
    case 1: 
      b(0,0) = 1/a(0,0);
      break;
    case 2: {
      Float det = a.determinant(2);
      b(0,0) =   a(1,1)/det;
      b(1,1) =   a(0,0)/det;
      b(0,1) = - a(0,1)/det;
      b(1,0) = - a(1,0)/det;
      break;
    }
    case 3:
    default: {
      Float det = a.determinant(3);
      b(0,0) = (a(1,1)*a(2,2) - a(1,2)*a(2,1))/det;
      b(0,1) = (a(0,2)*a(2,1) - a(0,1)*a(2,2))/det;
      b(0,2) = (a(0,1)*a(1,2) - a(0,2)*a(1,1))/det;
      b(1,0) = (a(1,2)*a(2,0) - a(1,0)*a(2,2))/det;
      b(1,1) = (a(0,0)*a(2,2) - a(0,2)*a(2,0))/det;
      b(1,2) = (a(0,2)*a(1,0) - a(0,0)*a(1,2))/det;
      b(2,0) = (a(1,0)*a(2,1) - a(1,1)*a(2,0))/det;
      b(2,1) = (a(0,1)*a(2,0) - a(0,0)*a(2,1))/det;
      b(2,2) = (a(0,0)*a(1,1) - a(0,1)*a(1,0))/det;
      break;
    }
  }
  return b;
}
tensor operator * (const tensor& a, const tensor& b)
{
    tensor c;
    typedef tensor::size_type size_type;
    for (size_type i = 0; i < 3; i++) for (size_type j = 0; j < 3; j++)
      for (size_type k = 0; k < 3; k++)
        c._x[i][j] += a._x[i][k] * b._x[k][j];
    return c;
}
Float dotdot (const tensor& a, const tensor & b) {
    Float r = 0;
    typedef tensor::size_type size_type;
    for (size_type i = 0; i < 3; i++) for (size_type j = 0; j < 3; j++)
        r += a._x[i][j] * b._x[i][j];
    return r;
}
Float
tensor::determinant (size_type d) const
{
    switch (d) {
      case 1 : return _x[0][0];
      case 2 : return _x[0][0]*_x[1][1] - _x[0][1]*_x[1][0];
      case 3 : return _x[0][0]*(_x[1][1]*_x[2][2] - _x[1][2]*_x[2][1])
	            - _x[0][1]*(_x[1][0]*_x[2][2] - _x[1][2]*_x[2][0])
	            + _x[0][2]*(_x[1][0]*_x[2][1] - _x[1][1]*_x[2][0]);
      default: {
	error_macro ("determinant: unexpected dimension " << d);
      }
    }
}
// t += a otimes b
void
cumul_otimes (tensor& t, const point& a, const point& b, size_t na, size_t nb)
{
  for (size_t i = 0; i < na; i++)
    for (size_t j = 0; j < nb; j++)
      t(i,j) += a[i] * b[j];
}
point
tensor::row(size_type i) const
{
    point r;
    r[0] = _x[i][0];
    r[1] = _x[i][1];
    r[2] = _x[i][2];
    return r;
}
point
tensor::col(size_type j) const
{
    point c;
    c[0] = _x[0][j];
    c[1] = _x[1][j];
    c[2] = _x[2][j];
    return c;
}
bool
invert_3x3 (const tensor& A, tensor& result) {
  Float det = determinant(A);
  if (1+det == 1) return false;
  Float invdet = 1.0/det;
  result(0,0) =  (A(1,1)*A(2,2)-A(2,1)*A(1,2))*invdet;
  result(1,0) = -(A(0,1)*A(2,2)-A(0,2)*A(2,1))*invdet;
  result(2,0) =  (A(0,1)*A(1,2)-A(0,2)*A(1,1))*invdet;
  result(0,1) = -(A(1,0)*A(2,2)-A(1,2)*A(2,0))*invdet;
  result(1,1) =  (A(0,0)*A(2,2)-A(0,2)*A(2,0))*invdet;
  result(2,1) = -(A(0,0)*A(1,2)-A(1,0)*A(0,2))*invdet;
  result(0,2) =  (A(1,0)*A(2,1)-A(2,0)*A(1,1))*invdet;
  result(1,2) = -(A(0,0)*A(2,1)-A(2,0)*A(0,1))*invdet;
  result(2,2) =  (A(0,0)*A(1,1)-A(1,0)*A(0,1))*invdet;
  return true;
}
// ------------------------------------------
// eigenvalues: the 3D case
// ------------------------------------------
static point eig3x3 (const tensor& a, tensor& q)
{
    Float A[3][3];
    for (size_t i = 0; i < 3; i++)
      for (size_t j = 0; j < 3; j++)
	A[i][j] = a(i,j);
    Float Q[3][3];
    Float D[3];
    eigen_decomposition(A, Q, D);
    point d;
    for (size_t i = 0; i < 3; i++) {
      d[i] = D[i];
      for (size_t j = 0; j < 3; j++)
        q(i,j) = Q[i][j];
    }
    return d;
}
// ------------------------------------------
// eigenvalues: the 2D case
// ------------------------------------------
static point eig2x2 (const tensor& a, tensor& q)
{
    // TODO: check if a is symetric !
    point d (0,0,0);
    if (a(0,1) == 0) {
      // a is already diagonal
      if (a(0,0) >= a(1,1)) {
        d[0] = a(0,0);
        d[1] = a(1,1);
        q(0,0) = q(1,1) = 1;
        q(0,1) = q(1,0) = 0;
      } else { // swap column 0 & 1:
        d[1] = a(0,0);
        d[0] = a(1,1);
        q(0,0) = q(1,1) = 0;
        q(0,1) = q(1,0) = 1;
      }
      return d;
    }
    // here a(0,1) != 0
    Float discr = sqr(a(1,1) - a(0,0)) + 4*sqr(a(0,1));
    Float trace = a(0,0) + a(1,1);
    d[0] = (trace + ::sqrt(discr))/2; 
    d[1] = (trace - ::sqrt(discr))/2;
    Float lost_precision = 1e-6;
    if (fabs(d[0]) + lost_precision*fabs(d[1]) == fabs(d[0])) { // d[1] == 0 up to machine precision
	d[1] = 0;
    }
    Float c0 = (d[0]-a(0,0))/a(0,1);
    Float n0 = ::sqrt(1+sqr(c0));
    q(0,0) =  1/n0;
    q(1,0) = c0/n0;
    Float c1 = (d[1]-a(0,0))/a(0,1);
    Float n1 = ::sqrt(1+sqr(c1));
    q(0,1) =  1/n1;
    q(1,1) = c1/n1;
    return d;
}
// ------------------------------------------
// eigenvalues: general case
// ------------------------------------------
point
tensor::eig (tensor& q, size_t d) const
{
  switch (d) {
    case 1 : {
      point d;
      q(0,0) = 1; d[0] = operator()(0,0); return d;
    }
    case 2 : return eig2x2 (*this, q);
    default: return eig3x3 (*this, q);
  }
}
point 
tensor::eig (size_t d) const
{
  tensor q;
  return eig (q, d);
}
// ------------------------------------------
// svd: the 3D case
// ------------------------------------------
point
tensor::svd (tensor& u, tensor& v, size_t dim) const
{
  Float U[3][3];
  for (size_t i = 0; i < dim; i++)
    for (size_t j = 0; j < dim; j++)
      U[i][j] = operator()(i,j);
  Float V[3][3];
  Float S[3];
  svdcmp (U, S, V, dim, dim, Float(0));
  // Sort eigenvalues and corresponding vectors.
  for (size_t i = 0; i < dim-1; i++) {
    size_t k = i;
    Float p = S[i];
    for (size_t j = i+1; j < dim; j++) {
      if (S[j] > p) {
        k = j;
        p = S[j];
      }
    }
    if (k != i) {
      S[k] = S[i];
      S[i] = p;
      for (size_t j = 0; j < dim; j++) {
        p = U[j][i];
        U[j][i] = U[j][k];
        U[j][k] = p;
        p = V[j][i];
        V[j][i] = V[j][k];
        V[j][k] = p;
      }
    }
  }
  // copy to tensor and point
  point s;
  for (size_t i = 0; i < dim; i++) {
    s[i] = S[i];
    for (size_t j = 0; j < dim; j++) {
      u(i,j) = U[i][j];
      v(i,j) = V[i][j];
    }
  }
  return s;
}
}// namespace rheolef
