///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
// 
// reference element connectivity tables:
//  - input : #include human-readable C++ code "hexa.h"..."triangle.h" etc
//  - output: code efficiently usable internaly
//            by the "reference_element" class
//
// author: Pierre.Saramito@imag.fr
//
// date: 24 may 2010
//
#include "rheolef/point.h"
using namespace rheolef;
using namespace std;


// --------------------------------------------------------------------
// global table definitions
// --------------------------------------------------------------------
static const size_t max_size_t = size_t(-1);
static const size_t max_element_type = 7;
static const char* element_name [max_element_type] = { "p", "e", "t", "q", "T", "P", "H" };

static const size_t max_subgeo_vertex = 8;  // for hexa : number of vertex
static const size_t max_n_subgeo      = 12; // for hexa : number of edges

size_t table_dimension [max_element_type];
size_t table_n_subgeo  [max_element_type][4];
size_t table_subgeo    [max_element_type][4][max_n_subgeo][max_subgeo_vertex];
size_t table_n_edge_per_face_max [max_element_type];
Float  table_vertex    [max_element_type][max_subgeo_vertex][3];
Float  table_measure   [max_element_type];

// --------------------------------------------------------------------
// generic fcts and specific includes
// --------------------------------------------------------------------
void init_generic_0d (size_t E, size_t d, size_t nv, Float meas) {
  table_dimension [E] = d;
  table_measure   [E] = meas;
  table_n_subgeo  [E][d] = 1;
  for (size_t k = d+1; k < 4; k++) {
   table_n_subgeo [E][k] = 0;
  }
  table_subgeo [E][d][0][min(nv,max_subgeo_vertex-1)] = max_size_t;
  for (size_t i = 0; i < nv; i++) {
   table_subgeo [E][d][0][i] = i;
  }
}
template <int Dimension>
void init_generic_1d (size_t E, size_t d, size_t nv, const Float v[][Dimension], Float meas) {
  init_generic_0d (E, d, nv, meas);
  table_n_subgeo [E][0] = nv;
  for (size_t i = 0; i < nv; i++) {
   table_subgeo [E][0][i][0] = i;
   table_subgeo [E][0][i][min(size_t(1),max_subgeo_vertex-1)] = max_size_t;
   for (size_t k = 0; k < d; k++) {
    table_vertex [E][i][k] = v[i][k];
   }
  }
}
template <int Dimension>
void init_generic_2d (size_t E, size_t d, size_t nv, const Float v[][Dimension],
	size_t ne, const size_t e[][2], Float meas) {
  init_generic_1d (E, d, nv, v, meas);
  table_n_subgeo [E][1] = ne;
  for (size_t i = 0; i < ne; i++) {
   table_subgeo [E][1][i][min(size_t(2),max_subgeo_vertex-1)] = max_size_t;
   for (size_t k = 0; k < 2; k++) {
    table_subgeo [E][1][i][k] = e[i][k];
   }
  }
}
template <size_t NEdgePerFaceMax>
void init_generic_3d (size_t E, size_t d, size_t nv, const Float v[][3], size_t nf, const size_t f[][NEdgePerFaceMax],
	size_t ne, const size_t e[][2], Float meas) {
  init_generic_2d (E, d, nv, v, ne, e, meas);
  table_n_edge_per_face_max [E] = NEdgePerFaceMax;
  table_n_subgeo [E][2] = nf;
  for (size_t i = 0; i < nf; i++) {
   table_subgeo [E][2][i][min(NEdgePerFaceMax,max_subgeo_vertex-1)] = max_size_t;
   for (size_t k = 0; k < NEdgePerFaceMax; k++) {
    table_subgeo [E][2][i][k] = f[i][k];
   }
  }
}
void init_p (size_t p) {
#include "point.icc"
  init_generic_0d (p, dimension, 1, measure);
}
void init_e (size_t e) {
#include "edge.icc"
  init_generic_1d (e, dimension, n_vertex, vertex, measure);
}
void init_t (size_t t) {
#include "triangle.icc"
  init_generic_2d (t, dimension, n_vertex, vertex, n_edge, edge, measure);
}
void init_q (size_t q) {
#include "quadrangle.icc"
  init_generic_2d (q, dimension, n_vertex, vertex, n_edge, edge, measure);
}
void init_T (size_t T) {
#include "tetra.icc"
  init_generic_3d (T, dimension, n_vertex, vertex, n_face, face, n_edge, edge, measure);
}
void init_P (size_t P) {
#include "prism.icc"
  init_generic_3d (P, dimension, n_vertex, vertex, n_face, face, n_edge, edge, measure);
}
void init_H (size_t H) {
#include "hexa.icc"
  init_generic_3d (H, dimension, n_vertex, vertex, n_face, face, n_edge, edge, measure);
}
void licence () {
// --------------------------------------------------------------------
// c++ code generation
// --------------------------------------------------------------------
  cout << "// file automaticaly generated by \"cxx_reference_element.cc\"" << endl
       << "//" << endl
       << "///" << endl
       << "/// This file is part of Rheolef." << endl
       << "///" << endl
       << "/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>" << endl
       << "///" << endl
       << "/// Rheolef is free software; you can redistribute it and/or modify" << endl
       << "/// it under the terms of the GNU General Public License as published by" << endl
       << "/// the Free Software Foundation; either version 2 of the License, or" << endl
       << "/// (at your option) any later version." << endl
       << "///" << endl
       << "/// Rheolef is distributed in the hope that it will be useful," << endl
       << "/// but WITHOUT ANY WARRANTY; without even the implied warranty of" << endl
       << "/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the" << endl
       << "/// GNU General Public License for more details." << endl
       << "///" << endl
       << "/// You should have received a copy of the GNU General Public License" << endl
       << "/// along with Rheolef; if not, write to the Free Software" << endl
       << "/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA" << endl
       << "/// " << endl
       << "/// =========================================================================" << endl
       ;
}
void cxx_reference_element_header() {

  licence ();
  // print enum symbol = e, t, q, ..
  cout << "typedef enum {" << endl;
  for (size_t i = 0; i < max_element_type; i++) {
    cout << "\t" << element_name[i] << " = " << i << "," << endl;
  }
  cout << "\tmax_size = " << max_element_type << endl
       << "} enum_type;" << endl
       ;
  cout << "static const size_type max_n_subgeo      = " << max_n_subgeo << ";" << endl
       << "static const size_type max_subgeo_vertex = " << max_subgeo_vertex << ";" << endl
    ;
}
void cxx_reference_element_body () {
  licence ();
  cout << "static const reference_element::size_type" << endl
       << "    N = numeric_limits<reference_element::size_type>::max();" << endl
       << endl
    ;
  // element dimension : 1,2 3
  cout << "const reference_element::size_type" << endl
       << "reference_element::_dimension [reference_element::max_size] = {" << endl
    ;
  for (size_t E = 0; E < max_element_type; E++) {
    cout << "\t" << table_dimension[E];
    if (E+1 != max_element_type) cout << ",";
    cout << endl;
  }
  cout << "};" << endl;

  // print char symbol = 'e', 't', ..
  cout << "const char" << endl
       << "reference_element::_name [reference_element::max_size] = {" << endl
    ;
  for (size_t E = 0; E < max_element_type; E++) {
    cout << "\t'" << element_name[E] << "'";
    if (E+1 != max_element_type) cout << ",";
    cout << endl;
  }
  cout << "};" << endl;

  // print subgeo sizes
  cout << "const reference_element::size_type" << endl
       << "reference_element::_n_subgeo [reference_element::max_size][4] = {" << endl
    ;
  for (size_t E = 0; E < max_element_type; E++) {
    cout << "    {" << endl;
    for (size_t d = 0; d < 4; d++) {
      cout << "\t" << table_n_subgeo [E][d];
      if (d+1 != 4) cout << ",";
      cout << endl;
    }
    cout << "    }";
    if (E+1 != max_element_type) cout << ",";
    cout << endl;
  }
  cout << "};" << endl;
  // use a heap for global vertices indexes
  // for efficient geo_element C++ implementation:
  // print heap size:
  cout << "const reference_element::size_type" << endl
       << "reference_element::_heap_size [reference_element::max_size] = {" << endl
    ;
  for (size_t E = 0; E < max_element_type; E++) {
    size_t heap_size = 0;
    for (size_t d = 0; d < 4; d++) { 
      heap_size += table_n_subgeo[E][d];
    }
    cout << "\t" << heap_size;
    if (E+1 != max_element_type) cout << ",";
    cout << endl;
  }
  cout << "};" << endl;
  // print heap offset
  cout << "const reference_element::size_type" << endl
       << "reference_element::_heap_offset [reference_element::max_size][4] = {" << endl
    ;
  for (size_t E = 0; E < max_element_type; E++) {
    cout << "    {" << endl;
    for (size_t d = 0; d < 4; d++) { 
      size_t offset = 0;
      for (size_t l = d+1; l < 4; l++) { 
        offset += table_n_subgeo[E][l];
      }
      cout << "\t" << offset;
      if (d+1 != 4) cout << ",";
      cout << endl;
    }
    cout << "    }";
    if (E+1 != max_element_type) cout << ",";
    cout << endl;
  }
  cout << "};" << endl;
  // print subgeo number of vertices
  size_t total_size = max_element_type*4*max_n_subgeo;
  size_t total_i = 0;
  cout << "const reference_element::size_type" << endl
       << " reference_element::_subgeo_size [" << total_size << "] =" << endl
       << "{" << endl
    ;
  for (size_t E = 0; E < max_element_type; E++) {
    for (size_t d = 0; d < 4; d++) { 
      size_t nsubgeo = table_n_subgeo[E][d];
      for (size_t k = 0; k < nsubgeo; k++) {
        size_t subgeo_n_vertex = 0;
        for (size_t i = 0; i < max_subgeo_vertex
	  && table_subgeo[E][d][k][i] != max_size_t; i++) {
            subgeo_n_vertex++;
        }
        cout << "\t" << subgeo_n_vertex;
        if (total_i+1 != total_size) cout << ",";
        cout << " // " << total_i+1 <<endl;
	total_i++;
      }
      for (size_t k = nsubgeo; k < max_n_subgeo; k++) {
        cout << "\tN";
        if (total_i+1 != total_size) cout << ",";
        cout << " // " << total_i+1 <<endl;
	total_i++;
      }
    }
  }
  cout << "};" << endl;
  // print subgeo local vertices
  // GNU C++ seems to be buggy for 4-th dimension static array initializer: flat it
  total_size = max_element_type*4*max_n_subgeo*max_subgeo_vertex,
  total_i = 0;
  cout << "const reference_element::size_type" << endl
       << "reference_element::_subgeo_local_vertex [" << total_size << "] =" << endl
       << "{" << endl
    ;
  for (size_t E = 0; E < max_element_type; E++) {
    for (size_t d = 0; d < 4; d++) { 
      size_t nsubgeo = table_n_subgeo[E][d];
      for (size_t k = 0; k < nsubgeo; k++) {
        cout << "      // (" << element_name[E] << " " << d << " " << k+1 << ")" << endl;
        bool finished = false;
        for (size_t i = 0; i < max_subgeo_vertex; i++) {
	  if (!finished && table_subgeo[E][d][k][i] != max_size_t) {
            cout << "\t" << table_subgeo[E][d][k][i];
          } else {
            cout << "\tN";
	    finished = true;
          }
          if (total_i+1 != total_size) cout << ",";
          cout << " // " << total_i+1 <<endl;
	  total_i++;
        }
      }
      for (size_t k = nsubgeo; k < max_n_subgeo; k++) {
        cout << "      // (" << element_name[E] << " " << d << " " << k+1 << ")" << endl;
        for (size_t i = 0; i < max_subgeo_vertex; i++) {
          cout << "\tN";
          if (total_i+1 != total_size) cout << ",";
          cout << " // " << total_i+1 <<endl;
	  total_i++;
        }
      }
    }
  }
  cout << "};" << endl;
}
// --------------------------------------------------------------------
int main(int argc, char**argv) {
// --------------------------------------------------------------------
  size_t E = 0;
  init_p (E++);
  init_e (E++);
  init_t (E++);
  init_q (E++);
  init_T (E++);
  init_P (E++);
  init_H (E++);
  if (argc > 1 && argv[1] == string("-h")) {
    cxx_reference_element_header();
  } else {
    cxx_reference_element_body();
  }
}
