#ifndef _RHEO_BASIS_SYMBOLIC_HERMITE_H
#define _RHEO_BASIS_SYMBOLIC_HERMITE_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/basis_symbolic.h"

namespace rheolef { 
class basis_symbolic_hermite_on_geo: public basis_symbols {
public:

// typedefs:

	typedef std::vector<int>::size_type size_type;
	typedef GiNaC::ex                   polynom_type;
	typedef GiNaC::ex                   value_type;
	typedef std::pair<GiNaC::symbol,GiNaC::ex>        vvpair;
	typedef std::pair<vvpair,basic_point<GiNaC::ex> > vpair;
	struct  end_type {};
	typedef reference_element::dof_family_type dof_family_type;

// allocators:

	basis_symbolic_hermite_on_geo() 
	 : basis_symbols(), _name("unamed"), 
	   _hat_K(), _node(0), _poly(0), 
	   _basis(), _grad_basis(), _hessian_basis() {}

// accessors:

	size_type size() const { return _basis.size(); }
	size_type size_family(reference_element::dof_family_type family) const; 
	size_type dimension() const { return _hat_K.dimension(); }
	const reference_element& hat_K() const { return _hat_K; }
	std::string name() const { return _name; }
	const basic_point<GiNaC::ex>& node (size_type i) const { return _node[i]; }
	const polynom_type& polynom (size_type i) const { return _poly[i]; }

// modifiers:

	void set_name(std::string str) { _name = str; }
	void set_hat_K(reference_element::enum_type t) {
		_hat_K.set_type(t); }
	basic_point<GiNaC::ex>& node (size_type i) { return _node[i]; }
	polynom_type& polynom (size_type i) { return _poly[i]; }
	
	void resize(size_type n) { 
		_node.resize(n);  _poly.resize(n);
		_basis.resize(n); _grad_basis.resize(n); 
		basic_point<GiNaC::ex> ini_g;
		basic_point<basic_point<GiNaC::ex> > ini_h(ini_g,ini_g,ini_g);
		_hessian_basis.resize(n, ini_h); }

	void add_polynom (const polynom_type& p) { 
		_poly.push_back(p); }
	basis_symbolic_hermite_on_geo& operator<< (const polynom_type& p) {
		add_polynom (p); return *this; }

	void add_node (const basic_point<GiNaC::ex>& x) { 
		_node.push_back(x); }
	void add_node (const Float& x0, const Float& x1=0, const Float& x2=0) { 
		add_node(basic_point<GiNaC::ex>(x0,x1,x2)); } 
	void add_node_derivative (const GiNaC::symbol& var, const GiNaC::ex& sign, const basic_point<GiNaC::ex>& x) 
		//! the value at that dof will be d u / d( var )
	 { 
		_node_derivative.push_back(x); 
		_derivative_variable.push_back(var);
		_derivative_sign.push_back(sign);
	 }
	void add_node_derivative (const GiNaC::symbol& var, const GiNaC::ex& sign, 
		const Float& x0, const Float& x1=0, const Float& x2=0) 
	 { 
		add_node_derivative(var, sign, basic_point<GiNaC::ex>(x0,x1,x2)); 
	 } 
	basis_symbolic_hermite_on_geo& operator<< (const basic_point<GiNaC::ex>& x) {
		add_node (x); return *this; }
	basis_symbolic_hermite_on_geo& operator<< (const vpair& vx) {
		add_node_derivative (vx.first.first, vx.first.second, vx.second); return *this; }
	
	void make_node_basis ();
	basis_symbolic_hermite_on_geo& operator<< (end_type(*)()) {
		make_node_basis(); return *this; }

// method:

	value_type eval (const polynom_type& p,
		const basic_point<polynom_type>& x, size_type d = 3) const;
	GiNaC::matrix confluent_vandermonde_matrix (
		const std::vector<polynom_type>& p, size_type d = 3) const; 

// outputs:

	void put_cxx_header (std::ostream& out) const;
	void put_cxx_body   (std::ostream& out) const;

// utility:

	polynom_type indexed_symbol (polynom_type expr0) const;

// data:

protected:
	std::string                       	_name;
	reference_element                       _hat_K;
	std::vector<basic_point<GiNaC::ex> >	_node;
	std::vector<basic_point<GiNaC::ex> >	_node_derivative;
	std::vector<GiNaC::symbol>		_derivative_variable;
	std::vector<GiNaC::ex>			_derivative_sign;
	std::vector<polynom_type>  		_poly;
	std::vector<dof_family_type>  		_poly_family;
	std::vector<polynom_type>  		_basis;
	std::vector<basic_point<polynom_type> > _grad_basis;
	std::vector<basic_point<basic_point<polynom_type> > > _hessian_basis;
};
class basis_symbolic_hermite 
 : public basis_symbols, public std::vector<basis_symbolic_hermite_on_geo> {
public:

// typedefs:

	typedef basis_symbolic_hermite_on_geo::size_type size_type;
	typedef basis_symbolic_hermite_on_geo::polynom_type polynom_type;
	typedef basis_symbolic_hermite_on_geo::end_type end_type;
	typedef basis_symbolic_hermite_on_geo::vvpair vvpair;
	typedef basis_symbolic_hermite_on_geo::vpair vpair;
	typedef basis_symbolic_hermite_on_geo::dof_family_type dof_family_type;

// allocators:

	basis_symbolic_hermite(std::string nam, size_type deg)
	: basis_symbols(),
	  std::vector<basis_symbolic_hermite_on_geo>(reference_element::max_size),
	  _name(nam),
	  _degree(deg)
	  {
	      GiNaC::Digits = 2*std::numeric_limits<Float>::digits10;
	      for (size_type i = 0; i < reference_element::max_size; i++) {
	        operator[](i).x = x;
	        operator[](i).y = y;
	        operator[](i).z = z;
		operator[](i).set_hat_K(reference_element::enum_type(i));
		operator[](i).set_name(nam);
	      }
	  }

// accessors:

	size_type degree() const { return _degree; }

	const basis_symbolic_hermite_on_geo& on(
	    reference_element::enum_type t) const {
	    return operator[] (t);
	}
	basis_symbolic_hermite_on_geo& on(
	    reference_element::enum_type t) {
	    return operator[] (t);
	}
	basis_symbolic_hermite_on_geo& on(char t) {
	    reference_element hat_K;
	    hat_K.set_name(t);
	    return operator[] (hat_K.type()); }

	std::string name() const { return _name; }

// modifiers:

	void set_name(std::string str) { _name = str; }

// syntax helpers:

	static polynom_type poly (const polynom_type& p) { return p; } 
	static basic_point<GiNaC::ex> node (const basic_point<GiNaC::ex>& x) {
		return x; }
	static basic_point<GiNaC::ex> node (
		const GiNaC::ex& x0, const GiNaC::ex& x1=0, const GiNaC::ex& x2=0) { 
		return basic_point<GiNaC::ex>(x0,x1,x2); } 
	static vpair node_derivative (const GiNaC::symbol& var, const GiNaC::ex& sign, const basic_point<GiNaC::ex>& x) {
		return vpair(vvpair(var,sign),x); }
	static vpair node_derivative (const GiNaC::symbol& var,const GiNaC::ex& sign, 
		const GiNaC::ex& x0, const GiNaC::ex& x1=0, const GiNaC::ex& x2=0) { 
		return vpair(vvpair(var,sign),basic_point<GiNaC::ex>(x0,x1,x2)); } 
	static end_type end () { return end_type(); } 

// outputs:

	void put_cxx_header(std::ostream& out) const;
	void put_cxx_body  (std::ostream& out) const;
	void put_cxx_main (int argc, char **argv) const;

// data:
protected:
	std::string   _name;
	size_type     _degree;
};
}// namespace rheolef
#endif // _RHEO_BASIS_SYMBOLIC_HERMITE_H
