#ifndef BOOST_PP_IS_ITERATING
#ifndef _RHEOLEF_POLYMORPHIC_DATA_VECTOR_H
#define _RHEOLEF_POLYMORPHIC_DATA_VECTOR_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================

#include "rheolef/distributor.h"
#include "rheolef/vector_of_iterator.h"
#include "rheolef/array.h"	// for pair<size_t,T> serialization

#include <map>
#include <typeinfo>
#include <boost/array.hpp>
#include <boost/mpl/vector.hpp>
#include <boost/mpl/size.hpp>
#include <boost/mpl/for_each.hpp>
#include <boost/mpl/at.hpp>

#include <boost/preprocessor/repetition.hpp>
#include <boost/preprocessor/arithmetic/sub.hpp>
#include <boost/preprocessor/iteration/iterate.hpp>

namespace rheolef {
namespace mpl = boost::mpl;

// forward declaration:
template <class T, class V, int NV> class polymorphic_data_map;

/// @brief static memory area handler for polymorphic_array
template <class T, class V, int NV>
struct polymorphic_data_vector {};

// --------------------------------------------------------------------------------
// new_entry<Tk>() returns a pointer Tk* newly allocated into the data structure
// via a push_back. We need a specialisation of a template member function
// into a template class. This is not allowed... So, we use the same trick as 
// for the "reserve(n)" member: we use a template class-function, with specialisation.
//
template <class Tk, class T, class V, int NV>
struct my_new_entry_t {
  Tk* operator() (polymorphic_data_vector<T,V,NV>& x) {
    error_macro ("unexpected type " << typeid_name_macro(Tk));
    return 0;
  }
};
// ----------------------------------------------------------
// run here the recursive inclusion of this file:
// ----------------------------------------------------------

#ifndef _RHEOLEF_POLYMORPHIC_MAX_SIZE
#define _RHEOLEF_POLYMORPHIC_MAX_SIZE 7
#endif

#define BOOST_PP_ITERATION_LIMITS (0, _RHEOLEF_POLYMORPHIC_MAX_SIZE)
#define BOOST_PP_FILENAME_1    "rheolef/polymorphic_data_vector.h" // this file
#include BOOST_PP_ITERATE()

} // namespace rheolef
#endif // _RHEOLEF_POLYMORPHIC_DATA_VECTOR_H
#else // BOOST_PP_IS_ITERATING
// ----------------------------------------------------------
// here, this file is recursively included with growing "N" :
// ----------------------------------------------------------
#define N    BOOST_PP_ITERATION()

template <class T, class V>
struct polymorphic_data_vector<T,V,N> {

// typedefs:

    typedef T                                    element_type;
    typedef typename std::vector<int>::size_type size_type;
    static const size_t  _n_variant  = mpl::size<V>::value; // should be = N

#define _RHEOLEF_typedef(z,k,unused)			\
    typedef typename mpl::at_c<V,k>::type T##k;
    BOOST_PP_REPEAT(N, _RHEOLEF_typedef, ~)
#undef _RHEOLEF_typedef

// cstors:

#define _RHEOLEF_stack_initializer(z,k,unused)			\
        _stack_##k(), 
#define _RHEOLEF_stack_reserve_n(z,k,unused)			\
        _stack_##k.reserve (n);
#define _RHEOLEF_stack_resize_n(z,k,unused)			\
	_stack_##k.resize (n);
#define _RHEOLEF_stack_reserve_variant(z,k,unused)		\
        _stack_##k.reserve (variant_size[k]);
#define _RHEOLEF_stack_resize_variant(z,k,unused)		\
        _stack_##k.resize (variant_size[k]);

    polymorphic_data_vector (size_type n=0) : 
	BOOST_PP_REPEAT(N, _RHEOLEF_stack_initializer, ~)
        _variant(),
        _counter(),
        _size(0)
    {
	std::fill (_counter.begin(), _counter.end(), 0);
	BOOST_PP_REPEAT(N, _RHEOLEF_stack_reserve_n, ~)
        _variant.reserve (n);
    }
    void resize (size_type n) {
	BOOST_PP_REPEAT(N, _RHEOLEF_stack_resize_n, ~)
	_variant.resize (n);
    }
    polymorphic_data_vector (const boost::array<size_type,_n_variant>& variant_size) : 
	BOOST_PP_REPEAT(N, _RHEOLEF_stack_initializer, ~)
        _variant(),
        _counter(),
        _size(0)
    {
	std::fill (_counter.begin(), _counter.end(), 0);
	BOOST_PP_REPEAT(N, _RHEOLEF_stack_reserve_variant, ~)
        size_type total_size = 0;
        for (size_type k = 0; k < _n_variant; k++) {
            total_size += variant_size[k];
        }
        _variant.reserve (total_size);
    }
    void resize (const boost::array<size_type,_n_variant>& variant_size) {
	BOOST_PP_REPEAT(N, _RHEOLEF_stack_resize_variant, ~)
        size_type total_size = 0;
        for (size_type k = 0; k < _n_variant; k++) {
            total_size += variant_size[k];
        }
	_variant.resize (total_size);
    }

#undef _RHEOLEF_stack_initializer
#undef _RHEOLEF_stack_reserve_n
#undef _RHEOLEF_stack_reserve_variant
#undef _RHEOLEF_stack_resize_n
#undef _RHEOLEF_stack_resize_variant

// accessors & modifiers:

    size_type size () const { return _size; }
    size_type size (size_type k) const { return _counter[k]; }

    // to use when x.resize(n) has not been called (e.g. a sort of x.push_back(value))
    // you can use x.reserve(n)
    template <class Tk> Tk* new_entry ();

    // to use when x.resize() has been called (e.g. a sort of x[i] = value)
    void set (size_type i,                const T& value, size_type k_variant);
    void set (size_type i, size_type idx, const T& value, size_type k_variant);

    const size_type* begin_sizes () const { return _counter.begin(); }
    void build (const polymorphic_data_vector<T,V,N>& pool);
    void build (const polymorphic_data_map<T,V,N>&);
    void reset () {
	std::fill (_counter.begin(), _counter.end(), 0);
	_size = 0;
    };
#define _RHEOLEF_begin_accessor(z,k,unused)							\
          std::pair<size_type,T##k>* begin_##k()       { return _stack_##k.begin().operator->(); }  	\
    const std::pair<size_type,T##k>* begin_##k() const { return _stack_##k.begin().operator->(); }

    BOOST_PP_REPEAT(N, _RHEOLEF_begin_accessor, ~)

#define _RHEOLEF_update_counter(z,k,unused)	\
	_counter[k] = _stack_##k.size();

    void update_sizes () { 
        BOOST_PP_REPEAT(N, _RHEOLEF_update_counter, ~)
	_size = 0;
	for (size_type k = 0; k < _n_variant; k++) {
          _size += _counter[k];
	}
    }
#undef _RHEOLEF_begin_accessor
#undef _RHEOLEF_update_counter

// data:

#define _RHEOLEF_stack_declare(z,k,unused)			\
    std::vector<std::pair<size_type,T##k> >   _stack_##k;

    BOOST_PP_REPEAT(N, _RHEOLEF_stack_declare, ~)
    std::vector<size_type>   		    _variant;
    boost::array<size_type,_n_variant>	    _counter;
    size_type				    _size;

#undef _RHEOLEF_stack_declare
};
// specialization for Tk
#define _RHEOLEF_new_entry(z,k,unused)						\
template <class T, class V>							\
struct my_new_entry_t <typename polymorphic_data_vector<T,V,N>::T##k, T, V, N> {\
  typedef typename polymorphic_data_vector<T,V,N>::T##k T##k;			\
  typedef typename polymorphic_data_vector<T,V,N>::size_type size_type;		\
  T##k* operator() (polymorphic_data_vector<T,V,N>& x) {			\
	x._stack_##k.push_back(std::pair<size_type,T##k> ());			\
        x._stack_##k[x._counter[k]].first = x._size;				\
	T##k* ptr = &(x._stack_##k[x._counter[k]].second);			\
	x._variant.push_back(k);						\
	x._variant[x._size] = k;						\
	x._counter[k]++;							\
	x._size++;								\
	return ptr;								\
  }										\
};
BOOST_PP_REPEAT(N, _RHEOLEF_new_entry, ~)
#undef _RHEOLEF_new_entry

template <class T, class V>
template <class Tk>
inline
Tk*
polymorphic_data_vector<T,V,N>::new_entry ()
{
  my_new_entry_t<Tk,T,V,N> new_entry_f;
  return new_entry_f (*this);
}
// to use when resize() has been called
template <class T, class V>
void
polymorphic_data_vector<T,V,N>::set (size_type i, size_type idx, const T& value, size_type k) 
{
    switch (k) {
#define _RHEOLEF_case(z,k,unused)						\
      case k: {									\
	size_type ik = _counter[k];						\
        _stack_##k [ik].first  = idx;						\
	_stack_##k [ik].second = value;						\
	_variant[i] = k;							\
	_counter[k]++;								\
	break;									\
      }
BOOST_PP_REPEAT(N, _RHEOLEF_case, ~)
#undef _RHEOLEF_case
      default :{
	error_macro ("unexpected type #"<<k<<"/"<<_n_variant<<" variant of "<<typeid_name_macro(T));
      }
    }
}
// to use when resize() has been called
template <class T, class V>
void
polymorphic_data_vector<T,V,N>::set (size_type i, const T& value, size_type k) 
{
    set (i, i, value, k);
}
template <class T, class V>
void
polymorphic_data_vector<T,V,N>::build (const polymorphic_data_vector<T,V,N>& pool)
{
#define _RHEOLEF_stack_copy(z,k,unused)							\
	_stack_##k.resize (pool._stack_##k.size());					\
	copy (pool._stack_##k.begin(), pool._stack_##k.end(), _stack_##k.begin());

    BOOST_PP_REPEAT(N, _RHEOLEF_stack_copy, ~)

    _variant.resize (pool._variant.size());
    copy (pool._variant.begin(), pool._variant.end(), _variant.begin());
    copy (pool._counter.begin(), pool._counter.end(), _counter.begin());
    _size = pool._size;

#undef _RHEOLEF_stack_copy
}

#endif // BOOST_PP_IS_ITERATING
