///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/parstream.h"
#include "rheolef/rheostream.h" // scatch()
using namespace std;
namespace rheolef {
// ----------------------------------------------------------------------------
// global variables
// ----------------------------------------------------------------------------

iparstream pcin  (cin);
oparstream pcout (cout);
oparstream pcerr (cerr);

// ----------------------------------------------------------------------------
/// @brief parallel version of scatch(istream&,string)
// ----------------------------------------------------------------------------
bool
par_scatch (iparstream& ips, const communicator& comm, std::string ch)
{
  // TODO: comm is in ips.comm()
  typedef std::size_t size_type;
  size_type io_proc = iparstream::io_proc();
  size_type my_proc = comm.rank();
  bool status;
  if (my_proc == io_proc) {
    status = scatch(ips.is(),ch);
  }
#ifdef _RHEOLEF_HAVE_MPI
  mpi::broadcast (comm, status, io_proc);
#endif // _RHEOLEF_HAVE_MPI
  return status;
}
// --------------------------------------------------------------
// intput
// --------------------------------------------------------------
/// @brief This routine returns the rank of a process that can perform i/o
iparstream::size_type
iparstream::io_proc() {
#ifndef _RHEOLEF_HAVE_MPI
    return 0;
#else // _RHEOLEF_HAVE_MPI
    boost::optional<int> opt_io_proc = environment::io_rank();
    check_macro (opt_io_proc, "no process can perform i/o");
    int io_proc = opt_io_proc.get();
    if (io_proc == mpi::any_source) {
        /// every process can perform I/O using the standard facilities:
        io_proc = 0;
    }
    return size_type(io_proc);
#endif // _RHEOLEF_HAVE_MPI
}
oparstream::size_type
oparstream::io_proc() 
{
  return iparstream::io_proc(); 
}
/// @brief This routine opens a physical input file
void
iparstream::open (
  std::string filename, 
  std::string suffix,
  const communicator& comm)
{
  warning_macro ("iparstream::open("<<filename<<","<<suffix<<")...");
  close();
  if (_use_alloc && _ptr_is != 0) {
    delete_macro (_ptr_is);
    _use_alloc = false;
    _ptr_is = 0;
  }
  if (size_type(comm.rank()) == iparstream::io_proc()) {
    _ptr_is = new_macro (irheostream(filename, suffix));
  } else {
    _ptr_is = new_macro (irheostream);
  }
  _comm = comm;
  _use_alloc = true;
  warning_macro ("iparstream::open("<<filename<<","<<suffix<<") done");
}
void
iparstream::close ()
{
  if (_use_alloc && _ptr_is != 0) {
    if (size_type(_comm.rank()) == iparstream::io_proc()) {
      irheostream* ptr_irs = (irheostream*)(_ptr_is);
      (*ptr_irs).close();
    }
  }
}
iparstream::~iparstream ()
{
  close();
  if (_use_alloc && _ptr_is != 0) {
    delete_macro (_ptr_is);
    _use_alloc = false;
    _ptr_is = 0;
  }
}
bool
iparstream::good() const
{
    bool status;
    if (size_type(comm().rank()) != iparstream::io_proc()) {
	status = true;
    } else if (_ptr_is == 0) {
        status = false;
    } else {
        status = (*_ptr_is).good();
    }
    mpi::broadcast(comm(), status, 0);
    return status;
}
// --------------------------------------------------------------
// output
// --------------------------------------------------------------
/// @brief This routine opens a physical output file
void
oparstream::open (
  std::string filename, 
  std::string suffix,
  const communicator& comm)
{
  close();
  if (_use_alloc && _ptr_os != 0) {
    delete_macro (_ptr_os);
    _use_alloc = false;
    _ptr_os = 0;
  }
  if (size_type(comm.rank()) == oparstream::io_proc()) {
    _ptr_os = new_macro (orheostream(filename, suffix));
  } else {
    _ptr_os = new_macro (orheostream);
  }
  _comm = comm;
  _use_alloc = true;
}
void
oparstream::close ()
{
  if (_use_alloc && _ptr_os != 0) {
    if (size_type(_comm.rank()) == oparstream::io_proc()) {
      orheostream* ptr_ors = (orheostream*)(_ptr_os);
      (*ptr_ors).close();
    }
  }
}
oparstream::~oparstream ()
{
  close();
  if (_use_alloc && _ptr_os != 0) {
    delete_macro (_ptr_os);
    _use_alloc = false;
    _ptr_os = 0;
  }
}
bool
oparstream::good() const
{
    bool status;
    if (size_type(comm().rank()) != iparstream::io_proc()) {
	status = true;
    } else if (_ptr_os == 0) {
        status = false;
    } else {
        status = (*_ptr_os).good();
    }
    mpi::broadcast(comm(), status, 0);
    return status;
}

} // namespace rheolef
