///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/csr.h"
#include "rheolef/asr.h"

#include "rheolef/msg_util.h"
#include "rheolef/asr_to_csr.h"
#include "rheolef/csr_to_asr.h"
#include "rheolef/csr_amux.h"
using namespace std;
namespace rheolef {
// ----------------------------------------------------------------------------
// class member functions
// ----------------------------------------------------------------------------

template<class T>
csr_seq_rep<T>::csr_seq_rep(const distributor& row_ownership, const distributor& col_ownership, size_type nnz1) 
 : vector_of_iterator<pair_type>(row_ownership.size()+1),
   _row_ownership (row_ownership),
   _col_ownership (col_ownership),
   _data (nnz1)
{
}
template<class T>
void
csr_seq_rep<T>::resize (const distributor& row_ownership, const distributor& col_ownership, size_type nnz1) 
{
   vector_of_iterator<pair_type>::resize (row_ownership.size()+1);
   _row_ownership = row_ownership;
   _col_ownership = col_ownership;
   _data.resize (nnz1);
}
template<class T>
csr_seq_rep<T>::csr_seq_rep(size_type loc_nrow1, size_type loc_ncol1, size_type loc_nnz1) 
 : vector_of_iterator<pair_type> (loc_nrow1+1),
   _row_ownership (distributor::decide, communicator(), loc_nrow1),
   _col_ownership (distributor::decide, communicator(), loc_ncol1),
   _data (loc_nnz1)
{
}
template<class T>
void
csr_seq_rep<T>::resize (size_type loc_nrow1, size_type loc_ncol1, size_type loc_nnz1)
{
   vector_of_iterator<pair_type>::resize (loc_nrow1+1);
   _row_ownership = distributor (distributor::decide, communicator(), loc_nrow1);
   _col_ownership = distributor (distributor::decide, communicator(), loc_ncol1);
   _data.resize (loc_nnz1);
}
template<class T>
csr_seq_rep<T>::csr_seq_rep(const csr_seq_rep<T>& a)
 : vector_of_iterator<pair_type>(a.nrow()+1),
   _row_ownership (a.row_ownership()),
   _col_ownership (a.col_ownership()),
   _data(a._data)
{
   // TODO: performs copy of arrays
   fatal_macro ("physical copy of csr");
}     
template<class T>
csr_seq_rep<T>::csr_seq_rep(const asr_seq_rep<T>& a)
  : vector_of_iterator<pair_type>(a.nrow()+1),
   _row_ownership (a.row_ownership()),
   _col_ownership (a.col_ownership()),
   _data(a.nnz())
{
    typedef pair<size_type,T> pair_type;
    typedef typename asr_seq_rep<T>::row_type::value_type const_pair_type;
    
    asr_to_csr (
	a.begin(),
        a.end(), 
	always_true<const_pair_type>(), 
	pair_identity<const_pair_type,pair_type>(), 
        vector_of_iterator<pair_type>::begin(), 
        _data.begin());
}
template<class T>
void
csr_seq_rep<T>::to_asr(asr_seq_rep<T>& b) const
{
    typedef pair<size_type,T> pair_type;
    typedef typename asr_seq_rep<T>::row_type::value_type const_pair_type;

    csr_to_asr (
        vector_of_iterator<pair_type>::begin(), 
        vector_of_iterator<pair_type>::end(),
        _data.begin().operator->(), 
	pair_identity<pair_type,const_pair_type>(), 
	b.begin().operator->());
}
template<class T>
iparstream& 
csr_seq_rep<T>::get (iparstream& ps)
{
    typedef pair<size_type,T> pair_type;
    typedef typename asr_seq_rep<T>::row_type::value_type const_pair_type;
    asr_seq_rep<T> a;
    a.get(ps);
    resize (a.nrow(), a.ncol(), a.nnz());
    asr_to_csr (
	a.begin(), 
        a.end(), 
	always_true<const_pair_type>(), 
	pair_identity<const_pair_type,pair_type>(), 
        vector_of_iterator<pair_type>::begin(), 
        _data.begin());
    return ps;
}
template<class T>
oparstream&
csr_seq_rep<T>::put (oparstream& ops, size_type istart) const
{
    std::ostream& os = ops.os();
    const_iterator ia = begin();
    for (size_type i = 0, n = nrow(); i < n; i++) {
        for (const_data_iterator iter_jva = ia[i], last_jva = ia[i+1];
	    iter_jva != last_jva; iter_jva++) {

	    os << i+istart << " "
               << (*iter_jva).first << " "
               << (*iter_jva).second << endl;	
  	}
    }
    return ops;
}
template<class T>
void
csr_seq_rep<T>::dump (const string& name, size_type istart) const
{
    std::ofstream os (name.c_str());
    std::cerr << "! file \"" << name << "\" created." << std::endl;
    oparstream ops(os);
    put (ops);
}
// ----------------------------------------------------------------------------
// basic linear algebra
// ----------------------------------------------------------------------------

template<class T>
void
csr_seq_rep<T>::mult(
    const vec<T,sequential>& x,
    vec<T,sequential>&       y)
    const
{
    csr_amux (
        vector_of_iterator<pair_type>::begin(), 
        vector_of_iterator<pair_type>::end(), 
        x.begin(), 
        set_op<T,T>(),
        y.begin());
}
#ifdef TO_CLEAN
// y += a*x; not used by csr & vec_expr
template<class T>
void
csr_seq_rep<T>::mult_add(
    const vec<T,sequential>& x,
    vec<T,sequential>&       y)
    const
{
    csr_amux (
        vector_of_iterator<pair_type>::begin(), 
        vector_of_iterator<pair_type>::end(), 
	x.begin(),
	set_add_op<T,T>(),
	y.begin());
}
#endif // TO_CLEAN
// ----------------------------------------------------------------------------
// expression c=a+b and c=a-b with a temporary c=*this
// ----------------------------------------------------------------------------
template<class T>
template<class BinaryOp>
void
csr_seq_rep<T>::assign_add (
    const csr_seq_rep<T>& a, 
    const csr_seq_rep<T>& b,
    BinaryOp binop)
{
    check_macro (a.nrow() == b.nrow() && a.ncol() == b.ncol(),
	"incompatible csr add(a,b): a("<<a.nrow()<<":"<<a.ncol()<<") and "
	"b("<<b.nrow()<<":"<<b.ncol()<<")");
    //
    // first pass: compute nnz_c and resize
    //
    size_type nnz_c = 0;
    const size_type infty = std::numeric_limits<size_type>::max();
    const_iterator ia = a.begin();
    const_iterator ib = b.begin();
    for (size_type i = 0, n = a.nrow(); i < n; i++) {
        for (const_data_iterator iter_jva = ia[i], last_jva = ia[i+1],
                                 iter_jvb = ib[i], last_jvb = ib[i+1];
	    iter_jva != last_jva || iter_jvb != last_jvb; ) {

            size_type ja = iter_jva == last_jva ? infty : (*iter_jva).first;
            size_type jb = iter_jvb == last_jvb ? infty : (*iter_jvb).first;
	    if (ja == jb) {
		iter_jva++;
		iter_jvb++;
	    } else if (ja < jb) {
		iter_jva++;
            } else {
		iter_jvb++;
            }
	    nnz_c++;
  	}
    }
    resize (a.row_ownership(), b.col_ownership(), nnz_c);
    data_iterator iter_jvc = _data.begin().operator->();
    iterator ic = begin();
    *ic++ = iter_jvc;
    //
    // second pass: add and store in c
    //
    for (size_type i = 0, n = a.nrow(); i < n; i++) {
        for (const_data_iterator iter_jva = ia[i], last_jva = ia[i+1],
                                 iter_jvb = ib[i], last_jvb = ib[i+1];
	    iter_jva != last_jva || iter_jvb != last_jvb; ) {

            size_type ja = iter_jva == last_jva ? infty : (*iter_jva).first;
            size_type jb = iter_jvb == last_jvb ? infty : (*iter_jvb).first;
	    if (ja == jb) {
		*iter_jvc++ = std::pair<size_type,T> (ja, binop((*iter_jva).second, (*iter_jvb).second));
		iter_jva++;
		iter_jvb++;
	    } else if (ja < jb) {
		*iter_jvc++ = *iter_jva++;
            } else {
		*iter_jvc++ = *iter_jvb++;
            }
  	}
        *ic++ = iter_jvc;
    }
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template class csr_seq_rep<Float>;
template void csr_seq_rep<Float>::assign_add (
	const csr_seq_rep<Float>&, const csr_seq_rep<Float>&, std::plus<Float>);
template void csr_seq_rep<Float>::assign_add (
	const csr_seq_rep<Float>&, const csr_seq_rep<Float>&, std::minus<Float>);
} // namespace rheolef
