# ifndef _RHEO_UZAWA_ABTBC_H
# define _RHEO_UZAWA_ABTBC_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

/*Class:uzawa_abtbc
NAME: @code{uzawa_abtbc}, @code{uzs_abtbc} -- stabilized mixed linear solver
@findex uzawa\_abtbc
@cindex mixed linear problem
@cindex Stokes problem
@cindex Uzawa algorithm
@cindex finite element method
@cindex stabilized mixed finite element method
@cindex incompresible elasticity
DESCRIPTION: 
  @noindent
  Uzawa algorithm applied to the mixed problem
  @example
       [ a  b^T ] [ u ]    [ f ]
       [        ] [   ]  = [   ]
       [ b  -c  ] [ p ]    [ g ]
  @end example
  with the bock diagonal preconditionner : (inv(a),I).
  Such mixed linear problems appears for instance with the discretization
  of Stokes problems with stabilized P1-P1 element, or with nearly
  incompressible elasticity.
  The algorithm requires the resolution of linear
  systems such as: @code{a*x = b}.
  These systems may be solved either by direct or iterative 
  algorithms. Thus, a general matrix solver class is submitted
  to the algorithm.

  @noindent
  @code{uzawa_abtb} is a specialization of the @code{uzs_abtbc} that
  uses a direct solver.

AUTHOR: 
   | Nicolas.Roquet@imag.fr
    LMC-IMAG, 38041 Grenoble cedex 9, France
DATE:   15 nov 1997
METHODS: @uzs_abtbc
End:
*/

#include "rheolef/matrix_solver.h"
namespace rheolef { 

//<uzawa_abtbc:  
template<
    class MatrixSolver, 
    class MatrixPreconditioner, 
    class Matrix, 
    class DiagMatrix,
    class Vector, 
    class Real>
int
uzs_abtbc(
    const MatrixSolver&         m_solver,
    const Matrix&               a,
    const MatrixPreconditioner& ap,
    const Matrix&               b,
    const Matrix&               c,
    const DiagMatrix&           I,
    Vector&                     u,
    Vector&                     p,
    const Vector&               f,
    const Vector&               g,
    const Real&                 rho,
    int&                        max_iter, 
    Real&                       tol, 
    ostream*                    p_cres = 0)
{
  Float residu = 1. ;
  for (int k=1 ; k <= max_iter ; k++) {
      Vector oldp = p;
      Vector oldu = u;
      int status_ms = m_solver(a, ap, u, f - b.trans_mult(p));  
      Vector r = I.solve(b*u - g + c*p);
      p = p + rho*r;
      residu = norm(r);
      if (p_cres) *p_cres << "[uzawa_abtbc] " << k << " " << residu << "\n";
      if (residu <= tol) {
	  tol = residu;
	  max_iter = k;
	  return 0;
      }
  }
  tol = residu;
  return 1;
}

template<
    class MatrixPreconditionner,
    class Matrix,
    class Vector,
    class Real>
int
uzawa_abtbc(
    const Matrix&                a,
    const MatrixPreconditionner& ap,
    const Matrix&                b,
    const Matrix&                c,
    Vector&                      u,
    Vector&                      p,
    const Vector&                f,
    const Vector&                g,
    const Real&                  rho,
    int&                         max_iter,
    Real&                        tol,
    ostream*                     p_cres = &std::cerr)
{
    return uzs_abtbc (ldlt_solver<MatrixPreconditionner, Matrix, Vector, Vector>(),
        a, ap, b, c, EYE, u, p, f, g, rho, max_iter, tol, p_cres);
}
//>uzawa_abtbc:  
}// namespace rheolef
# endif // _RHEO_UZAWA_ABTBC_H
