///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
// 
// Subdivision tree on a mesh with grid stump 
// and octree-like embranching. 
//
// author: jocelyn.etienne@imag.fr
//
// date: 4 Apr 2002
// 

#include "rheolef/geomap.h"
#include "rheolef/project.h"
using namespace std;
namespace rheolef { 

static void init_localizer (const geo& omega) {
    omega.init_localizer (omega.boundary());
}
// -------------------------------------------------------------
// constructors: without advection field
// -------------------------------------------------------------
geomap::geomap (const geo& Th_2, const space& Vh_1, 
    bool allow_approximate_edges) :
    _Th_1 (Vh_1.get_geo()), _Th_2 (Th_2), _advection(), advected(false), 
    _quadrature("default"), _order(0),  
    _allow_approximate_edges(allow_approximate_edges), use_space(true), _Vh_1(Vh_1),
    _is_inside(), _barycentric_weight(0), _option()
{
    init_localizer (_Th_2);
    for (size_type i_comp = 1; i_comp < _Vh_1.n_component(); i_comp++) {
	check_macro (_Vh_1.get_approx(i_comp) == _Vh_1.get_approx(0),
	     "heterogeneous multi-spaces not yet supported");
    }
    size_type i_comp = 0;
    size_t sz = _Vh_1.size_component(i_comp);
    vector<bool> todo (sz, true);
    resize(sz);
    _is_inside.resize(sz, true);
    const geo& g_1 = _Vh_1.get_geo();
    size_type K_index = 0;
    for (geo::const_iterator K = g_1.begin(); K != g_1.end(); K++, K_index++) {
	tiny_vector<size_type> idx;
	_Vh_1.set_dof (*K, idx, i_comp);
	for (size_type iloc = 0; iloc < idx.size(); iloc++) {
	    if (!todo[idx[iloc]]) continue;
	    size_type i_dof = idx[iloc];
	    todo[i_dof] = false;
	    point x0 = _Vh_1.x_dof (*K, iloc);
	    size_type e;
	    point y = x0;
	    // Is the dof in Th_2 triangulation ?
	    if (_allow_approximate_edges) {
		Th_2.localize_nearest(x0, y, e);
	    } else {
		check_macro(_Th_2.localize(x0, e), "Triangulations do not fit !"
		    <<"-- If you remeshed the same domain, try allow_approximate=true");
	    }
	    at(i_dof) = _Th_2.hatter(y, e);
	}
    }
}
geomap::geomap (const geo& Th_2, const space& Vh_1, 
	Float barycentric_weight) :
    _Th_1 (Vh_1.get_geo()), _Th_2 (Th_2), _advection(), advected(false), 
    _quadrature("default"), _order(0),  
    _allow_approximate_edges(false), use_space(true), _Vh_1(Vh_1),
    _is_inside(), _barycentric_weight(barycentric_weight), _option()
{
    init_localizer (_Th_2);
    for (size_type i_comp = 1; i_comp < _Vh_1.n_component(); i_comp++) {
	check_macro (_Vh_1.get_approx(i_comp) == _Vh_1.get_approx(0),
	     "heterogeneous multi-spaces not yet supported");
    }
    size_type i_comp = 0;
    size_t sz = _Vh_1.size_component(i_comp);
    vector<bool> todo (sz, true);
    resize(sz);
    _is_inside.resize(sz, true);
    const geo& g_1 = _Vh_1.get_geo();
    size_type K_index = 0;
    for (geo::const_iterator K = g_1.begin(); K != g_1.end(); K++, K_index++) {
	tiny_vector<size_type> idx;
	_Vh_1.set_dof (*K, idx, i_comp);
	point xb(0);
	for (size_type iloc = 0; iloc < idx.size(); iloc++)
		xb = xb + _Vh_1.x_dof (*K, iloc);
	xb = 1./idx.size()*xb;
	for (size_type iloc = 0; iloc < idx.size(); iloc++) {
	    if (!todo[idx[iloc]]) continue;
	    size_type i_dof = idx[iloc];
	    todo[i_dof] = false;
	    point x0 = (1-_barycentric_weight)*_Vh_1.x_dof (*K, iloc) 
	    	+ _barycentric_weight*xb;
	    size_type e;
	    point y = x0;
	    // Is the dof in Th_2 triangulation ?
	    if (_allow_approximate_edges) {
		Th_2.localize_nearest(x0, y, e);
	    } else {
		check_macro(_Th_2.localize(x0, e), "Triangulations do not fit !"
		    <<"-- If you remeshed the same domain, try allow_approximate=true");
	    }
	    at(i_dof) = _Th_2.hatter(y, e);
	}
    }
}
// ----------------------------------------------------------------
// init data structure : without advection field
// ----------------------------------------------------------------
//! TO BE MODIFIED: memory consumming !
void geomap::init() {
    init_localizer (_Th_2);
    const geo& Th_1=_Th_1;

    get_quadrature(_quadrature, _order, geo_element::t, quad_point, quad_weight);
    size_t quad_per_element = quad_point.size();
    size_t sz = Th_1.n_element()*quad_per_element;
    resize(sz);
    _is_inside.resize(sz, true);

    vector<bool> todo (Th_1.n_vertex()+Th_1.n_edge(), true);
    size_type K_index_Th_1 = 0;
    point x_q;

    for (geo::const_iterator iK_Th_1= Th_1.begin(); iK_Th_1 != Th_1.end(); iK_Th_1++, K_index_Th_1++) 
     {
	 cerr << "Elem " << K_index_Th_1 << endl;
	 const geo_element& K_Th_1 =*iK_Th_1;
	 // TO BE MODIFIED: This may slow down mixed meshes (triangles+quadrangles,...)
//	if (!((*K_Th_1).type() ==geo_element::t))
//	 {
//	     cerr << "Not yet implemented for non triangular meshes";
//	 }
     
     // TO BE MODIFIED: Only for triangles.
	for (size_t q=0; q <3; q++)
	    if (todo[K_Th_1[q]])
	    {
    //	cerr << K_index_Th_1*quad_per_element+q << " ";
	    at(K_Th_1[q]) = advect(quad_point[q], K_index_Th_1); 
	    }
	for (size_t e=0; e <3; e++)
	    {
	    cerr << "Edge " << K_Th_1.edge(e) << "[" << K_Th_1[e] <<"-" << K_Th_1[(e+1)%3] << "]" 
		<< endl 
		<< " from " << Th_1.begin_node()[K_Th_1[e]] 
		<< " to " << Th_1.begin_node()[K_Th_1[(e+1)%3]] << endl;
	    if (todo[Th_1.n_vertex() +K_Th_1.edge(e)])
	    {
	    if (K_Th_1[e]<K_Th_1[(e+1)%3])
		for (size_t q=0; q<_order-1; q++)
		 {
		    at(Th_1.n_vertex() +(_order-1)*K_Th_1.edge(e)+q) 
			=advect(quad_point[3+(_order-1)*e+q], K_index_Th_1);
		    cerr << " ->" << Th_1.n_vertex() +(_order-1)*K_Th_1.edge(e)+q << ": (" 
			    << at(Th_1.n_vertex() +(_order-1)*K_Th_1.edge(e)+q).hat() 
			    << ", " << at(Th_1.n_vertex() +(_order-1)*K_Th_1.edge(e)+q).element()
			    << ")" 
			    << Th_1.dehatter(quad_point[3+(_order-1)*e+q], K_index_Th_1)
			    << " ~ " << quad_point[3+(_order-1)*e+q] << endl;
		 }
	    else
		for (size_t q=0; q<_order-1; q++)
		 {
		     at(Th_1.n_vertex() +(_order-1)*K_Th_1.edge(e)+_order-2-q) 
			=advect(quad_point[3+(_order-1)*e+q], K_index_Th_1);
		    cerr << " <-" << Th_1.n_vertex() +(_order-1)*K_Th_1.edge(e)+_order-2-q << ": (" 
			    << at(Th_1.n_vertex() +(_order-1)*K_Th_1.edge(e)+_order-2-q).hat() 
			    << ", " << at(Th_1.n_vertex() +(_order-1)*K_Th_1.edge(e)+_order-2-q).element()
			    << ")" 
			    << Th_1.dehatter(quad_point[3+(_order-1)*e+q], K_index_Th_1)
			    << " ~ " << quad_point[3+(_order-1)*e+q] << endl;
		 }
	    todo[Th_1.n_vertex() +K_Th_1.edge(e)]=false;
	    }
	    else
	    { //TO BE REMOVED: Just test...
	    if (K_Th_1[e]<K_Th_1[(e+1)%3])
		for (size_t q=0; q<_order-1; q++)
		 {
		    if (at(Th_1.n_vertex() +(_order-1)*K_Th_1.edge(e)+q).element() 
			!=advect(quad_point[3+(_order-1)*e+q], K_index_Th_1).element() )
			error_macro("NO FIT -> ");
		    cerr    << "*-> " << Th_1.n_vertex() +(_order-1)*K_Th_1.edge(e)+q << ": "
			    << "(" << at(Th_1.n_vertex() +(_order-1)*K_Th_1.edge(e)+q).hat() 
			    << "," << at(Th_1.n_vertex() +(_order-1)*K_Th_1.edge(e)+q).element()
			    << " (" << advect(quad_point[3+(_order-1)*e+q], K_index_Th_1).hat()
			    << "," << advect(quad_point[3+(_order-1)*e+q], K_index_Th_1).element()
			    << ") " << endl
			    << Th_1.dehatter(quad_point[3+(_order-1)*e+q], K_index_Th_1) 
			    << " ~ " << quad_point[3+(_order-1)*e+q] << endl;
		 }
	    else
		for (size_t q=0; q<_order-1; q++)
		 {
		     if (at(Th_1.n_vertex() +(_order-1)*K_Th_1.edge(e)+_order-2-q).element()
			!=advect(quad_point[3+(_order-1)*e+q], K_index_Th_1).element() )
			error_macro("NO FIT <-  ");
		    cerr << "*<-"    << Th_1.n_vertex() +(_order-1)*K_Th_1.edge(e)+_order-2-q << ": "
			    << "(" << at(Th_1.n_vertex() +(_order-1)*K_Th_1.edge(e)+_order-2-q).hat() 
			    << "," << at(Th_1.n_vertex() +(_order-1)*K_Th_1.edge(e)+_order-2-q).element()
			    << " (" << advect(quad_point[3+(_order-1)*e+q], K_index_Th_1).hat()
			    << "," << advect(quad_point[3+(_order-1)*e+q], K_index_Th_1).element() 
			    << ") "
			    << Th_1.dehatter(quad_point[3+(_order-1)*e+q], K_index_Th_1) 
			    << " ~ " << quad_point[3+(_order-1)*e+q] << endl;
		 }
	    }
	    }
	for (size_t q=3+3*(_order-1); q<quad_per_element; q++)
	    at(Th_1.n_vertex() +(_order-1)*Th_1.n_edge() 
		+ (quad_per_element -3+3*(_order-1))*K_index_Th_1 +q)
		=advect(quad_point[q], K_index_Th_1);

    }
}
// ---------------------------------------------------------------------------
// initializer with advection field (with some auxilliary functions)
// ---------------------------------------------------------------------------
// x1 = x0 + va
meshpoint geomap::robust_advect_1 (const point& x0, const point& va, bool& is_inside) const {
    static const Float tol = ::sqrt(numeric_limits<Float>::epsilon());
    point x;
    Float t;
    size_type e;
    // Is the characteristic origin in the mesh ?
    bool b_trace ;
    if (norm2(va) > tol) {
        b_trace = _Th_2.trace(x0, va, x, t, e);
    } else {
	_Th_2.localize_nearest(x0, x, e);
	t = 1;
	b_trace = true;
    }
    if (! b_trace) {    
	// x is outside of the mesh:
        // then use x0 as characteristic origin
	point y;
	_Th_2.localize_nearest(x0, y, e);
	x = y;
    }
    is_inside = b_trace && (1-t < tol);
    meshpoint x1 = _Th_2.hatter(x,e);
    return x1;
}
// xN = x0 + va
// split the advection in N sub-steps and check when it goes out
meshpoint geomap::robust_advect_N (const point& x0, const point& v0, const field& vh, size_t n, bool& is_inside) const
{
  size_t i = 0;
  point xi = x0;
  point vi = v0/Float(n);
  do {
    meshpoint hat_xi1 = robust_advect_1 (xi, vi, is_inside);
    if (i+1 >= n || !is_inside) return hat_xi1;
    xi = _Th_2.dehatter (hat_xi1);
    // TODO: when vh is localized on the same mesh as x0..xN
    // gives as a guess the previous element K containing xi
    // for evaluation vh(xi): do not need any new localization
    vi = vh.vector_evaluate(xi)/Float(n);
    i++;
  } while (true);
}
static Float h_moy (geo::const_iterator_node p, const geo_element& K) {
  if (K.type() == geo_element::t) {
    Float area = fabs( vect2d( p[K[1]] -p[K[0]], p[K[2]] -p[K[0]] ))/2;
    return ::sqrt(area);
  }
  if (K.type() == geo_element::q) {
    Float area1 = fabs( vect2d( p[K[1]] -p[K[0]], p[K[2]] -p[K[0]] ))/2;
    Float area2 = fabs( vect2d( p[K[2]] -p[K[0]], p[K[3]] -p[K[0]] ))/2;
    return ::sqrt(area1 + area2);
  }
  error_macro("Only triangles or quadrangle supported yet.");
}
// in order to localize  y = x + v(x)
// use adapt n substeps such that: n*h0 = v0
// with the local mesh size h0=h_moy(K)
static inline size_t adapt_n_track_step (Float h0, const point& v0, size_t n_track_step_max) {
    if (n_track_step_max <= 1) return 1;
    size_t n = size_t(norm(v0)/h0 + 0.5);
    n = std::max(n, size_t(1));
    n = std::min(n, n_track_step_max);
    return n;
}
void geomap::init (const field& advection) {
  const space& Va  = advection.get_space();
  const geo&   g_1 = _Vh_1.get_geo();
  init_localizer (Va.get_geo());
  if (Va.n_component() !=  Va.dimension())
	error_macro("Advection field should be a " << Va.dimension() <<"D vectors field."); 
  if (_Vh_1.n_component() != 1) {
    for (size_type i_comp = 1; i_comp < _Vh_1.n_component(); i_comp++) {
      check_macro (_Vh_1.get_approx(i_comp) == _Vh_1.get_approx(0),
	 "heterogeneous multi-spaces not yet supported");
    }
  }
  if (Va.get_approx() != _Vh_1.get_approx())
    error_macro("Advection field should have approximation " << _Vh_1.get_approx() 
      << ", found " << Va.get_approx() );
  
  // loop on _Vh_1 dof's
  size_t d = _Vh_1.dimension();
  size_type offset1 = (d < 2) ? 0 : Va.start(1);
  size_type offset2 = (d < 3) ? 0 : Va.start(2);
  size_type i_comp = 0;
  vector<bool> todo (_Vh_1.size_component(i_comp), true);
  size_t sz = _Vh_1.size_component(i_comp);
  resize(sz);
  _is_inside.resize(sz, true);
  for (geo::const_iterator pK = g_1.begin(); pK != g_1.end(); pK++) {
    tiny_vector<size_type> idx;
    _Vh_1.set_dof (*pK, idx, i_comp);
    Float hK = h_moy(g_1.begin_node(), *pK);
    for (size_type iloc = 0; iloc < idx.size(); iloc++) {
      if (!todo[idx[iloc]]) continue;
      size_type i_dof = idx[iloc];
      todo[i_dof] = false;
      point x0 = _Vh_1.x_dof (*pK, iloc); // TODO: improve x_dof call
      point va ( 
          advection.at(i_dof),
          (d < 2) ? Float(0) : advection.at(i_dof+offset1),
          (d < 3) ? Float(0) : advection.at(i_dof+offset2)
        );
      size_t n = adapt_n_track_step (hK, va, _option.n_track_step);
      bool flag;
      at(i_dof) = robust_advect_N (x0, va, advection, n, flag);
      _is_inside [i_dof] = flag;
    }
  }
}
//--------------------------------------------------
// compose
//--------------------------------------------------
//! Gives the characteristic foot for a quadrature point.
/*! q is the hat coordinates of the quadrature point in element iK_Th_1. */
meshpoint
geomap::advect(const point& q, size_t iK_Th_1)
 {
    point x_q =_Th_1.dehatter(q, iK_Th_1);
     point y_q =x_q;
    Float t_f;
    size_t iK_Th_2;
     if (!advected) // if there's no advection, just locate q within the mesh Th_2
     {
	if (_allow_approximate_edges)
	    _Th_2.localize_nearest(x_q, y_q, iK_Th_2);
	else
	    check_macro(_Th_2.localize(x_q, iK_Th_2), "Triangulations do not fit !"
		<<"-- If you remeshed the same domain, try allow_approximate=true");
     }
    else // if there's advection, trace the characteristic back to its foot
     {
	point v_xq =point(0,0,0);
	for (size_t i=0; i<_advection.dimension(); i++) 
	    v_xq[i]=_advection.evaluate(q, iK_Th_1, i);
	if (!_Th_2.trace(x_q, v_xq, y_q, t_f, iK_Th_2))     
	    // y_q lies outside of the mesh: then use x_q as characteristic origin
	    _Th_2.localize_nearest(x_q, y_q, iK_Th_2);
//	cerr << x_q << "+" << v_xq << "=" << y_q << endl;
     }
    return _Th_2.hatter(y_q, iK_Th_2);
 }

tiny_matrix<Float>
dof_from_int (const geo& g, const space& Vh, Float weight) {
  geo_element e=*(g.begin());
  geomap::size_type n = Vh.get_basis().size(e);
  std::vector<Float> p (n);
  point hat_x, xb(0);
  for (geomap::size_type iloc = 0; iloc < n; iloc++)
       xb = xb + Vh.x_dof (e, iloc);
  xb = 1./n*xb;
  meshpoint x =g.hatter(weight*xb + (1-weight)*Vh.x_dof (e, 0), 0);
  Vh.get_basis().eval (e, x.hat(), p);
  Float diag=p[0];
  Float ediag=p[1];
  Float det=(diag-ediag)*(diag+2*ediag);
  tiny_matrix<Float> m(n,n);
  m.fill(-ediag/det);
  for (geomap::size_type i=0; i<n; i++) m(i,i)+=1/(diag-ediag);
  return m;
}
field compose_with_weight (const field& f, const geomap& g) {
    // multi-component: assume same approx for each component !
    typedef field::size_type size_type;
    const space& _Vh_1 =g.get_space();
    const geo& omega=_Vh_1.get_geo();
    check_macro (g.get_target_triangulation() == f.get_space().get_geo(),
        "compose : Meshes do not match : " << g.get_target_triangulation().name()
        << " and " << f.get_space().get_geo().name() << ".");
    field fog(_Vh_1);
    size_type offset = 0;
    tiny_vector<size_type> i_dof;
    tiny_vector<Float> value;
    tiny_matrix<Float> m=dof_from_int(omega,_Vh_1,g.barycentric_weight());
    for (size_type i_comp = 0; i_comp < f.get_space().n_component(); i_comp++) {
        geomap::const_iterator geomap_el = g.begin();
        //size_type sz_comp = f.get_space().size_component(i_comp);
        size_type sz_comp = _Vh_1.size_component(i_comp);
        check_macro (sz_comp == g.size(), "incompatible " << i_comp << "-th component size "
                << sz_comp << " and geomap size " << g.size());
	geo::const_iterator iK = omega.begin();
	geo::const_iterator endK = omega.end();
	for ( ; iK != endK ; iK++)
	 {
	    _Vh_1.set_dof(*iK, i_dof);
	    value.resize((*iK).size());
            for (size_type i = 0; i < (*iK).size() ; i++) 
	     	value[i] = f.evaluate (geomap_el[i_dof[i]], i_comp);
	    value=m*value;
            for (size_type i = 0; i < (*iK).size() ; i++) 
            	fog.at(offset+i_dof[i]) = value[i];
         }
        offset += sz_comp;
    }
    return fog;
}
field compose (const field& f, const geomap& g) {
    // multi-component: assume same approx for each component !
    if (!g.no_barycentric_weight()) compose_with_weight(f,g);
    typedef field::size_type size_type;
    const space& _Vh_1 = g.get_space();
    check_macro (g.get_target_triangulation() == f.get_space().get_geo(),
	"compose : Meshes do not match : " << g.get_target_triangulation().name() 
	<< " and " << f.get_space().get_geo().name() << ".");
    field fog(_Vh_1);
    size_type offset = 0;
    for (size_type i_comp = 0; i_comp < f.get_space().n_component(); i_comp++) {
	geomap::const_iterator ig = g.begin();
	//size_type sz_comp = f.get_space().size_component(i_comp);
	size_type sz_comp = _Vh_1.size_component(i_comp);
	check_macro (sz_comp == g.size(), "incompatible " << i_comp << "-th component size "
		<< sz_comp << " and geomap size " << g.size());
	
	for (size_type i_dof = 0; i_dof < sz_comp; i_dof++, ig++) {
	    Float value = f.evaluate (*ig, i_comp);
	    fog.at(offset+i_dof) = value;
	}
	offset += sz_comp;
    }
    return fog;
}
// send a call-back function when characteristic foot is outside :
//   it is e.g. an upstream boundary condition
// implemented only for scalar-valued function yet
field compose (const field& f, const geomap& g, Float (*f_outside)(const point&))
{    
    // multi-component: assume same approx for each component !
    check_macro(g.no_barycentric_weight(), "Unsupported barycentric weight in geomap");
    typedef field::size_type size_type;
    const space& _Vh_1 =g.get_space();
    check_macro (g.get_target_triangulation() == f.get_space().get_geo(),
	"compose : Meshes do not match : " << g.get_target_triangulation().name() 
	<< " and " << f.get_space().get_geo().name() << ".");
    field fog(_Vh_1);
    size_type offset = 0;
    for (size_type i_comp = 0; i_comp < _Vh_1.n_component(); i_comp++) {
	geomap::const_iterator ig = g.begin();
	size_type sz_comp = _Vh_1.size_component(i_comp);
	check_macro (sz_comp == g.size(), "incompatible " << i_comp << "-th component size "
	<< sz_comp << " and geomap size " << g.size());
	for (size_type i_dof = 0; i_dof < sz_comp; i_dof++, ig++) {
	    Float value;
	    if (g.is_inside (i_dof)) {
	      value = f.evaluate (*ig, i_comp);
	    } else {
	    /* HAVE TO USE TARGET TRIANGULATION, NOT ORIGIN ! */
	      point x = g.get_target_triangulation().dehatter (*ig);
	      value = f_outside (x);
	    }
	    fog.at(offset+i_dof) = value;
	}
	offset += sz_comp;
    }    
    return fog;
}
//--------------------------------------------------
//   Symbolic composed fields
//--------------------------------------------------
Float fieldog::operator() (const point& x) const {
    point v = g._advection.vector_evaluate (x);
    point xv; // x+v or the nearest in the convected direction
    Float t;
    size_type K_idx;
    const geo& Th = g.get_origin_triangulation();
    bool inside = Th.trace (x, v, xv, t, K_idx);
    if (! inside) { // x is outside of the mesh
	Th.localize_nearest (x, xv, K_idx);
    }
    return f (xv);
}
Float fieldog::operator() (const meshpoint& hat_x_in_K) const {
     error_macro("Not implemented yet");
}
vector<Float> fieldog::quadrature_values (size_t iK) const {
    vector<Float> val (g.quad_point.size());
    for (size_t q=0; q<g.quad_point.size(); q++)
	val[q] =f.evaluate(g.at(iK*g.quad_point.size()+q));
    return val;
}
vector<Float> fieldog::quadrature_d_dxi_values (size_t i, size_t iK) const {
    vector<Float> val (g.quad_point.size());
    for (size_t q=0; q<g.quad_point.size(); q++)
	val[q] =f.evaluate_d_dxi(i, g.at(iK*g.quad_point.size()+q));
    return val;
}
}// namespace rheolef
