#ifndef _RHEO_FORM_ELEMENT_REP_H
#define _RHEO_FORM_ELEMENT_REP_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

// used as a virtual base class for concrete form_element classes
// such as "mass"

#include "rheolef/point.h"

#include "rheolef/tensor.h"
#include "rheolef/basis.h"
#include "rheolef/numbering.h"
#include "rheolef/quadrature.h"
#include "rheolef/basis_on_lattice.h"
#include "rheolef/fem_helper.h"
#include "rheolef/field.h"
#include "rheolef/piola_v1.h"

#include <boost/numeric/ublas/matrix.hpp>
namespace ublas = boost::numeric::ublas;
namespace rheolef { 

class form_element_rep {
public:

// typedef:

    typedef size_t size_type;

// allocators:

    form_element_rep();
    virtual ~form_element_rep();

// virtual accessors (overloaded by concrete form_elements such as mass):

    virtual void operator() (const geo_element& K, ublas::matrix<Float>& m) const = 0;
    virtual void check_after_initialize () const;
    virtual size_type n_derivative() const;

// modifiers (const, since data are mutable: avoid copies):

    void initialize (const space& X, const space& Y) const;

    // for scalar-weighted forms:
    void set_weight (const field& wh) const;

// accessors (for building elementary forms):

    const space&     get_first_space() const;
    const space&     get_second_space() const;
    const basis&     get_first_basis() const;
    const basis&     get_second_basis() const;
    const numbering& get_first_numbering() const;
    const numbering& get_second_numbering() const;
    const basis&     get_p1_transformation () const;

    const geo&       get_global_geo() const;
    size_type        coordinate_dimension() const;
    fem_helper::coordinate_type coordinate_system_type() const;
    geo::const_iterator_node    begin_vertex() const;

    bool is_weighted() const;
    const field& get_weight () const;
    const basis& get_weight_basis () const;
    Float weight (const geo_element& K, size_type q, size_t i_comp = 0) const;
    void  weight (const geo_element& K, size_type q, tensor& W) const;

// toolbox for building elementary forms:

    point piola_transformation (const geo_element& K, size_type q) const;
    void jacobian_piola_transformation (const geo_element& K, size_type q, tensor& DF) const;
    Float det_jacobian_piola_transformation (const geo_element& K, size_type q) const;
    Float det_jacobian_piola_transformation (const tensor& DF, size_type d) const;
    tensor pseudo_inverse_jacobian_piola_transformation (const tensor& DF, size_type map_d) const;

    Float weight_coordinate_system (const geo_element& K, size_type q) const;
    static void cumul_otimes (
    	ublas::matrix<Float>& m, 
    	const Float& w,
    	basis_on_quadrature::const_iterator phi,
    	basis_on_quadrature::const_iterator last_phi,
    	basis_on_quadrature::const_iterator first_psi,
    	basis_on_quadrature::const_iterator last_psi);
    static void cumul_otimes (
	ublas::matrix<Float>& m, 
	const Float& w,
	basis_on_quadrature::const_iterator phi,
	basis_on_quadrature::const_iterator last_phi,
	basis_on_quadrature::const_iterator_grad first_grad_psi,
	basis_on_quadrature::const_iterator_grad last_grad_psi);
    static void cumul_otimes (
	ublas::matrix<Float>& m, 
	const Float& w,
	basis_on_quadrature::const_iterator phi,
	basis_on_quadrature::const_iterator last_phi,
	basis_on_quadrature::const_iterator_hessian first_hessian_psi,
	basis_on_quadrature::const_iterator_hessian last_hessian_psi);
    static void cumul_otimes (
	ublas::matrix<Float>& m, 
	const Float& w,
	basis_on_quadrature::const_iterator_grad grad_phi,
	basis_on_quadrature::const_iterator_grad last_grad_phi,
	basis_on_quadrature::const_iterator_grad first_grad_psi,
	basis_on_quadrature::const_iterator_grad last_grad_psi);
    static void cumul_otimes (
        ublas::matrix<Float>& m,
	const tensor& Dw,
	basis_on_quadrature::const_iterator_grad grad_phi,
	basis_on_quadrature::const_iterator_grad last_grad_phi,
	basis_on_quadrature::const_iterator_grad first_grad_psi,
	basis_on_quadrature::const_iterator_grad last_grad_psi,
	size_type map_d);

    void build_scalar_mass (const geo_element& K, ublas::matrix<Float>& m, size_type weight_i_comp=0) const;
    void build_general_mass(const geo_element& K, ublas::matrix<Float>& m) const;
    void build_d_ds        (const geo_element& K, ublas::matrix<Float>& m) const;
    void build_d2_ds2 	   (const geo_element& K, ublas::matrix<Float>& m) const;
    void build_d_ds_d_ds   (const geo_element& K, ublas::matrix<Float>& m) const;
    void build_grad_grad   (const geo_element& K, ublas::matrix<Float>& m) const;
    void build_gradt_grad  (const geo_element& K, ublas::matrix<Float>& m) const;
    void build_div_div     (const geo_element& K, ublas::matrix<Float>& m) const;
    void d_dx              (const geo_element& K, ublas::matrix<Float>& m, size_type idx) const;

    void set_use_coordinate_system_weight (bool use) const;
    bool use_coordinate_system_weight () const;
    void set_use_coordinate_system_dual_weight (bool use) const;
    bool use_coordinate_system_dual_weight () const;

// data:
protected:
    mutable space                  _X;	// origin dof numbering
    mutable space                  _Y;  // destination dof numbering

    mutable quadrature             _quad;
    mutable basis_on_quadrature	   _bx_table;
    mutable basis_on_quadrature	   _by_table;
    mutable basis_on_quadrature    _tr_p1_table;
    mutable bool                   _initialized;
    
    mutable field                  _wh;
    mutable bool                   _is_weighted;
    mutable basis_on_quadrature    _bw_table;

    mutable bool                   _use_coordinate_system_weight;
    mutable bool                   _use_coordinate_system_dual_weight;

    // automatically generated implementation in ../form_element/form_element.cc :
    static form_element_rep* make_ptr (std::string name);
    friend class form_element;
};
// -----------------------------------------------------------------------
// inlined
// -----------------------------------------------------------------------

inline
const space& 
form_element_rep::get_first_space() const
{
  return _X;
}
inline
const space& 
form_element_rep::get_second_space() const
{
  return _Y;
}
inline
const basis& 
form_element_rep::get_first_basis() const
{
  return get_first_space().get_basis();
}
inline
const basis& 
form_element_rep::get_second_basis() const
{
  return get_second_space().get_basis();
}
inline
const numbering& 
form_element_rep::get_first_numbering() const
{
  return get_first_space().get_numbering();
}
inline
const numbering& 
form_element_rep::get_second_numbering() const
{
  return get_second_space().get_numbering();
}
inline
const basis&
form_element_rep::get_p1_transformation () const
{
  return get_first_space().get_p1_transformation();
}
inline
const geo&
form_element_rep::get_global_geo() const
{
    return get_first_space().get_global_geo();
}
inline
fem_helper::coordinate_type
form_element_rep::coordinate_system_type() const
{
    return get_global_geo().coordinate_system_type();
}
inline
form_element_rep::size_type
form_element_rep::coordinate_dimension() const
{
  return get_global_geo().dimension();
}
inline
geo::const_iterator_node
form_element_rep::begin_vertex() const
{
  return get_global_geo().begin_node();
}
inline
const field&
form_element_rep::get_weight () const
{
  return _wh;
}
inline
bool
form_element_rep::is_weighted() const
{
  return _is_weighted;
}
inline
const basis&
form_element_rep::get_weight_basis () const
{
  return get_weight().get_space().get_basis();
}
inline
void
form_element_rep::set_use_coordinate_system_weight (bool use) const
{
    _use_coordinate_system_weight = use;
}
inline
bool
form_element_rep::use_coordinate_system_weight () const
{
    return _use_coordinate_system_weight;
}
inline
void
form_element_rep::set_use_coordinate_system_dual_weight (bool use) const
{
    _use_coordinate_system_dual_weight = use;
}
inline
bool
form_element_rep::use_coordinate_system_dual_weight () const
{
    return _use_coordinate_system_dual_weight;
}
}// namespace rheolef
#endif // _RHEO_FORM_ELEMENT_REP_H
