///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "rheolef/cad.h"
#include "rheolef/iorheo.h"
#include "rheolef/rheostream.h"
#include "rheolef/cad-bezier.h"
using namespace std;
namespace rheolef { 

// -----------------------------------------------------------------------
// accessors
// -----------------------------------------------------------------------
// TODO: should be independent of the cad size !
//  => compute-it at build-time
cad_rep::size_type 
cad_rep::n_face() const
{
    size_type n = 0;
    for (vector<cad_domain>::const_iterator p = _s.begin(); p != _s.end(); p++) {
	const cad_domain& d = (*p);
	n += d.size();
    }
    return n;
}
void
cad_rep::eval (const cad_element& S, const point& ref_coord, point& real_coord) const
{
    switch (S.type()) {
      case cad_element::p:
          for (size_type k = 0; k < dimension(); k++)
	    real_coord[k] = _x[S[0]][k];
	  break;
      case cad_element::e :
          for (size_type k = 0; k < dimension(); k++)
            real_coord[k] = eval_bezier_edge (ref_coord[0], S, S.degree(), k, _x);
          break;
      case cad_element::t:
          for (size_type k = 0; k < dimension(); k++)
            real_coord[k] = eval_bezier_triangle (ref_coord[0], ref_coord[1], S, S.degree(), k, _x);
          break;
      case cad_element::q:
          for (size_type k = 0; k < dimension(); k++)
            real_coord[k] = eval_bezier_quadrangle (ref_coord[0], ref_coord[1], S, S.degree(), S.degree2(), k, _x);
          break;

      default: 
	  error_macro ("unexpected type");
    }
}
// -----------------------------------------------------------------------
// input
// -----------------------------------------------------------------------
istream& 
operator >> (istream& is, cad_rep& x)
{
  iorheo::flag_type fmt = iorheo::flags(is) & iorheo::format_field;
  if      (fmt [iorheo::qmg])     { x.get_qmg (is); }
  else                            { x.get_rheo (is); }
  return is;
}
cad_rep::cad_rep (const string& file_name)
{
  irheostream is(file_name, "cad");
  check_macro(is.good(), "\"" << file_name << "[.cad[.gz]]\" not found.");
  is >> *this;
}
void
cad_rep::get_rheo (istream& is)
{
    fatal_macro ("input cad rheo format not yet supported");
}
// -----------------------------------------------------------------------
// output
// -----------------------------------------------------------------------
ostream& 
operator << (ostream& os, const cad_rep& x)
{
    typedef cad_rep::size_type size_type;

    bool verbose = iorheo::getverbose(os);
    bool clean   = iorheo::getclean(os);
    bool execute = iorheo::getexecute(os);
    string basename = iorheo::getbasename(os);
    if (basename.length() == 0) basename = "output";
    iorheo::flag_type fmt = iorheo::flags(os) & iorheo::format_field;

    if (fmt [iorheo::geomview]) { 

  	bool      adapt_degree = iorheo::getbezieradapt(os);
        size_type nsub         = iorheo::getsubdivide(os);
	x.geomview(basename, execute, clean, verbose, nsub, adapt_degree);

    } else if (fmt [iorheo::gnuplot]) { 

  	bool      adapt_degree = iorheo::getbezieradapt(os);
        size_type nsub         = iorheo::getsubdivide(os);
	x.gnuplot (basename, execute, clean, verbose, nsub, adapt_degree);

    } else { 

        x.put_rheo (os);
    }
    return os;
}
// -----------------------------------------------------------------------
// rheo output
// -----------------------------------------------------------------------
void
cad_element::put_rheo (ostream& os) const
{
    switch (type()) {
      case p: os << "p"; break;
      case e: os << "e " << _deg; break;
      case t: os << "t " << _deg; break;
      case q: os << "q " << _deg << " " << _deg2; break;
      default: fatal_macro("unexpected cad element");
    }
    os << "\t";
    for (size_type i = 0; i < size(); i++) {
	os << " " << operator[](i);
    }
}
void
cad_domain::put_rheo (ostream& os) const
{
    os << "domain" << endl
       << _name << endl
       << "1 " 
       << _dim << " " 
       << _boundary.size() << " " 
       << _lowdim.size() << " " 
       << size() << endl;

    for (vector<string>::const_iterator p = _boundary.begin(); p != _boundary.end(); p++)
      os << *p << " ";
    if (_boundary.size() != 0) os << endl;

    for (vector<string>::const_iterator p = _lowdim.begin(); p != _lowdim.end(); p++)
      os << *p << " ";
    if (_lowdim.size() != 0) os << endl;

    for (vector<cad_element>::const_iterator p = begin(); p != end(); p++) {
	(*p).put_rheo(os);
	os << endl;
    }
    os << endl;
}
void 
cad_rep::put_rheo(ostream& os) const
{
    os << "#!cad" << endl
       << endl
       << "cad" << endl
       << "1 " 
       << _dim << " " 
       << _x.size() << " " 
       << _p.size();
    if (_dim >= 1) { os << " " << _e.size(); }
    if (_dim >= 2) { os << " " << _s.size(); }
    if (_dim >= 3) { os << " " << _v.size(); }
    os << endl
       << endl;

    for (vector<point>::const_iterator p = _x.begin(); p != _x.end(); p++) {
	(*p).put (os, _dim);
	os << endl;
    }
    os << endl;
    for (vector<cad_domain>::const_iterator p = _p.begin(); p != _p.end(); p++)
		(*p).put_rheo(os);

    if (_dim >= 1) {
	for (vector<cad_domain>::const_iterator p = _e.begin(); p != _e.end(); p++)
		(*p).put_rheo(os);
    }
    if (_dim >= 2) {
        for (vector<cad_domain>::const_iterator p = _s.begin(); p != _s.end(); p++)
		(*p).put_rheo(os);
    }
    if (_dim >= 3) {
	for (vector<cad_domain>::const_iterator p = _v.begin(); p != _v.end(); p++)
		(*p).put_rheo(os);
    }
}
}// namespace rheolef
