/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdscomputer.h"
#include "rdscomputer_p.h"
#include <RdsSid>
#include <RdsUtils>

RdsComputer::RdsComputer()
		: RdsAdObject()
{

}

RdsComputer::RdsComputer(const RdsComputer &computer)
		: RdsAdObject(computer.dn())
{
	QXT_INIT_PRIVATE(RdsComputer);
}

RdsComputer::RdsComputer(const QString &dn)
		: RdsAdObject(dn)
{
	QXT_INIT_PRIVATE(RdsComputer);
}


RdsComputer::~RdsComputer()
{
}



ReturnValue RdsComputer::primaryGroup() const
{
	ReturnValue ret = sid();
	if (ret.isError()) return(ret);
	RdsSid sid = ret.value<RdsSid>();
	ret = readAttribute("primaryGroupID");
	if (ret.isError()) return(ret);

	LdapValues attr = ret.value<LdapValues>();
	sid.setRid(attr[0].toInt());
	return(RdsUtils::getObjectBySid(sid));
}

ReturnValue RdsComputer::setPrimaryGroup(const QString &groupdn)
{
	ReturnValue ret = rdsLdapSession()->read(groupdn, QStringList() << "objectSid");
	if (ret.isError()) return(ret);

	LdapResult result = ret.value<LdapResult>();
	if (!result.contains("objectsid") || (result["objectsid"].count() == 0))
	{
		return(ReturnValue(1, "objectSid attribute does not exist"));
	}

	RdsSid sid(result["objectsid"][0]);

	RdsLdapActions actions;
	actions.add(RdsLdapActions::Add, "member", dn());
	ret = rdsLdapSession()->modify(groupdn, actions);
	if (ret.isError() && (ret.errNumber() != 68) && (ret.errNumber() != 20)) return(ret);

	RdsLdapActions actions2;
	actions2.add(RdsLdapActions::Replace, "primaryGroupID", QString("%1").arg(sid.rid()));
	ret = modify(actions2);
	return(ret);
}

ReturnValue RdsComputer::cn() const
{
	ReturnValue ret = readAttribute("cn");
	if (ret.isError()) return(ret);
	LdapValues values = ret.value<LdapValues>();
	return(QString(values[0]));
}

ReturnValue RdsComputer::setCn(const QString &cn)
{
	return(rename("CN=" + cn));
}

ReturnValue RdsComputer::groups() const
{
	QStringList list;
	ReturnValue ret = primaryGroup();
	if (!ret.isError()) list << ret.toString();
	
	ret = read(QStringList() << "memberof");
	if (ret.isError()) return(ret);

	LdapResult result = ret.value<LdapResult>();
	if (!result.contains("memberof")) return(list);

	foreach(QByteArray value, result["memberof"])
	{
		list << value;
	}

	return(list);
}

ReturnValue RdsComputer::joinGroup(const QString &groupdn)
{
	RdsLdapActions actions;
	actions.add(RdsLdapActions::Add, "member", dn());
	return(rdsLdapSession()->modify(groupdn, actions));

}

ReturnValue RdsComputer::leaveGroup(const QString &groupdn)
{
	RdsLdapActions actions;
	actions.add(RdsLdapActions::Remove, "member", dn());
	return(rdsLdapSession()->modify(groupdn, actions));
}

ReturnValue RdsComputer::description() const
{
	ReturnValue ret = read();
	if (ret.isError()) return(ret);
	LdapResult entry = ret.value<LdapResult>();

	if (!entry.contains("description") || entry["description"].count() == 0) return("");
	else return(entry["description"][0]);
}

ReturnValue RdsComputer::setDescription(const QString &desc)
{
	RdsLdapActions actions;

	if (desc == "")
		actions.add(RdsLdapActions::Remove, "description", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "description", desc);

	return(modify(actions));
}

ReturnValue RdsComputer::notes() const
{
	ReturnValue ret = readAttribute("info");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsComputer::setNotes(const QString &notes)
{
	RdsLdapActions actions;

	if (notes == "")
		actions.add(RdsLdapActions::Remove, "info", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "info", notes);

	return(modify(actions));
}

ReturnValue RdsComputer::location() const
{
	ReturnValue ret = readAttribute("location");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsComputer::setLocation(const QString &location)
{
	RdsLdapActions actions;

	if (location == "")
		actions.add(RdsLdapActions::Remove, "location", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "location", location);

	return(modify(actions));
}

ReturnValue RdsComputer::hostName() const
{
	ReturnValue ret = readAttribute("dNSHostName");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsComputer::setHostName(const QString &hostname)
{
	RdsLdapActions actions;

	if (hostname == "")
		actions.add(RdsLdapActions::Remove, "dNSHostName", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "dNSHostName", hostname);

	return(modify(actions));
}

ReturnValue RdsComputer::computerName() const
{
	ReturnValue ret = readAttribute("sAMAccountName");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	QString str = QString(value[0]);
	str.replace(QRegExp("[$]$"), "");
	return(str);
}

ReturnValue RdsComputer::setComputerName(const QString &computer)
{
	RdsLdapActions actions;

	if (computer == "")
		actions.add(RdsLdapActions::Remove, "sAMAccountName", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "sAMAccountName", computer + "$");

	return(modify(actions));
}

ReturnValue RdsComputer::operatingSystem() const
{
	ReturnValue ret = readAttribute("operatingSystem");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsComputer::setOperatingSystem(const QString &os)
{
	RdsLdapActions actions;

	if (os == "")
		actions.add(RdsLdapActions::Remove, "operatingSystem", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "operatingSystem", os);

	return(modify(actions));
}

ReturnValue RdsComputer::operatingSystemVersion() const
{
	ReturnValue ret = readAttribute("operatingSystemVersion");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsComputer::setOperatingSystemVersion(const QString &version)
{
	RdsLdapActions actions;

	if (version == "")
		actions.add(RdsLdapActions::Remove, "operatingSystemVersion", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "operatingSystemVersion", version);

	return(modify(actions));
}

ReturnValue RdsComputer::servicePack() const
{
	ReturnValue ret = readAttribute("operatingSystemServicePack");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsComputer::setServicePack(const QString &servicepack)
{
	RdsLdapActions actions;

	if (servicepack == "")
		actions.add(RdsLdapActions::Remove, "operatingSystemServicePack", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "operatingSystemServicePack", servicepack);

	return(modify(actions));
}

QDataStream& operator<<(QDataStream& d, const RdsComputer& object)
{
	d << object.dn();
	return(d);
}

QDataStream& operator>>(QDataStream& d, RdsComputer& object)
{
	QString dn;
	d >> dn;
	object.setDn(dn);
	return(d);
}

