/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdsvolumemanager.h"
#include "rdsvolumemanager_p.h"

#include <QDebug>
#include <QStringList>

#include "RdsSettings"
#include "RdsEntity"
#include "RdsVolume"
#include "rdsvolume_p.h"
#include "rdssinglevolume_p.h"

#define ITERCOUNT 20
QTRPC_SERVICEPROXY_PIMPL_IMPLEMENT(RdsVolumeManager);

bool& RdsVolumeManagerPrivate::isConnected()
{
	static bool isConnected = false;
	return isConnected;
}

QStringList& RdsVolumeManagerPrivate::volumeBlacklist()
{
	static QStringList list;
	return list;
}

QMutex& RdsVolumeManagerPrivate::mutex()
{
	static QMutex mutex(QMutex::Recursive);
	return mutex;
}

RdsVolumeManager::RdsVolumeManager()
		: RdsEntityManager()
{
	QXT_INIT_PRIVATE();
	qxt_d().init();
}

RdsVolumeManager::RdsVolumeManager(const RdsVolumeManager &other)
		: RdsEntityManager(other)
{
	QXT_INIT_PRIVATE();
	qxt_d().init();
}

void RdsVolumeManagerPrivate::init()
{
	QMutexLocker locker(&mutex());
	if (isConnected())
		return;
	else
		isConnected() = true;

	RdsVolumeManagerThread* thread = new RdsVolumeManagerThread();
	thread->start();

	RdsUDisksInterface* interface = new RdsUDisksInterface(RdsUDisksInterface::staticInterfaceName(), "/org/freedesktop/UDisks", QDBusConnection::systemBus());

	interface->moveToThread(thread);
	qxt_p().moveToThread(thread);

	QObject::connect(&qxt_p(), SIGNAL(destroyed()), interface, SLOT(deleteLater()), Qt::QueuedConnection);
	QObject::connect(&qxt_p(), SIGNAL(destroyed()), thread, SLOT(deleteLater()), Qt::QueuedConnection);
	QObject::connect(interface, SIGNAL(DeviceAdded(const QDBusObjectPath &)), &qxt_p(), SLOT(deviceAdded(const QDBusObjectPath &)), Qt::QueuedConnection);
	QObject::connect(interface, SIGNAL(DeviceChanged(const QDBusObjectPath &)), &qxt_p(), SLOT(deviceChanged(const QDBusObjectPath &)), Qt::QueuedConnection);
	QObject::connect(interface, SIGNAL(DeviceRemoved(const QDBusObjectPath &)), &qxt_p(), SLOT(deviceRemoved(const QDBusObjectPath &)), Qt::QueuedConnection);

	RdsUDisksInterface inter(RdsUDisksInterface::staticInterfaceName(), "/org/freedesktop/UDisks", QDBusConnection::systemBus());
	QDBusPendingReply<QList<QDBusObjectPath> > ret = inter.EnumerateDevices();
	ret.waitForFinished();
	if (ret.isError())
		return;
	foreach(QDBusObjectPath path, ret.value())
	{
		RdsUDisksDeviceInterface device(RdsUDisksInterface::staticInterfaceName(), path.path(), QDBusConnection::systemBus());
		if (!RdsVolumePrivate::isVolume(device))
			continue;
		volumes[path.path()] = device.idUuid();
		deviceFiles[path.path()] = device.deviceFile();
		ReturnValue ret = qxt_p().volume(device.idUuid());
		if (ret.isError())
		{
			qWarning() << "Failed to read volume for initialization," << ret;
			continue;
		}
		RdsVolume v = ret;
		ret = v.isAdded();
		if (ret.isError())
		{
			qWarning() << "Error while checking if volume is added during initialization," << ret;
			continue;
		}
		if (!ret.toBool())
		{
			//qDebug() << v.name().toString() << "is not added";
			continue;
		}

		ret = v.isActivated();
		if (ret.isError())
		{
			qWarning() << "Error while checking if volume is activated during initialization," << ret;
			continue;
		}
		if (ret.toBool())
		{
			//qDebug() << v.name().toString() << "is already activated";
			continue;
		}

		//qDebug() << "Auto-mounting" << v.name().toString();
		ret = v.mount();
		if (ret.isError())
		{
			qCritical() << "Error while mounting volume," << ret;
			continue;
		}
	}
}

void RdsVolumeManager::deviceAdded(const QDBusObjectPath &path)
{
	QMutexLocker locker(&qxt_d().mutex());
	RdsUDisksDeviceInterface device(RdsUDisksInterface::staticInterfaceName(), path.path(), QDBusConnection::systemBus());
	QString uuid = device.idUuid();
	if (qxt_d().volumeBlacklist().contains(uuid))
		return;
	if (uuid.isEmpty())
		return;
	//qDebug() << "This device was added" << uuid << thread();
	RdsVolume v(uuid);
	ReturnValue ret = v.mount();
	if (ret.isError())
		qCritical() << "Failed to auto-mount device!" << ret;
	qxt_d().volumes[path.path()] = uuid;
	qxt_d().deviceFiles[path.path()] = device.deviceFile();
	addEntity(uuid);
}

void RdsVolumeManager::deviceChanged(const QDBusObjectPath &path)
{
	QMutexLocker locker(&qxt_d().mutex());
	RdsUDisksDeviceInterface device(RdsUDisksInterface::staticInterfaceName(), path.path(), QDBusConnection::systemBus());
	QString uuid = device.idUuid();
	if (uuid.isEmpty())
		return;
	//qDebug() << "This device was changed" << uuid;
	qxt_d().volumes[path.path()] = uuid;
	qxt_d().deviceFiles[path.path()] = device.deviceFile();
	updateEntity(uuid);
}

void RdsVolumeManager::deviceRemoved(const QDBusObjectPath &path)
{
	QMutexLocker locker(&qxt_d().mutex());
	RdsUDisksDeviceInterface device(RdsUDisksInterface::staticInterfaceName(), path.path(), QDBusConnection::systemBus());
	QString uuid = qxt_d().volumes.take(path.path());
	QString deviceFile  = qxt_d().deviceFiles.take(path.path());
	if (deviceFile.size() > 4)
		QProcess::execute("umount", QStringList() << "-l" << deviceFile);
	if (uuid.isEmpty())
		return;
	//qDebug() << "This device was removed" << uuid;
	RdsVolume v(uuid);
	if (v.isAdded().toBool())
		updateEntity(uuid);
	else
		removeEntity(uuid);
}

RdsVolumeManager::~RdsVolumeManager()
{
	QMutexLocker locker(&qxt_d().mutex());
}

RdsVolumeManager& RdsVolumeManager::operator=(const RdsVolumeManager & other)
{
	Q_UNUSED(other);
	return *this;
}

ReturnValue RdsVolumeManager::auth(QtRpc::AuthToken token)
{
	createInternalObject();
	if (token.serverData().contains("authenticated") && (token.serverData().value("authenticated").toBool() == true))
		return(true);
	else
		return(ReturnValue(1, "Not Authenticated"));
}

ReturnValue RdsVolumeManager::list() const
{
	return RdsVolume::list();
}

ReturnValue RdsVolumeManager::volume(const QString &vol)
{
	//qDebug() << "Getting" << vol;
	if (!RdsSingleVolumePrivate::hasUDisks())
		return (RdsVolume*)(new RdsSingleVolume());
	return new RdsVolume(vol);
}

ReturnValue RdsVolumeManager::addVolume(const QString &vol, const QString &name)
{
	//qDebug() << "Adding" << vol;
	RdsVolume v(vol);
	ReturnValue ret = v.id();
	if (ret.isError())
		return ret;
	rdssettings().lock();
	rdssettings()->beginGroup("volume-" + name);
	rdssettings()->setValue("uuid", ret);
	rdssettings()->endGroup();
	rdssettings().unlock();
	v.setName(name);
	return true;
}

ReturnValue RdsVolumeManager::removeVolume(const QString &vol)
{
	RdsVolume v(vol);
	if (v.id().toString() == rootVolume().toString() && !rootVolume().toString().isEmpty())
	{
		return ReturnValue(1, "You cannot remove the root device");
	}
	rdssettings().lock();
	rdssettings()->beginGroup("volume-" + vol);
	foreach(QString str, rdssettings()->childKeys())
	{
		rdssettings()->remove(str);
	}
	rdssettings()->endGroup();
	rdssettings().unlock();
	return true;
}

ReturnValue RdsVolumeManager::rootVolume() const
{
	if (!RdsSingleVolumePrivate::hasUDisks())
		return "Hard Drive";
	QStringList volumes = list().toStringList();
	foreach(QString volume, volumes)
	{
		RdsVolume v(volume);
		if (v.mountPoint().toString() == "/")
			return v.id();
	}
	return ReturnValue(1, "Failed to find root device. Is DBus broken?");
}

ReturnValue RdsVolumeManager::defaultVolume() const
{
	QString winner;
	qlonglong wsize = 0;
	QStringList volumes = list().toStringList();
	foreach(QString volume, volumes)
	{
		RdsVolume v(volume);
		qlonglong size = v.freeSpace().toLongLong();
		if (size > wsize)
		{
			wsize = size;
			winner = volume;
		}
	}
	return winner;
}

ReturnValue RdsVolumeManager::listEntities(const QString &dn, bool loadmore) const
{
	ReturnValue ret;
	if ((dn == "") || (dn == "root"))
	{
		RdsEntity entity;
		entity.setId("root");
		entity.setType("root");
		entity.setVisible(false);
		entity.setName("");
		entity.setParent("");

		//qDebug() << "This is the volume list I got" << list().toStringList();

		foreach(QString volume, list().toStringList())
		{
			ret = listEntities(volume, loadmore);
			if (!ret.isError()) entity.children() << ret.value<RdsEntity>();
		}
		return ReturnValue::fromValue<RdsEntity>(entity);
	}
	if (!RdsSingleVolumePrivate::hasUDisks())
	{
		if (dn != "Hard Drive")
			return ReturnValue(1, "Failed to find a volume by that name");
		RdsEntity entity;

		RdsSingleVolume v;
		entity.setId(dn);
		entity.setType("volume");
		entity.setVisible(true);
		entity.setName(v.name().toString());
		entity.setParent("root");
		entity.metadata()["added"] = v.isAdded().toBool();
		entity.metadata()["mounted"] = v.isActivated().toBool();
		entity.metadata()["uuid"] = v.id().toString();
		entity.metadata()["size"] = v.size().toULongLong();
		entity.metadata()["free"] = v.freeSpace().toULongLong();
		entity.metadata()["type"] = v.type().toString();
		entity.metadata()["status"] = v.status().toString();
		entity.metadata()["statustext"] = v.statusText().toString();
		entity.metadata()["statusdetails"] = v.statusDetails().toString();
		entity.metadata()["ejectable"] = v.isEjectable().toBool();

		return ReturnValue::fromValue<RdsEntity>(entity);
	}
	RdsEntity entity;
	QString volume = dn;
	RdsVolume v(volume);
	if (v.id().toString().isEmpty())
		return ReturnValue(1, "Failed to find a volume by that name");

	entity.setId(dn);
	entity.setType("volume");
	entity.setVisible(true);
	entity.setName(v.name().toString());
	entity.setParent("root");
	entity.metadata()["added"] = v.isAdded().toBool();
	entity.metadata()["mounted"] = v.isActivated().toBool();
	entity.metadata()["uuid"] = v.id().toString();
	entity.metadata()["size"] = v.size().toULongLong();
	entity.metadata()["free"] = v.freeSpace().toULongLong();
	entity.metadata()["type"] = v.type().toString();
	entity.metadata()["status"] = v.status().toString();
	entity.metadata()["statustext"] = v.statusText().toString();
	entity.metadata()["statusdetails"] = v.statusDetails().toString();
	entity.metadata()["ejectable"] = v.isEjectable().toBool();

	return ReturnValue::fromValue<RdsEntity>(entity);
}

RdsVolumeManagerThread::RdsVolumeManagerThread()
{
}

void RdsVolumeManagerThread::run()
{
	//qDebug() << "Started thread for RdsVolumeManager" << this << thread();
	exec();
}


