/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdsusergroupcomputermanager.h"
#include <RdsUtils>
#include <RdsLdapSession>
#include <RdsUser>
#include <RdsGroup>
#include <RdsComputer>
#include <RdsOrganizationalUnit>
#include <RdsEntity>
#include <QDebug>
#include <RdsSid>

using namespace QtRpc;

RdsUserGroupComputerManager::RdsUserGroupComputerManager(QObject *parent)
		: RdsEntityManager(parent)
{
}


RdsUserGroupComputerManager::~RdsUserGroupComputerManager()
{
}

ReturnValue RdsUserGroupComputerManager::auth(AuthToken token)
{
	createInternalObject();
	if (token.serverData().contains("authenticated") && (token.serverData().value("authenticated").toBool() == true))
		return(true);
	else
		return(ReturnValue(1, "Not Authenticated"));
}

ReturnValue RdsUserGroupComputerManager::listEntities(const QString &dn, bool loadmore) const
{
	if ((dn == "") || (dn == "root"))
	{
		RdsEntity entity;
		entity.setId("root");
		entity.setType("root");
		entity.setVisible(false);
		entity.setName("");
		entity.setParent("");
		entity.setGroup(true);

		RdsOrganizationalUnit ou(dn);
		ReturnValue ret = ou.listOus();
		if (!ret.isError())
		{
			foreach(QString entdn, ret.toStringList())
			{
				ret = listEntities(entdn, loadmore);
				if (!ret.isError()) entity.children() << ret.value<RdsEntity>();
			}
		}

		ret = ou.listUsers();
		if (!ret.isError())
		{
			foreach(QString entdn, ret.toStringList())
			{
				ret = listEntities(entdn, loadmore);
				if (!ret.isError()) entity.children() << ret.value<RdsEntity>();
			}
		}

		ret = ou.listGroups();
		if (!ret.isError())
		{
			foreach(QString entdn, ret.toStringList())
			{
				ret = listEntities(entdn, loadmore);
				if (!ret.isError()) entity.children() << ret.value<RdsEntity>();
			}
		}

		ret = ou.listComputers();
		if (!ret.isError())
		{
			foreach(QString entdn, ret.toStringList())
			{
				ret = listEntities(entdn, loadmore);
				if (!ret.isError()) entity.children() << ret.value<RdsEntity>();
			}
		}
		
		ret = RdsUtils::getObjectBySid(RdsSid("S-1-1-0"));
		if (!ret.isError())
		{
			ret = listEntities(ret.toString(), loadmore);
			if (!ret.isError()) entity.children() << ret.value<RdsEntity>();
		}
		
		ret = RdsUtils::getObjectBySid(RdsSid("S-1-5-18"));
		if (!ret.isError())
		{
			ret = listEntities(ret.toString(), loadmore);
			if (!ret.isError()) entity.children() << ret.value<RdsEntity>();
		}
		
		ret = listEntities("CN=Builtin," + RdsUtils::baseDn(), loadmore);
		if (!ret.isError()) entity.children() << ret.value<RdsEntity>();
		
		return(QVariant::fromValue<RdsEntity>(entity));
	}
	else if(dn.toLower().startsWith("cn=everyone,cn=wellknown security principals"))
	{
		RdsEntity entity;
		entity.setId(dn);
		entity.setType("special");
		entity.setVisible(true);
		entity.setName("Everyone");
		entity.setGroup(false);
		entity.metadata()["description"] = RdsUser(dn).description();
		
		return(QVariant::fromValue<RdsEntity>(entity));
	}
	else if(dn.toLower().startsWith("cn=system,cn=wellknown security principals"))
	{
		RdsEntity entity;
		entity.setId(dn);
		entity.setType("special");
		entity.setVisible(true);
		entity.setName("System");
		entity.setGroup(false);
		entity.metadata()["description"] = RdsUser(dn).description();
		
		return(QVariant::fromValue<RdsEntity>(entity));
	}
	else
	{
		bool isbuiltin = (dn == "CN=Builtin," + RdsUtils::baseDn());
		
		ReturnValue ret = rdsLdapSession()->read(dn, QStringList() << "objectClass");
		if (ret.isError()) return(ret);
		LdapResult result = ret.value<LdapResult>();
		if (!result.contains("objectclass")) return(ReturnValue(1, "Failed to read objectclass of " + dn));
		
		//Organizational unit
		if (result["objectclass"].contains(QByteArray("organizationalUnit")) || isbuiltin)
		{
			RdsOrganizationalUnit ou(dn);

			RdsEntity entity;
			entity.setId(dn);
			entity.setType("ou");
			entity.setVisible(true);
			entity.setName(isbuiltin ? "Built In" : ou.ou().toString());
			entity.setGroup(true);
			entity.metadata()["description"] = RdsUser(dn).description();
			
			if (dn == RdsOrganizationalUnit().dn())
			{
				entity.setParent("");
			}
			else
			{
				int index = dn.indexOf(",");
				QString base = dn.mid(index + 1);
				if (base == RdsOrganizationalUnit().dn())
				{
					entity.setParent("root");
				}
				else
				{
					entity.setParent(base);
				}
			}

			if (loadmore)
			{
				ret = ou.listOus();
				if (!ret.isError())
				{
					foreach(QString entdn, ret.toStringList())
					{
						ret = listEntities(entdn, false);
						if (!ret.isError()) entity.children() << ret.value<RdsEntity>();
					}
				}

				ret = ou.listUsers();
				if (!ret.isError())
				{
					foreach(QString entdn, ret.toStringList())
					{
						ret = listEntities(entdn, false);
						if (!ret.isError()) entity.children() << ret.value<RdsEntity>();
					}
				}

				ret = ou.listGroups();
				if (!ret.isError())
				{
					foreach(QString entdn, ret.toStringList())
					{
						ret = listEntities(entdn, false);
						if (!ret.isError()) entity.children() << ret.value<RdsEntity>();
					}
				}

				ret = ou.listComputers();
				if (!ret.isError())
				{
					foreach(QString entdn, ret.toStringList())
					{
						ret = listEntities(entdn, loadmore);
						if (!ret.isError()) entity.children() << ret.value<RdsEntity>();
					}
				}
			}
			else
			{
				entity.setHasMore(true);
			}
			

			return(QVariant::fromValue<RdsEntity>(entity));
		}
		else if (result["objectclass"].contains(QByteArray("computer")))
		{
			RdsComputer user(dn);

			RdsEntity entity;
			entity.setId(dn);
			entity.setType("computer");
			entity.setVisible(true);
			entity.setName(user.cn().toString());
			entity.metadata()["description"] = RdsUser(dn).description();

			if (dn == RdsOrganizationalUnit().dn())
			{
				entity.setParent("");
			}
			else
			{
				int index = dn.indexOf(",");
				QString base = dn.mid(index + 1);
				if ((base == RdsOrganizationalUnit().dn()) || (base == RdsOrganizationalUnit().computerDn()))
				{
					entity.setParent("root");
				}
				else
				{
					entity.setParent(base);
				}
			}

			return(QVariant::fromValue<RdsEntity>(entity));
		}
		else if (result["objectclass"].contains(QByteArray("user")))
		{
			RdsUser user(dn);

			RdsEntity entity;
			entity.setId(dn);
			entity.setType("user");
			entity.setVisible(true);
			entity.setName(user.cn().toString());
			entity.metadata()["flags"] = user.flags();
			entity.metadata()["description"] = RdsUser(dn).description();

			if (dn == RdsOrganizationalUnit().dn())
			{
				entity.setParent("");
			}
			else
			{
				int index = dn.indexOf(",");
				QString base = dn.mid(index + 1);
				if ((base == RdsOrganizationalUnit().dn()) || (base == RdsOrganizationalUnit().userDn()))
				{
					entity.setParent("root");
				}
				else
				{
					entity.setParent(base);
				}
			}

			return(QVariant::fromValue<RdsEntity>(entity));
		}
		else if (result["objectclass"].contains(QByteArray("group")))
		{
			RdsGroup group(dn);

			RdsEntity entity;
			entity.setId(dn);
			entity.setType("group");
			entity.setVisible(true);
			entity.setName(group.cn().toString());
			entity.metadata()["description"] = RdsUser(dn).description();

			if (dn == RdsOrganizationalUnit().dn())
			{
				entity.setParent("");
			}
			else
			{
				int index = dn.indexOf(",");
				QString base = dn.mid(index + 1);
				if ((base == RdsOrganizationalUnit().dn()) || (base == RdsOrganizationalUnit().userDn()))
				{
					entity.setParent("root");
				}
				else
				{
					entity.setParent(base);
				}
			}

			return(QVariant::fromValue<RdsEntity>(entity));
		}
		else
		{
			return ReturnValue(1, "Unknown entity type: " + dn);
		}
	}
}
