/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "config.h"
#include "rdssambaconfigparser.h"
#include "rdssambaconfigparser_p.h"
#include <QDebug>
#include <QProcess>
#include <QHash>
#include <QMap>
#include <RdsUtils>

#include <RdsDaemonManager>

#define RDS_LOCK QMutexLocker __locker(&RdsSambaConfigParserPrivate::mutex());

QTRPC_SERVICEPROXY_PIMPL_IMPLEMENT(RdsSambaConfigParser);
RDS_REGISTER_DAEMON(RdsSambaConfigParser, "Samba");

bool& RdsSambaConfigParserPrivate::parsed()
{
	static bool parsed(false);
	return parsed;
}

QMutex& RdsSambaConfigParserPrivate::mutex()
{
	static QMutex mutex(QMutex::Recursive);
	return mutex;
}

QMap<QString, QMap<QString, QPair<QStringList, QString> > >& RdsSambaConfigParserPrivate::values()
{
	static QMap<QString, QMap<QString, QPair<QStringList, QString> > > values;
	return values;
}

QMap<QString, QMap<QString, QPair<QStringList, QString> > >& RdsSambaConfigParserPrivate::rdsValues()
{
	static QMap<QString, QMap<QString, QPair<QStringList, QString> > > rdsValues;
	return rdsValues;
}

QMap<QString, QMap<QString, QPair<QStringList, QString> > >& RdsSambaConfigParserPrivate::disabledValues()
{
	static QMap<QString, QMap<QString, QPair<QStringList, QString> > > disabledValues;
	return disabledValues;
}

QMap<QString, QMap<QString, QPair<QStringList, QString> > >& RdsSambaConfigParserPrivate::disabledRdsValues()
{
	static QMap<QString, QMap<QString, QPair<QStringList, QString> > > disabledRdsValues;
	return disabledRdsValues;
}

RdsSambaConfigParser::RdsSambaConfigParser()
		: ServiceProxy(NULL),
		RdsDaemonInstance()
{
	qxt_d().configFile = RDS_SAMBA_CONFIGFILE;
	RDS_LOCK;
	if (!qxt_d().parsed())
		reload();
}

RdsSambaConfigParser::RdsSambaConfigParser(const RdsSambaConfigParser &other)
		: ServiceProxy(NULL),
		RdsDaemonInstance()
{
	qxt_d().configFile = other.qxt_d().configFile;
	RDS_LOCK;
	if (!qxt_d().parsed())
		reload();
}

RdsSambaConfigParser::RdsSambaConfigParser(const QString &filename)
		: ServiceProxy(NULL),
		RdsDaemonInstance()
{
	Q_ASSERT_X(false, "WRONG", "Don't do that. It doesn't work yet.");
	qxt_d().configFile = filename;
	reload();
}

RdsSambaConfigParser::RdsSambaConfigParser(const char* filename)
		: ServiceProxy(NULL),
		RdsDaemonInstance()
{
	Q_ASSERT_X(false, "WRONG", "Don't do that. It doesn't work yet.");
	qxt_d().configFile = filename;
	reload();
}

RdsSambaConfigParser::~RdsSambaConfigParser()
{
}

RdsSambaConfigParser& RdsSambaConfigParser::operator=(const RdsSambaConfigParser & other)
{
	qxt_d().configFile = other.qxt_d().configFile;
	return *this;
}

ReturnValue RdsSambaConfigParser::stopService()
{
	return RdsUtils::runCommand(RDS_INITD_SAMBA4, QStringList() << "stop");
}

ReturnValue RdsSambaConfigParser::startService()
{
	return RdsUtils::runCommand(RDS_INITD_SAMBA4, QStringList() << "restart");
}

ReturnValue RdsSambaConfigParser::restartService()
{
	return RdsUtils::runCommand(RDS_INITD_SAMBA4, QStringList() << "restart");
}

ReturnValue RdsSambaConfigParser::reloadService()
{
	return RdsUtils::runCommand(RDS_INITD_SAMBA4, QStringList() << "restart");
}

ReturnValue RdsSambaConfigParser::reloadConfig()
{
	return reload();
}

bool RdsSambaConfigParser::parse(const QString &data)
{
	RDS_LOCK;
	qxt_d().parsed() = true;
	qxt_d().values() = QMap<QString, QMap<QString, QPair<QStringList, QString> > >();
	qxt_d().rdsValues() = QMap<QString, QMap<QString, QPair<QStringList, QString> > >();
	qxt_d().disabledValues() = QMap<QString, QMap<QString, QPair<QStringList, QString> > >();
	qxt_d().disabledRdsValues() = QMap<QString, QMap<QString, QPair<QStringList, QString> > >();
	QStringList comments;
	comments = QStringList() << QString();
	QStringList *currentComment = &comments;
	bool commented = false;
	bool readValue = false;
	bool rdsValue = false;
	quint32 wordNum = 1;
	quint32 lineNumber = 1;
	QString disabledSection;
	QString section;
	QString key;
	QString value;
	QString word;
	QString line;

	for (QString::const_iterator i = data.begin(); i != data.end(); ++i)
	{
		if (*i == '\n')
			lineNumber++;

		if (commented && !readValue)
		{
			comments[comments.count() - 1] += *i;
		}
		if (*i == ';' || *i == '#' || *i == ' ' || *i == '\t' || *i == '\n')
		{
			if (!word.isEmpty())
			{
				line += word + " ";
			}
			if (!word.isEmpty())
			{
				if (word == "=")
				{
					readValue = true;
					comments[comments.count() - 1] = QString();
					if (commented && !disabledSection.isEmpty())
					{
						if (rdsValue)
						{
							currentComment = &qxt_d().disabledRdsValues()[disabledSection][key].first;
						}
						else
						{
							currentComment = &qxt_d().disabledValues()[disabledSection][key].first;
						}
					}
					else if (!commented)
					{
						if (rdsValue)
						{
							currentComment = &qxt_d().rdsValues()[section][key].first;
						}
						else
						{
							currentComment = &qxt_d().values()[section][key].first;
						}
					}
					if (currentComment != &comments)
					{
						comments.removeLast();
						for (QStringList::iterator i = comments.begin(); i != comments.end(); ++i)
						{
							if (i->startsWith("(RDS)"))
							{
								i = comments.erase(i);
								i--;
							}
						}
						*currentComment << comments;
						comments = QStringList() << QString();
					}
					currentComment = &comments;
				}
				else
				{
					if (readValue)
					{
						value = (value + " " + word).simplified();
					}
					else
					{
						key = (key + " " + word).simplified();
					}
				}
				word = QString::Null();
				wordNum++;
			}
			if (*i == '\n' || *i == ';' || *i == '#')
			{
				if (!line.isEmpty())
				{
					if (readValue)
					{
						if (commented)
						{
							if (!disabledSection.isEmpty())
							{
								if (rdsValue)
								{
									qxt_d().disabledRdsValues()[disabledSection][key].second = value;
								}
								else
								{
									qxt_d().disabledValues()[disabledSection][key].second = value;
								}
							}
						}
						else
						{
							if (!section.isEmpty())
							{
								if (rdsValue)
								{
									qxt_d().rdsValues()[section][key].second = value;
								}
								else
								{
									qxt_d().values()[section][key].second = value;
								}
							}
						}
					}
					else
					{
						if (key.startsWith("[") && key.endsWith("]"))
						{
							if (!commented)
							{
								section = key.mid(1, key.count() - 2);
								qxt_d().values()[section];
								qxt_d().rdsValues()[section];
							}
							disabledSection = key.mid(1, key.count() - 2);
							qxt_d().disabledValues()[disabledSection];
							qxt_d().disabledRdsValues()[disabledSection];
						}
						else if (key.startsWith("[") || key.endsWith("]"))
						{
							qCritical() << "Syntax error on line" << lineNumber << ":" << line;
						}
					}
				}
				if (*i == '\n')
				{
					if (!comments.at(comments.count() - 1).isEmpty())
					{
						if (comments.at(comments.count() - 1).startsWith("(RDS)"))
							comments[comments.count() - 1] = QString();
						else
							comments << QString();
					}
					key = QString::null;
					line = QString::null;
					value = QString::null;
					wordNum = 1;
					readValue = false;
					rdsValue = false;
					commented = false;
				}
			}
			if (*i == ';' && *(i + 1) == ';')
			{
				i++;
				if (rdsValue || (!word.isEmpty()) || (!line.isEmpty()))
				{
					commented = true;
				}
				else
				{
					rdsValue = true;
				}
				continue;
			}
			if (*i == '#' || *i == ';') // We already check if it's a ;; up there
			{
				commented = true;
				continue;
			}
			continue;
		}
		// This makes escape sequences work correctly... Commented out because we don't handle them
// 		if (*i == '\\')
// 		{
// 			i++;
// 		}
		// Escapable newlines!
		if (*i == '\\' && *(i + 1) == '\n')
			i++;
		word += *i;
	}
	return true;
}

QString RdsSambaConfigParser::toString() const
{
	RDS_LOCK;
	QString output = "#(RDS) This file was automatically generated by RDS\n#(RDS) If you make any manual edits be sure to reload the configuration file in RDS else all changes will be lost\n";

	QStringList sections;
	QStringList disabledSections;
	sections << qxt_d().values().keys() << qxt_d().rdsValues().keys();
	sections = QStringList::fromSet(sections.toSet());
	foreach(QString section, qxt_d().disabledValues().keys())
	{
		if (!sections.contains(section) && !disabledSections.contains(section))
			disabledSections << section;
	}
	foreach(QString section, qxt_d().disabledRdsValues().keys())
	{
		if (!sections.contains(section) && !disabledSections.contains(section))
			disabledSections << section;
	}
	//qDebug() << disabledSections;
	foreach(QString section, sections)
	{
		output += "\n\n[" + section + "]\n";
		if (qxt_d().values().contains(section))
		{
			for (QMap<QString, QPair<QStringList, QString> >::const_iterator i = qxt_d().values().value(section).begin(); i != qxt_d().values().value(section).end(); ++i)
			{
				if (i.value().first.count() > 0)
					output += "	#" + i.value().first.join("	#") + "";
				output += "	" + i.key() + " = " + i.value().second + "\n";
			}
		}
		if (qxt_d().disabledValues().contains(section))
		{
			for (QMap<QString, QPair<QStringList, QString> >::const_iterator i = qxt_d().disabledValues().value(section).begin(); i != qxt_d().disabledValues().value(section).end(); ++i)
			{
				if (i.value().first.count() > 0)
					output += "	#" + i.value().first.join("	#") + "";
				output += "	; " + i.key() + " = " + i.value().second + "\n";
			}
		}
		if (qxt_d().rdsValues().contains(section))
		{
			for (QMap<QString, QPair<QStringList, QString> >::const_iterator i = qxt_d().rdsValues().value(section).begin(); i != qxt_d().rdsValues().value(section).end(); ++i)
			{
				if (i.value().first.count() > 0)
					output += "	#" + i.value().first.join("	#") + "";
				output += "	;; " + i.key() + " = " + i.value().second + "\n";
			}
		}
		if (qxt_d().disabledRdsValues().contains(section))
		{
			for (QMap<QString, QPair<QStringList, QString> >::const_iterator i = qxt_d().disabledRdsValues().value(section).begin(); i != qxt_d().disabledRdsValues().value(section).end(); ++i)
			{
				if (i.value().first.count() > 0)
					output += "	#" + i.value().first.join("	#") + "";
				output += "	; ;;" + i.key() + " = " + i.value().second + "\n";
			}
		}
	}
	foreach(QString section, disabledSections)
	{
		output += "\n\n; [" + section + "]\n";
		if (qxt_d().disabledValues().contains(section))
		{
			for (QMap<QString, QPair<QStringList, QString> >::const_iterator i = qxt_d().disabledValues().value(section).begin(); i != qxt_d().disabledValues().value(section).end(); ++i)
			{
				if (i.value().first.count() > 0)
					output += "	#" + i.value().first.join("	#") + "";
				output += "	; " + i.key() + " = " + i.value().second + "\n";
			}
		}
		if (qxt_d().disabledRdsValues().contains(section))
		{
			for (QMap<QString, QPair<QStringList, QString> >::const_iterator i = qxt_d().disabledRdsValues().value(section).begin(); i != qxt_d().disabledRdsValues().value(section).end(); ++i)
			{
				if (i.value().first.count() > 0)
					output += "	#" + i.value().first.join("	#") + "";
				output += "	; ;;" + i.key() + " = " + i.value().second + "\n";
			}
		}
	}
	return output;
}

ReturnValue RdsSambaConfigParser::save() const
{
	RDS_LOCK;
	QFile file(qxt_d().configFile);
	if (!file.open(QFile::WriteOnly | QFile::Truncate))
		return ReturnValue(1, "Failed to open file for writing " + QString::number(file.error()));
	file.write(toString().toLocal8Bit());
	return true;
}

ReturnValue RdsSambaConfigParser::reload()
{
	QFile file(qxt_d().configFile);
	if (!file.open(QFile::ReadWrite))
		return ReturnValue(1, "Failed to open file for reading");

	if (!parse(file.readAll()))
		return ReturnValue(1, "Failed to parse config file");
	return true;
}

ReturnValue RdsSambaConfigParser::fileName() const
{
	return qxt_d().configFile;
}

ReturnValue RdsSambaConfigParser::setFileName(const QString &fileName)
{
	Q_ASSERT_X(false, "WRONG", "Don't do that. It doesn't work yet.");
	qxt_d().configFile = fileName;
	return reload();
}

ReturnValue RdsSambaConfigParser::getValue(const QString &section, const QString &key) const
{
	RDS_LOCK;
	return qxt_d().values().value(section).value(key).second;
}

ReturnValue RdsSambaConfigParser::setValue(const QString &section, const QString &key, const QString &value)
{
	RDS_LOCK;
	qxt_d().values()[section][key].second = value;
	if (qxt_d().disabledValues().contains(section))
		qxt_d().disabledValues()[section].remove(key);
	return(true);
}

ReturnValue RdsSambaConfigParser::getRdsValue(const QString &section, const QString &key) const
{
	RDS_LOCK;
	return qxt_d().rdsValues().value(section).value(key).second;
}

ReturnValue RdsSambaConfigParser::setRdsValue(const QString &section, const QString &key, const QString &value)
{
	RDS_LOCK;
	qxt_d().rdsValues()[section][key].second = value;
	if (qxt_d().disabledRdsValues().contains(section))
		qxt_d().disabledRdsValues()[section].remove(key);
	return(true);
}


ReturnValue RdsSambaConfigParser::enableRdsKey(const QString &section, const QString &key)
{
	RDS_LOCK;
	if (qxt_d().disabledRdsValues().contains(section))
		qxt_d().rdsValues()[section][key] = qxt_d().disabledRdsValues()[section].take(key);
	else
		qxt_d().rdsValues()[section][key].second = QString();
	return(true);
}

ReturnValue RdsSambaConfigParser::disableRdsKey(const QString &section, const QString &key)
{
	RDS_LOCK;
	if (qxt_d().rdsValues().contains(section))
		qxt_d().disabledRdsValues()[section][key] = qxt_d().rdsValues()[section].take(key);
	else
		qxt_d().disabledRdsValues()[section][key].second = QString();
	return(true);
}

ReturnValue RdsSambaConfigParser::removeRdsKey(const QString &section, const QString &key)
{
	RDS_LOCK;
	if (qxt_d().rdsValues().contains(section))
		qxt_d().rdsValues()[section].remove(key);
	return(true);
}

ReturnValue RdsSambaConfigParser::enableSection(const QString &section)
{
	RDS_LOCK;
	qxt_d().values()[section] = QMap<QString, QPair<QStringList, QString> >();
	qxt_d().rdsValues()[section] = QMap<QString, QPair<QStringList, QString> >();
	return(true);
}

ReturnValue RdsSambaConfigParser::enableKey(const QString &section, const QString &key)
{
	RDS_LOCK;
	if (qxt_d().disabledValues().contains(section))
		qxt_d().values()[section][key] = qxt_d().disabledValues()[section].take(key);
	else
		qxt_d().values()[section][key].second = QString();
	return(true);
}

ReturnValue RdsSambaConfigParser::disableSection(const QString &section)
{
	RDS_LOCK;
	qxt_d().disabledValues()[section] = qxt_d().values().take(section);
	qxt_d().disabledRdsValues()[section] = qxt_d().rdsValues().take(section);
	return(true);
}

ReturnValue RdsSambaConfigParser::removeSection(const QString &section)
{
	RDS_LOCK;
	qxt_d().values().remove(section);
	qxt_d().rdsValues().remove(section);
	qxt_d().disabledValues().remove(section);
	qxt_d().disabledRdsValues().remove(section);
	return(true);
}

ReturnValue RdsSambaConfigParser::disableKey(const QString &section, const QString &key)
{
	RDS_LOCK;
	if (qxt_d().values().contains(section))
		qxt_d().disabledValues()[section][key] = qxt_d().values()[section].take(key);
	else
		qxt_d().disabledValues()[section][key];
	return(true);
}

ReturnValue RdsSambaConfigParser::removeKey(const QString &section, const QString &key)
{
	RDS_LOCK;
	if (qxt_d().values().contains(section))
		qxt_d().values()[section].remove(key);
	if (qxt_d().disabledValues().contains(section))
		qxt_d().disabledValues()[section].remove(key);
	return(true);
}

ReturnValue RdsSambaConfigParser::listSections(bool listCommented) const
{
	RDS_LOCK;
	QStringList sections;
	sections << qxt_d().values().keys();
	for (QMap<QString, QMap<QString, QPair<QStringList, QString> > >::const_iterator i = qxt_d().disabledValues().begin(); i != qxt_d().disabledValues().end(); ++i)
	{
		if (!sections.contains(i.key()))
			sections << i.key();
	}
	if (listCommented)
	{
		for (QMap<QString, QMap<QString, QPair<QStringList, QString> > >::const_iterator i = qxt_d().disabledValues().begin(); i != qxt_d().disabledValues().end(); ++i)
		{
			if (!sections.contains(i.key()) && i.value().count() > 0)
				sections << i.key();
		}
		for (QMap<QString, QMap<QString, QPair<QStringList, QString> > >::const_iterator i = qxt_d().disabledRdsValues().begin(); i != qxt_d().disabledRdsValues().end(); ++i)
		{
			if (!sections.contains(i.key()) && i.value().count() > 0)
				sections << i.key();
		}
	}
	return sections;
}

ReturnValue RdsSambaConfigParser::listKeys(const QString &section, bool listCommented) const
{
	RDS_LOCK;
	QStringList keys;
	keys << qxt_d().values().value(section).keys();
	if (listCommented && qxt_d().disabledValues().contains(section))
	{
		for (QMap<QString, QPair<QStringList, QString> >::const_iterator i = qxt_d().disabledValues().value(section).begin(); i != qxt_d().disabledValues().value(section).end(); ++i)
		{
			if (!keys.contains(i.key()))
				keys << i.key();
		}
	}
	return keys;
}

ReturnValue RdsSambaConfigParser::sectionExists(const QString &section, bool commented) const
{
	RDS_LOCK;
	if (qxt_d().values().contains(section))
		return true;
	if (qxt_d().rdsValues().contains(section))
		return true;
	if (!commented)
		return false;
	if (qxt_d().disabledValues().contains(section))
		return true;
	if (qxt_d().disabledRdsValues().contains(section))
		return true;
	return false;
}

ReturnValue RdsSambaConfigParser::keyExists(const QString &section, const QString &key, bool commented) const
{
	RDS_LOCK;
	if (qxt_d().values().value(section).contains(key))
	{
		return true;
	}
	if (!commented)
		return false;
	if (qxt_d().disabledValues().value(section).contains(key))
		return true;
	return false;
}

ReturnValue RdsSambaConfigParser::renameSection(const QString &section, const QString &newname)
{
	RDS_LOCK;
	if (qxt_d().values().contains(section))
		qxt_d().values()[newname] = qxt_d().values().take(section);
	if (qxt_d().rdsValues().contains(section))
		qxt_d().rdsValues()[newname] = qxt_d().rdsValues().take(section);
	if (qxt_d().disabledValues().contains(section))
		qxt_d().disabledValues()[newname] = qxt_d().disabledValues().take(section);
	if (qxt_d().disabledRdsValues().contains(section))
		qxt_d().disabledRdsValues()[newname] = qxt_d().disabledRdsValues().take(section);
	return true;
}

ReturnValue RdsSambaConfigParser::listRdsKeys(const QString &section, bool listCommented) const
{
	RDS_LOCK;
	QStringList keys;
	keys << qxt_d().rdsValues().value(section).keys();
	if (listCommented && qxt_d().disabledRdsValues().contains(section))
	{
		for (QMap<QString, QPair<QStringList, QString> >::const_iterator i = qxt_d().disabledRdsValues().value(section).begin(); i != qxt_d().disabledRdsValues().value(section).end(); ++i)
		{
			if (!keys.contains(i.key()))
				keys << i.key();
		}
	}
	return keys;
}

ReturnValue RdsSambaConfigParser::rdsKeyExists(const QString &section, const QString &key, bool commented) const
{
	RDS_LOCK;
	if (qxt_d().rdsValues().value(section).contains(key))
		return true;
	if (!commented)
		return false;
	if (qxt_d().disabledRdsValues().value(section).contains(key))
		return true;
	return false;
}

QMap<QString, QMap<QString, QString> > RdsSambaConfigParser::toMap(bool includeRdsValues) const
{
	RDS_LOCK;
	QMap<QString, QMap<QString, QString> > ret;
	typedef QMap<QString, QMap<QString, QPair<QStringList, QString> > > SambaValues;
	typedef QMap<QString, QPair<QStringList, QString> > SambaSection;
	for (SambaValues::const_iterator i = qxt_d().values().begin(); i != qxt_d().values().end(); ++i)
	{
		for (SambaSection::const_iterator j = i.value().begin(); j != i.value().end(); ++j)
		{
			ret[i.key()][j.key()] = j.value().second;
		}
	}
	if (!includeRdsValues)
		return ret;
	for (SambaValues::const_iterator i = qxt_d().rdsValues().begin(); i != qxt_d().rdsValues().end(); ++i)
	{
		for (SambaSection::const_iterator j = i.value().begin(); j != i.value().end(); ++j)
		{
			ret[i.key()][j.key()] = j.value().second;
		}
	}
	return ret;
}
