/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdsmountpointmanager.h"
#include "rdsmountpointmanager_p.h"
#include "rdsmountpoint_p.h"
#include <RdsMountPoint>
#include <RdsUtils>
#include <RdsEntity>
#include <QFile>
#include <QStringList>
#include <QDir>
#include <QDebug>

using namespace QtRpc;
QTRPC_SERVICEPROXY_PIMPL_IMPLEMENT(RdsMountPointManager);

RdsMountPointManager::RdsMountPointManager(QObject *parent)
		: RdsEntityManager(parent)
{
	QXT_INIT_PRIVATE(RdsMountPointManager);
}

RdsMountPointManager::RdsMountPointManager(const RdsMountPointManager &other)
		: RdsEntityManager(NULL)
{
	QXT_INIT_PRIVATE(RdsMountPointManager);
	(void)other;
}

RdsMountPointManager& RdsMountPointManager::operator=(const RdsMountPointManager & other)
{
	(void)other;
	return *this;
}


RdsMountPointManager::~RdsMountPointManager()
{
}

ReturnValue RdsMountPointManager::auth(QtRpc::AuthToken token)
{
	createInternalObject();
	if (token.serverData().contains("authenticated") && (token.serverData().value("authenticated").toBool() == true))
		return(true);
	else
		return(ReturnValue(1, "Not Authenticated"));
}

ReturnValue RdsMountPointManager::list() const
{
	QStringList mountpoints;

	QFile fsfile("/etc/fstab");
	if (!fsfile.open(QFile::ReadOnly))
	{
		return(ReturnValue(1, "Failed to open /etc/fstab"));
	}

	QString line;
	while (!fsfile.atEnd())
	{
		line = fsfile.readLine();
		line = line.simplified();
		if (line.startsWith("#")) continue;

		line = line.mid(line.indexOf(" ") + 1);
		line = line.left(line.indexOf(" "));
		line = RdsMountPointPrivate::decodeMountPath(line);

		mountpoints << RdsUtils::normalizePath(line);
	}

	return(mountpoints);
}

ReturnValue RdsMountPointManager::mountPoint(QString path) const
{
	return(new RdsMountPoint(path));
}

ReturnValue RdsMountPointManager::listEntities(const QString &id, bool loadmore) const
{
	if ((id == "") || (id == "root"))
	{
		RdsEntity entity;
		entity.setId("root");
		entity.setType("root");
		entity.setVisible(false);
		entity.setName("");
		entity.setParent("");

		ReturnValue ret = list();
		if (ret.isError()) return(ret);

		foreach(QString mnt, ret.toStringList())
		{
			if (mnt.startsWith("/volumes/"))
			{
				mnt = mnt.mid(9);
				ret = listEntities("@VOLUME/" + mnt + "/", false);
				if (ret.isError()) continue;
				entity.children() << ret.value<RdsEntity>();
			}
		}

		ret = listEntities("/", false);
		if (!ret.isError()) entity.children() << ret.value<RdsEntity>();

		return(QVariant::fromValue<RdsEntity>(entity));
	}
	else if (id == "/")
	{
		RdsEntity entity;
		entity.setId("/");
		entity.setType("filesystem");
		entity.setVisible(true);
		entity.setName("File System");
		entity.setParent("root");
		entity.setGroup(true);

		if (loadmore)
		{
			QDir dir("/");

			foreach(QString child, dir.entryList(QStringList(), QDir::QDir::AllDirs))
			{
				if ((child == "..") || (child == ".")) continue;

				QString childid = QString("/%1").arg(child);


				ReturnValue ret = listEntities(childid, false);
				if (ret.isError()) continue;
				entity.children() << ret.value<RdsEntity>();
			}
		}
		else
		{
			entity.setHasMore(true);
		}

		return(QVariant::fromValue<RdsEntity>(entity));
	}
	else if (id.startsWith("/"))
	{
		QString newid = RdsUtils::normalizePath(id);

		QDir dir(id);
		QDir tmpdir = dir;
		QString parent = "/";
		if (tmpdir.cdUp()) parent = RdsUtils::normalizePath(tmpdir.path());

		RdsEntity entity;
		entity.setId(id);
		entity.setType("folder");
		entity.setVisible(true);
		entity.setName(dir.dirName());
		entity.setParent(parent);
		entity.setGroup(true);

		if (loadmore)
		{
			foreach(QString child, dir.entryList(QStringList(), QDir::QDir::AllDirs))
			{
				if ((child == "..") || (child == ".")) continue;

				QString childid = QString("%1/%2").arg(newid).arg(child);


				ReturnValue ret = listEntities(childid, false);
				if (ret.isError()) continue;
				entity.children() << ret.value<RdsEntity>();
			}
		}
		else
		{
			entity.setHasMore(true);
		}

		return(QVariant::fromValue<RdsEntity>(entity));
	}
	else if (id.startsWith("@VOLUME/") && (id.count('/') == 2) && id.endsWith("/"))
	{
		ReturnValue ret = RdsUtils::getSystemPath(id);
		if (ret.isError()) return(ret);
		QString path = ret.toString();

		ret = RdsUtils::getVolumeName(id);
		if (ret.isError()) return(ret);
		QString name = ret.toString();

		RdsEntity entity;
		entity.setId(id);
		entity.setType("volume");
		entity.setVisible(true);
		entity.setName(name);
		entity.setParent("root");
		entity.setGroup(true);

		if (loadmore)
		{
			QDir dir(path);

			foreach(QString child, dir.entryList(QStringList(), QDir::QDir::AllDirs))
			{
				if ((child == "..") || (child == ".")) continue;

				QString childid = QString("@VOLUME/%1/%2").arg(name).arg(child);


				ReturnValue ret = listEntities(childid, false);
				if (ret.isError()) continue;
				entity.children() << ret.value<RdsEntity>();
			}
		}
		else
		{
			entity.setHasMore(true);
		}

		return(QVariant::fromValue<RdsEntity>(entity));
	}
	else if (id.startsWith("@VOLUME/"))
	{
		QString newid = RdsUtils::normalizePath(id);

		ReturnValue ret = RdsUtils::getSystemPath(id);
		if (ret.isError()) return(ret);
		QString path = ret.toString();

		ret = RdsUtils::getVolumeName(id);
		if (ret.isError()) return(ret);
		QString name = ret.toString();

		QDir dir(path);
		QDir tmpdir = dir;
		QString parent = "/volumes/" + name;
		if (tmpdir.cdUp()) parent = RdsUtils::normalizePath(tmpdir.path());

		ret = RdsUtils::getVolumePath(parent);
		if (ret.isError()) return(ret);
		parent = ret.toString();

		RdsEntity entity;
		entity.setId(id);
		entity.setType("folder");
		entity.setVisible(true);
		entity.setName(dir.dirName());
		entity.setParent(parent);
		entity.setGroup(true);

		if (loadmore)
		{
			foreach(QString child, dir.entryList(QStringList(), QDir::QDir::AllDirs))
			{
				if ((child == "..") || (child == ".")) continue;

				QString childid = QString("%1/%2").arg(newid).arg(child);

				ReturnValue ret = listEntities(childid, false);
				if (ret.isError()) continue;
				entity.children() << ret.value<RdsEntity>();
			}
		}
		else
		{
			entity.setHasMore(true);
		}

		return(QVariant::fromValue<RdsEntity>(entity));
	}
	else
	{
		return(ReturnValue(1, "Malformed Path: " + id));
	}
}
