/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#ifndef RDSJOB_H
#define RDSJOB_H

#include <ServiceProxy>
#include <QxtPimpl>
#include <QSharedPointer>
#include <QDateTime>

class RdsJobPrivate;

/**
	@author Chris Vickery <chris@resara.com>
*/
class RdsJob : public QtRpc::ServiceProxy
{
	Q_OBJECT;
	QTRPC_SERVICEPROXY_PIMPL(RdsJob);
	QXT_DECLARE_PRIVATE(RdsJob);
public:
	RdsJob(QObject *parent = 0);
	RdsJob(const RdsJob& other);
	~RdsJob();
	RdsJob& operator=(const RdsJob & other);
	enum Status
	{
		Stopped,
		Running,
		Paused,
		Finished,
		Failed
	};
	enum LogLevel
	{
		Info,
		Warning,
		Error
	};

	struct LogEntry
	{
		QString log;
		RdsJob::LogLevel level;
		QDateTime timestamp;
	};
public slots:
	ReturnValue name() const;
	ReturnValue description() const;
	ReturnValue status() const;
	ReturnValue progress() const;
	ReturnValue progressText() const;
	ReturnValue progressDetails() const;
	ReturnValue getLog() const;
	ReturnValue canPause() const;
	ReturnValue id() const;

	ReturnValue start();
	ReturnValue stop();
	ReturnValue pause();
	ReturnValue resume();

	ReturnValue setName(const QString& name);
	ReturnValue setDescription(const QString& desc);
	ReturnValue registerJob();
	ReturnValue unregisterJob();
	// WARNING! When this function is called, the shared pointer takes ownership of the object. That means you may NEVER call this function on an RdsJob that is sitting on the stack or that you're going to delete manually. Just don't do it!
	// Although, this shared pointer is shared with the QtRpc2 shared pointer system, so don't worry about that.
	QSharedPointer<RdsJob> getSharedPointer() const;

protected:
	QSharedPointer<RdsJob> getProxyPointer() const;

signals:
	QtRpc::Event progressChanged(int percent, const QString& progressText, const QString& progressDetails);
	QtRpc::Event statusChanged(RdsJob::Status status);
	QtRpc::Event newLogMessages(const QList<RdsJob::LogEntry> &logs);
	QtRpc::Event finished(const ReturnValue& ret);

private slots:
	ReturnValue sendNotifications();
	ReturnValue sendLogMessages();

protected:
	ReturnValue setStatus(RdsJob::Status status);
	ReturnValue setProgress(int percent);
	ReturnValue setProgressText(const QString& text);
	ReturnValue setProgressDetails(const QString& details);
	ReturnValue log(const QString& log);
	ReturnValue log(const QString& log, RdsJob::LogLevel level);
	ReturnValue setCanPause(bool pause);
	ReturnValue finish(const ReturnValue& ret);
	virtual ReturnValue startInternal();
	virtual ReturnValue stopInternal();
	virtual ReturnValue pauseInternal();
	virtual ReturnValue resumeInternal();
};

QDataStream& operator<<(QDataStream& d, const RdsJob::LogEntry& log);
QDataStream& operator>>(QDataStream& d, RdsJob::LogEntry& log);
QDataStream& operator<<(QDataStream& d, const RdsJob::LogLevel& log);
QDataStream& operator>>(QDataStream& d, RdsJob::LogLevel& log);
QDataStream& operator<<(QDataStream& d, const RdsJob::Status& log);
QDataStream& operator>>(QDataStream& d, RdsJob::Status& log);

Q_DECLARE_METATYPE(RdsJob::LogLevel);
QTRPC_REGISTER_METATYPE(RdsJob::LogLevel);

Q_DECLARE_METATYPE(RdsJob::LogEntry);
QTRPC_REGISTER_METATYPE(RdsJob::LogEntry);
Q_DECLARE_METATYPE(QList<RdsJob::LogEntry>);
QTRPC_REGISTER_METATYPE(QList<RdsJob::LogEntry>);

Q_DECLARE_METATYPE(RdsJob::Status);
QTRPC_REGISTER_METATYPE(RdsJob::Status);

#define RDS_JOB_PROXY(type, x) \
	if (!getProxyPointer().isNull()) \
	{ \
		type* obj = qobject_cast<type*>(getProxyPointer().data()); \
		if (!obj) \
			return ReturnValue(1, "The proxy job that this job points to is of the wrong type!"); \
		return obj->x; \
	}

#endif
