/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdsjob.h"
#include "rdsjob_p.h"
#include "rdsjobmanager_p.h"

#define RDS_LOCK \
	QMutexLocker locker(qxt_d().mutex());

QTRPC_SERVICEPROXY_PIMPL_IMPLEMENT(RdsJob);

QDataStream& operator<<(QDataStream& d, const RdsJob::LogEntry& log)
{
	d << log.log;
	d << log.level;
	d << log.timestamp;
	return d;
}

QDataStream& operator>>(QDataStream& d, RdsJob::LogEntry& log)
{
	d >> log.log;
	d >> log.level;
	d >> log.timestamp;
	return d;
}

QDataStream& operator<<(QDataStream& d, const RdsJob::LogLevel& log)
{
	d << static_cast<int>(log);
	return d;
}

QDataStream& operator>>(QDataStream& d, RdsJob::LogLevel& log)
{
	int level;
	d >> level;
	log = static_cast<RdsJob::LogLevel>(level);
	return d;
}

QDataStream& operator<<(QDataStream& d, const RdsJob::Status& log)
{
	d << static_cast<int>(log);
	return d;
}

QDataStream& operator>>(QDataStream& d, RdsJob::Status& log)
{
	int status;
	d >> status;
	log = static_cast<RdsJob::Status>(status);
	return d;
}

ReturnValue RdsJob::startInternal()
{
	return ReturnValue(1, "Incorrectly called the base implementation of RdsJob::startInternal");
}

ReturnValue RdsJob::stopInternal()
{
	return ReturnValue(1, "Incorrectly called the base implementation of RdsJob::stopInternal");
}

ReturnValue RdsJob::pauseInternal()
{
	return ReturnValue(1, "Incorrectly called the base implementation of RdsJob::pauseInternal");
}

ReturnValue RdsJob::resumeInternal()
{
	return ReturnValue(1, "Incorrectly called the base implementation of RdsJob::resumeInternal");
}

RdsJob::RdsJob(QObject *parent)
		: QtRpc::ServiceProxy(parent)
{
	connect(&qxt_d().timer, SIGNAL(timeout()), this, SLOT(sendNotifications()), Qt::QueuedConnection);
	connect(&qxt_d().logTimer, SIGNAL(timeout()), this, SLOT(sendLogMessages()), Qt::QueuedConnection);
}

RdsJob::RdsJob(const RdsJob& other)
{
	RDS_LOCK;
	operator=(other);
}

RdsJob::~RdsJob()
{
	if (qxt_d().proxyJob.isNull())
	{
		stop();
		RdsJobManagerPrivate::data().unregisterJob(qxt_d().name);
	}
	sendNotifications();
	sendLogMessages();
}

RdsJob& RdsJob::operator=(const RdsJob & other)
{
	RDS_LOCK;
	disconnect(&qxt_d().timer, SIGNAL(timeout()), this, SLOT(sendNotifications()));
	disconnect(&qxt_d().logTimer, SIGNAL(timeout()), this, SLOT(sendLogMessages()));
	if (!qxt_d().proxyJob.isNull())
	{
		disconnect(qxt_d().proxyJob.data(), SIGNAL(progressChanged(int, const QString&, const QString&)), this, SIGNAL(progressChanged(int, const QString&, const QString&)));
		disconnect(qxt_d().proxyJob.data(), SIGNAL(statusChanged(RdsJob::Status)), this, SIGNAL(statusChanged(RdsJob::Status)));
		disconnect(qxt_d().proxyJob.data(), SIGNAL(newLogMsg(const QString&, RdsJob::LogLevel)), this, SIGNAL(newLogMsg(const QString&, RdsJob::LogLevel)));
		disconnect(qxt_d().proxyJob.data(), SIGNAL(finished(const ReturnValue&)), this, SIGNAL(finished(const ReturnValue&)));
	}

	qxt_d() = other.qxt_d();

	connect(&qxt_d().timer, SIGNAL(timeout()), this, SLOT(sendNotifications()), Qt::QueuedConnection);
	connect(&qxt_d().logTimer, SIGNAL(timeout()), this, SLOT(sendLogMessages()), Qt::QueuedConnection);
	if (qxt_d().proxyJob.isNull())
	{
		qxt_d().proxyJob = other.getSharedPointer();
	}

	connect(qxt_d().proxyJob.data(), SIGNAL(progressChanged(int, const QString&, const QString&)), this, SIGNAL(progressChanged(int, const QString&, const QString&)));
	connect(qxt_d().proxyJob.data(), SIGNAL(statusChanged(RdsJob::Status)), this, SIGNAL(statusChanged(RdsJob::Status)));
	connect(qxt_d().proxyJob.data(), SIGNAL(newLogMsg(const QString&, RdsJob::LogLevel)), this, SIGNAL(newLogMsg(const QString&, RdsJob::LogLevel)));
	connect(qxt_d().proxyJob.data(), SIGNAL(finished(const ReturnValue&)), this, SIGNAL(finished(const ReturnValue&)));
	return *this;
}

RdsJobPrivate& RdsJobPrivate::operator=(const RdsJobPrivate & other)
{
	QMutexLocker locker(&_mutex);
	QMutexLocker otherLocker(const_cast<QMutex*>(&other._mutex));
	name = other.name;
	description = other.description;
	status = other.status;
	percent = other.percent;
	progressText = other.progressText;
	progressDetails = other.progressDetails;
	log = other.log;
	canPause = other.canPause;
	proxyJob  = other.proxyJob;

	return *this;
}

ReturnValue RdsJob::name() const
{
	RDS_LOCK;
	RDS_JOB_PROXY(RdsJob, name());

	return qxt_d().name;
}

ReturnValue RdsJob::description() const
{
	RDS_LOCK;
	RDS_JOB_PROXY(RdsJob, description());

	return qxt_d().description;
}

ReturnValue RdsJob::status() const
{
	RDS_LOCK;
	RDS_JOB_PROXY(RdsJob, status());

	return qxt_d().status;
}

ReturnValue RdsJob::progress() const
{
	RDS_LOCK;
	RDS_JOB_PROXY(RdsJob, progress());

	return qxt_d().percent;
}

ReturnValue RdsJob::progressText() const
{
	RDS_LOCK;
	RDS_JOB_PROXY(RdsJob, progressText());

	return qxt_d().progressText;
}

ReturnValue RdsJob::progressDetails() const
{
	RDS_LOCK;
	RDS_JOB_PROXY(RdsJob, progressDetails());

	return qxt_d().progressDetails;
}

ReturnValue RdsJob::getLog() const
{
	RDS_LOCK;
	RDS_JOB_PROXY(RdsJob, getLog());

	return QVariant::fromValue(qxt_d().log);
}

ReturnValue RdsJob::canPause() const
{
	RDS_LOCK;
	RDS_JOB_PROXY(RdsJob, canPause());

	return qxt_d().canPause;
}

ReturnValue RdsJob::id() const
{
	RDS_LOCK;
	RDS_JOB_PROXY(RdsJob, id());

	if (qxt_d().id.isEmpty())
		return ReturnValue(1, "This job has not yet been registered");
	return qxt_d().id;
}

ReturnValue RdsJob::registerJob()
{
	RDS_LOCK;
	RDS_JOB_PROXY(RdsJob, registerJob());

	ReturnValue ret = RdsJobManagerPrivate::data().registerJob(metaObject()->className(), this);
	if (ret.isError())
		return ret;
	qxt_d().id = ret.toString();
	if (qxt_d().id.isEmpty())
		return ReturnValue(1, "Failed to get a valid id from the job manager");
	return ret;
}

ReturnValue RdsJob::unregisterJob()
{
	RDS_LOCK;
	RDS_JOB_PROXY(RdsJob, unregisterJob());

	stop();
	ReturnValue ret = RdsJobManagerPrivate::data().unregisterJob(qxt_d().id);
	if (ret.isError())
		return ret;
	qxt_d().id = QString();
	return ret;
}

ReturnValue RdsJob::start()
{
	RDS_LOCK;
	RDS_JOB_PROXY(RdsJob, start());

	if (qxt_d().status != Stopped)
		return ReturnValue(1, "This job has already been started");
	ReturnValue ret = startInternal();
	if (!ret.isError())
		setStatus(Running);
	return ret;
}

ReturnValue RdsJob::stop()
{
	RDS_LOCK;
	RDS_JOB_PROXY(RdsJob, stop());

	if (qxt_d().status != Running && qxt_d().status != Paused)
		return ReturnValue(1, "This job is not currently running");
	return stopInternal();
}

ReturnValue RdsJob::pause()
{
	RDS_LOCK;
	RDS_JOB_PROXY(RdsJob, pause());

	if (!qxt_d().canPause)
		return ReturnValue(1, "This job cannot be paused or resumed");
	if (qxt_d().status != Running)
		return ReturnValue(1, "This job is not currently running");
	return pauseInternal();
}

ReturnValue RdsJob::resume()
{
	RDS_LOCK;
	RDS_JOB_PROXY(RdsJob, resume());

	if (!qxt_d().canPause)
		return ReturnValue(1, "This job cannot be paused or resumed");
	if (qxt_d().status != Paused)
		return ReturnValue(1, "This job is not currently paused");
	return resumeInternal();
}

ReturnValue  RdsJob::setName(const QString& name)
{
	RDS_LOCK;
	RDS_JOB_PROXY(RdsJob, setName(name));

	qxt_d().name = name;
	return true;
}

ReturnValue RdsJob::setDescription(const QString& description)
{
	RDS_LOCK;
	RDS_JOB_PROXY(RdsJob, setDescription(description));

	qxt_d().description = description;
	return true;
}

ReturnValue RdsJob::setStatus(Status status)
{
	RDS_LOCK;
	RDS_JOB_PROXY(RdsJob, setStatus(status));

	qxt_d().status = status;
	emit statusChanged(status);
	return true;
}

ReturnValue RdsJob::setProgress(int percent)
{
	RDS_LOCK;
	if (qxt_d().percent == percent)
		return true;
	RDS_JOB_PROXY(RdsJob, setProgress(percent));

	qxt_d().percent = percent;
	if (!qxt_d().timer.isActive())
		qxt_d().timer.start(500);
	return true;
}

ReturnValue RdsJob::setProgressText(const QString& progressText)
{
	RDS_LOCK;
	RDS_JOB_PROXY(RdsJob, setProgressText(progressText));

	qxt_d().progressText = progressText;
	if (!qxt_d().timer.isActive())
		qxt_d().timer.start(500);
	return true;
}

ReturnValue RdsJob::setProgressDetails(const QString& progressDetails)
{
	RDS_LOCK;
	RDS_JOB_PROXY(RdsJob, setProgressDetails(progressDetails));

	qxt_d().progressDetails = progressDetails;
	if (!qxt_d().timer.isActive())
		qxt_d().timer.start(500);
	return true;
}

ReturnValue RdsJob::log(const QString& l)
{
	RDS_LOCK;
	return log(l, Info);
}

ReturnValue RdsJob::log(const QString& log, RdsJob::LogLevel level)
{
	RDS_LOCK;
	RDS_JOB_PROXY(RdsJob, log(log, level));

	LogEntry entry = {log, level, QDateTime::currentDateTime()};
	qxt_d().log << entry;
	qxt_d().logQueue << entry;

	if (!qxt_d().logTimer.isActive())
		qxt_d().logTimer.start(500);
	return true;
}

ReturnValue RdsJob::setCanPause(bool canPause)
{
	RDS_LOCK;
	RDS_JOB_PROXY(RdsJob, setCanPause(canPause));

	qxt_d().canPause = canPause;
	return true;
}

ReturnValue RdsJob::finish(const ReturnValue& ret)
{
	RDS_LOCK;
	RDS_JOB_PROXY(RdsJob, finish(ret));

	qxt_d().percent = 100;

	if (ret.isError())
		qxt_d().status = Failed;
	else
		qxt_d().status = Finished;

	qxt_d().progressText = "Finished";
	qxt_d().progressDetails = "Finished";

	qxt_d().timer.start(0);
	emit statusChanged(qxt_d().status);
	emit finished(ret);
	return true;
}

ReturnValue RdsJob::sendNotifications()
{
	RDS_LOCK;
	RDS_JOB_PROXY(RdsJob, sendNotifications());

	emit progressChanged(qxt_d().percent, qxt_d().progressText, qxt_d().progressDetails);
	qxt_d().timer.stop();
	return true;
}

ReturnValue RdsJob::sendLogMessages()
{
	RDS_LOCK;
	RDS_JOB_PROXY(RdsJob, sendLogMessages());

	QList<LogEntry> logs = qxt_d().logQueue;
	qxt_d().logQueue.clear();
	emit newLogMessages(logs);
	qxt_d().logTimer.stop();
	return true;
}

QSharedPointer<RdsJob> RdsJob::getSharedPointer() const
{
	RDS_LOCK;
	if (weakPointer().isNull())
	{
		// Take ownership, this means that this object cannot be deleted via any other method.
		QSharedPointer<RdsJob> ptr(const_cast<RdsJob*>(this), &QObject::deleteLater);
		const_cast<RdsJob*>(this)->weakPointer() = ptr;
		return ptr;
	}
	return weakPointer().toStrongRef().objectCast<RdsJob>();
}

QSharedPointer<RdsJob> RdsJob::getProxyPointer() const
{
	RDS_LOCK;
	return qxt_d().proxyJob;
}




