/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdsfilemanager.h"
#include "rdsfilemanager_p.h"
#include <RdsUtils>
#include <RdsMountPoint>
#include <RdsUtils>
#include <RdsEntity>
#include <QDir>
#include <RdsShare>
#include <RdsShareManager>
#include <RdsNtAcl>
#include <QTemporaryFile>
#include <QProcess>
#include <QDebug>
#include <RdsVolumeManager>
#include <RdsVolume>
#include <RdsSettings>
#include <RdsVolumeManager>
#include <RdsVolume>
#include "rdsmountpoint_p.h"
#include <QMutexLocker>

QMutex RdsFileManagerPrivate::volmutex;
RdsVolumeManager *RdsFileManagerPrivate::volmgr = NULL;
QStringList RdsFileManagerPrivate::volumes;

QTRPC_SERVICEPROXY_PIMPL_IMPLEMENT(RdsFileManager);

RdsFileManager::RdsFileManager(QObject *parent)
		: RdsEntityManager(parent)
{
	QXT_INIT_PRIVATE(RdsFileManager);
	qxt_d().showvolumes = true;
	qxt_d().showshares = true;
	qxt_d().showfilesystem = true;
	qxt_d().showfiles = false;

	RdsFileManagerPrivate::connectVolumeEvents(this);
}

RdsFileManager::RdsFileManager(const RdsFileManager &mgr)
		: RdsEntityManager(NULL)
{
	QXT_INIT_PRIVATE(RdsFileManager);
	qxt_d().showvolumes = mgr.qxt_d().showvolumes;
	qxt_d().showshares = mgr.qxt_d().showshares;
	qxt_d().showfilesystem = mgr.qxt_d().showfilesystem;
	qxt_d().showfiles = mgr.qxt_d().showfiles;

	RdsFileManagerPrivate::connectVolumeEvents(this);
}

RdsFileManager::~RdsFileManager()
{
}

RdsFileManager &RdsFileManager::operator=(const RdsFileManager & mgr)
{
	qxt_d().showvolumes = mgr.qxt_d().showvolumes;
	qxt_d().showshares = mgr.qxt_d().showshares;
	qxt_d().showfilesystem = mgr.qxt_d().showfilesystem;
	return *this;
}

ReturnValue RdsFileManager::auth(QtRpc::AuthToken token)
{
	createInternalObject();
	if (token.serverData().contains("authenticated") && (token.serverData().value("authenticated").toBool() == true))
		return(true);
	else
		return(ReturnValue(1, "Not Authenticated"));
}

bool RdsFileManager::isMounted(const QString &path)
{
	QString newpath = RdsUtils::normalizePath(path);
	QFileInfo info(newpath);

	if (info.isSymLink()) newpath = info.symLinkTarget();

	QTemporaryFile file;
	QString line;
	file.open();
	QString tmpfile = file.fileName();
	file.close();
	QProcess::execute("cp", QStringList() << "/proc/mounts" << tmpfile);
	if (!file.open())
	{
		return false;
	}
	while (!file.atEnd())
	{
		line = file.readLine();
		line = line.mid(line.indexOf(" ") + 1);
		line = line.left(line.indexOf(" "));
		line = RdsMountPointPrivate::decodeMountPath(line);
		if (line == newpath)
			return true;
	}

	return false;
}

ReturnValue RdsFileManager::listEntities(const QString &id, bool loadmore) const
{
	if ((id == "") || (id == "root"))
	{
		RdsEntity entity;
		entity.setId("root");
		entity.setType("root");
		entity.setVisible(false);
		entity.setName("");
		entity.setParent("");

		QString rootvolume = "";
		RdsVolumeManager mgr;
		ReturnValue ret = mgr.rootVolume();
		if (!ret.isError())
		{
			ret = mgr.volume(ret.toString());
			if (!ret.isError())
			{
				RdsVolume volume;
				volume = ret;

				ret = volume.name();
				if (!ret.isError()) rootvolume = ret.toString();
				else
				{
					qWarning() << "Failed to get root volume name" << ret;
				}
			}
			else
			{
				qWarning() << "Failed to get root volume" << ret;
			}
		}
		else
		{
			qWarning() << "Failed to get root volume ID" << ret;
		}

		if (qxt_d().showvolumes)
		{
			QDir dir("/volumes/");

			foreach(QString mnt, dir.entryList(QDir::Dirs))
			{
				if (mnt == "." || mnt == "..")
				{
					continue;
				}

				if (rdssettings()->value("volume-" + mnt + "/hidden").toBool()) continue;

				if (!isMounted("/volumes/" + mnt) && (mnt != rootvolume))
				{
					continue;
				}


				ReturnValue ret = listEntities("@VOLUME/" + mnt + "/", false);
				if (ret.isError()) continue;
				entity.children() << ret.value<RdsEntity>();
			}
		}

		if (qxt_d().showshares)
		{
			ReturnValue ret = RdsShareManager().listShares();
			if (!ret.isError())
			{
				foreach(QString share, ret.toStringList())
				{
					ret = listEntities("@SHARE/" + share + "/", loadmore);
					if (ret.isError()) continue;
					entity.children() << ret.value<RdsEntity>();
				}
			}
		}

		if (qxt_d().showfilesystem)
		{
			ReturnValue ret = listEntities("/", false);
			if (!ret.isError()) entity.children() << ret.value<RdsEntity>();
		}

		return(QVariant::fromValue<RdsEntity>(entity));
	}
	else if (id == "/")
	{
		RdsEntity entity;
		entity.setId("/");
		entity.setType("filesystem");
		entity.setVisible(true);
		entity.setName("File System");
		entity.setParent("root");
		entity.setGroup(true);

		if (loadmore)
		{
			QDir dir("/");

			foreach(QString child, dir.entryList(QStringList(), QDir::QDir::AllDirs))
			{
				if ((child == "..") || (child == ".")) continue;

				QString childid = QString("/%1").arg(child);

				ReturnValue ret = listEntities(childid, false);
				if (ret.isError()) continue;
				entity.children() << ret.value<RdsEntity>();
			}
			
			if(qxt_d().showfiles)
			{
				foreach(QString child, dir.entryList(QStringList() << "*", QDir::QDir::Files))
				{
					if ((child == "..") || (child == ".")) continue;
					
					QString childid = QString("/%1").arg(child);
					
					ReturnValue ret = listEntities(childid, false);
					if (ret.isError()) continue;
					entity.children() << ret.value<RdsEntity>();
				}
			}
		}
		else
		{
			entity.setHasMore(true);
		}

		return(QVariant::fromValue<RdsEntity>(entity));
	}
	else if (id.startsWith("/"))
	{
		QString newid = RdsUtils::normalizePath(id);
		bool isdir = QFileInfo(newid).isDir();
		
		QDir dir(id);
		QDir tmpdir = dir;
		QString parent = "/";
		if (tmpdir.cdUp()) parent = RdsUtils::normalizePath(tmpdir.path());

		RdsEntity entity;
		entity.setId(id);
		entity.setType(isdir ? "folder" : "file");
		entity.setVisible(true);
		entity.setName(dir.dirName());
		entity.setParent(parent);
		entity.setGroup(isdir);

		if (loadmore && isdir)
		{
			foreach(QString child, dir.entryList(QStringList(), QDir::QDir::AllDirs))
			{
				if ((child == "..") || (child == ".")) continue;

				QString childid = QString("%1/%2").arg(newid).arg(child);


				ReturnValue ret = listEntities(childid, false);
				if (ret.isError()) continue;
				entity.children() << ret.value<RdsEntity>();
			}
			
			if(qxt_d().showfiles)
			{
				foreach(QString child, dir.entryList(QStringList() << "*", QDir::QDir::Files))
				{
					if ((child == "..") || (child == ".")) continue;
					
					QString childid = QString("%1/%2").arg(newid).arg(child);
					
					
					ReturnValue ret = listEntities(childid, false);
					if (ret.isError()) continue;
					entity.children() << ret.value<RdsEntity>();
				}
			}
		}
		else
		{
			entity.setHasMore(isdir);
		}

		return(QVariant::fromValue<RdsEntity>(entity));
	}
	else if (id.startsWith("@VOLUME/") && (id.count('/') == 2) && id.endsWith("/"))
	{
		ReturnValue ret = RdsUtils::getSystemPath(id);
		if (ret.isError()) return(ret);
		QString path = ret.toString();

		ret = RdsUtils::getVolumeName(id);
		if (ret.isError()) return(ret);
		QString name = ret.toString();

		RdsEntity entity;
		entity.setId(id);
		entity.setType("volume");
		entity.setVisible(true);
		entity.setName(name);
		entity.setParent("root");
		entity.setGroup(true);

		if (loadmore)
		{
			QDir dir(path);

			foreach(QString child, dir.entryList(QStringList(), QDir::QDir::AllDirs))
			{
				if ((child == "..") || (child == ".")) continue;

				QString childid = QString("@VOLUME/%1/%2").arg(name).arg(child);


				ReturnValue ret = listEntities(childid, false);
				if (ret.isError()) continue;
				entity.children() << ret.value<RdsEntity>();
			}
			
			if(qxt_d().showfiles)
			{
				foreach(QString child, dir.entryList(QStringList() << "*", QDir::QDir::Files))
				{
					if ((child == "..") || (child == ".")) continue;
					
					QString childid = QString("@VOLUME/%1/%2").arg(name).arg(child);
					
					
					ReturnValue ret = listEntities(childid, false);
					if (ret.isError()) continue;
					entity.children() << ret.value<RdsEntity>();
				}
			}
		}
		else
		{
			entity.setHasMore(true);
		}

		return(QVariant::fromValue<RdsEntity>(entity));
	}
	else if (id.startsWith("@VOLUME/"))
	{
		QString newid = RdsUtils::normalizePath(id);

		ReturnValue ret = RdsUtils::getSystemPath(id);
		if (ret.isError()) return(ret);
		QString path = ret.toString();

		bool isdir = QFileInfo(path).isDir();
		
		ret = RdsUtils::getVolumeName(id);
		if (ret.isError()) return(ret);
		QString name = ret.toString();

		QDir dir(path);
		QDir tmpdir = dir;
		QString parent = "/volumes/" + name;
		if (tmpdir.cdUp()) parent = RdsUtils::normalizePath(tmpdir.path());

		ret = RdsUtils::getVolumePath(parent);
		if (ret.isError()) return(ret);
		parent = ret.toString();
		if (parent.count("/") == 1) parent += "/";

		RdsEntity entity;
		entity.setId(id);
		entity.setType(isdir ? "folder" : "file");
		entity.setVisible(true);
		entity.setName(dir.dirName());
		entity.setParent(parent);
		entity.setGroup(isdir);

		if (loadmore && isdir)
		{
			foreach(QString child, dir.entryList(QStringList(), QDir::QDir::AllDirs))
			{
				if ((child == "..") || (child == ".")) continue;

				QString childid = QString("%1/%2").arg(newid).arg(child);

				ReturnValue ret = listEntities(childid, false);
				if (ret.isError()) continue;
				entity.children() << ret.value<RdsEntity>();
			}
			
			if(qxt_d().showfiles)
			{
				foreach(QString child, dir.entryList(QStringList() << "*", QDir::QDir::Files))
				{
					if ((child == "..") || (child == ".")) continue;
					
					QString childid = QString("%1/%2").arg(newid).arg(child);
					
					ReturnValue ret = listEntities(childid, false);
					if (ret.isError()) continue;
					entity.children() << ret.value<RdsEntity>();
				}
			}
		}
		else
		{
			entity.setHasMore(isdir);
		}

		return(QVariant::fromValue<RdsEntity>(entity));
	}
	else if (id.startsWith("@SHARE/") && (id.count('/') == 2) && id.endsWith("/"))
	{
		ReturnValue ret = RdsUtils::getShareName(id);
		if (ret.isError()) return(ret);
		QString sharename = ret.toString();

		ret = RdsShareManager().share(sharename);
		if (ret.isError()) return(ret);
		RdsShare share;
		share = ret;

		ret = share.path();
		if (ret.isError()) return(ret);
		QString path = ret.toString();

		RdsEntity entity;
		entity.setId(id);
		entity.setType("share");
		entity.setVisible(true);
		entity.setName(sharename);
		entity.setParent("root");
		entity.setGroup(true);

		if (loadmore)
		{
			QDir dir(path);

			foreach(QString child, dir.entryList(QStringList(), QDir::QDir::AllDirs))
			{
				if ((child == "..") || (child == ".")) continue;

				QString childid = QString("@SHARE/%1/%2").arg(sharename).arg(child);


				ret = listEntities(childid, false);
				if (ret.isError()) continue;
				entity.children() << ret.value<RdsEntity>();
			}
			
			if(qxt_d().showfiles)
			{
				foreach(QString child, dir.entryList(QStringList() << "*", QDir::QDir::Files))
				{
					if ((child == "..") || (child == ".")) continue;
					
					QString childid = QString("@SHARE/%1/%2").arg(sharename).arg(child);
					
					
					ret = listEntities(childid, false);
					if (ret.isError()) continue;
					entity.children() << ret.value<RdsEntity>();
				}
			}
		}
		else
		{
			entity.setHasMore(true);
		}

		return(QVariant::fromValue<RdsEntity>(entity));
	}
	else if (id.startsWith("@SHARE/"))
	{
		ReturnValue ret = RdsUtils::getShareName(id);
		if (ret.isError()) return(ret);
		QString sharename = ret.toString();

		ret = RdsShareManager().share(sharename);
		if (ret.isError()) return(ret);

		RdsShare share;
		share = ret;

		ret = share.path();
		if (ret.isError()) return(ret);
		QString sharepath = RdsUtils::normalizePath(ret.toString());

		QString path = RdsUtils::normalizePath(sharepath + "/" + RdsUtils::getRelativePath(id).toString());
		bool isdir = QFileInfo(path).isDir();

		QDir dir(path);
		QDir parent = dir;
		if (!parent.cdUp()) return(ReturnValue(1, "Cannot find parent"));

		QString parentid;
		if (parent.path() == sharepath)
		{
			parentid = "@SHARE/" + sharename + "/";
		}
		else
		{
			parentid = RdsUtils::normalizePath(id);
			int index = parentid.lastIndexOf("/");
			parentid = parentid.left(index);
		}


		RdsEntity entity;
		entity.setId(id);
		entity.setType(isdir ? "folder" : "file");
		entity.setVisible(true);
		entity.setName(dir.dirName());
		entity.setParent(parentid);
		entity.setGroup(isdir);


		if (loadmore && isdir)
		{
			foreach(QString child, dir.entryList(QStringList(), QDir::QDir::AllDirs))
			{
				if ((child == "..") || (child == ".")) continue;

				QString tmp = RdsUtils::normalizePath(id);
				QString childid = QString("%1/%2").arg(tmp).arg(child);

				ret = listEntities(childid, false);
				if (ret.isError()) continue;
				entity.children() << ret.value<RdsEntity>();
			}
			
			if(qxt_d().showfiles)
			{
				foreach(QString child, dir.entryList(QStringList() << "*", QDir::QDir::Files))
				{
					if ((child == "..") || (child == ".")) continue;
					
					QString tmp = RdsUtils::normalizePath(id);
					QString childid = QString("%1/%2").arg(tmp).arg(child);
					
					ret = listEntities(childid, false);
					if (ret.isError()) continue;
					entity.children() << ret.value<RdsEntity>();
				}
			}
		}
		else
		{
			entity.setHasMore(isdir);
		}

		return(QVariant::fromValue<RdsEntity>(entity));
	}
	else
	{
		return(ReturnValue(1, "Malformed Path: " + id));
	}
}

ReturnValue RdsFileManager::showVolumes()
{
	return qxt_d().showvolumes;
}

ReturnValue RdsFileManager::setShowVolumes(bool show)
{
	qxt_d().showvolumes = show;
	return(true);
}

ReturnValue RdsFileManager::showShares()
{
	return qxt_d().showshares;
}

ReturnValue RdsFileManager::setShowShares(bool show)
{
	qxt_d().showshares = show;
	return(true);
}

ReturnValue RdsFileManager::showFileSystem()
{
	return qxt_d().showfilesystem;
}

ReturnValue RdsFileManager::setShowFileSystem(bool show)
{
	qxt_d().showfilesystem = show;
	return(true);
}

ReturnValue RdsFileManager::showFiles()
{
	return qxt_d().showfiles;
}

ReturnValue RdsFileManager::setShowFiles(bool show)
{
	qxt_d().showfiles = show;
	return(true);
}


ReturnValue RdsFileManager::followSymLink(const QString &path) const
{
	return RdsUtils::followSymLink(path);
}

ReturnValue RdsFileManager::createDirectory(const QString &path, bool recursive)
{
	return(RdsUtils::createDirectory(path, recursive));
}

ReturnValue RdsFileManager::remove(const QString &path, bool recursive)
{
	return(RdsUtils::remove(path, recursive));
}

ReturnValue RdsFileManager::move(const QString &path, const QString &newpath)
{
	return(RdsUtils::move(path, newpath));
}

ReturnValue RdsFileManager::rename(const QString &path, const QString &newname)
{
	return(RdsUtils::rename(path, newname));
}

ReturnValue RdsFileManager::fileContents(const QString &path) const
{
	return(RdsUtils::fileContents(path));
}

ReturnValue RdsFileManager::setFileContents(const QString &path, const QByteArray &data)
{
	return(RdsUtils::setFileContents(path, data));
}

ReturnValue RdsFileManager::unixPermissions(const QString &path) const
{
	return(RdsUtils::unixPermissions(path));
}

ReturnValue RdsFileManager::setUnixPermissions(const QString &path, int mode)
{
	return(RdsUtils::setUnixPermissions(path, mode));
}

ReturnValue RdsFileManager::owner(const QString &path) const
{
	return(RdsUtils::owner(path));
}

ReturnValue RdsFileManager::setOwner(const QString &path, const QString &owner)
{
	return(RdsUtils::setOwner(path, owner));
}

ReturnValue RdsFileManager::setOwner(const QString &path, int owner)
{
	return(RdsUtils::setOwner(path, owner));
}

ReturnValue RdsFileManager::group(const QString &path) const
{
	return(RdsUtils::group(path));
}

ReturnValue RdsFileManager::setGroup(const QString &path, const QString &group)
{
	return(RdsUtils::setGroup(path, group));
}

ReturnValue RdsFileManager::setGroup(const QString &path, int group)
{
	return(RdsUtils::setGroup(path, group));
}

ReturnValue RdsFileManager::ntPermissions(const QString &path) const
{
	return(RdsUtils::ntPermissions(path));
}

ReturnValue RdsFileManager::setNtPermissions(const QString &path, const RdsNtAcl &permissions)
{
	return(RdsUtils::setNtPermissions(path, permissions));
}

ReturnValue RdsFileManager::exists(const QString &path)
{
	ReturnValue ret = RdsUtils::getSystemPath(path);
	if (ret.isError()) return(ret);

	QFileInfo info(ret.toString());

	return(info.exists());
}

ReturnValue RdsFileManager::isFile(const QString &path)
{
	ReturnValue ret = RdsUtils::getSystemPath(path);
	if (ret.isError()) return(ret);

	QFileInfo info(ret.toString());

	return(info.isFile());
}

ReturnValue RdsFileManager::isDirectory(const QString &path)
{
	ReturnValue ret = RdsUtils::getSystemPath(path);
	if (ret.isError()) return(ret);

	QFileInfo info(ret.toString());

	return(info.isDir());
}

ReturnValue RdsFileManager::updateInheritedPermissions(const QString &path)
{
	ReturnValue ret = RdsUtils::getSystemPath(path);
	if (ret.isError()) return(ret);

	return(RdsNtAcl::updateInheritedPermissions(ret.toString()));
}

ReturnValue RdsFileManager::propagateInheritedPermissions(const QString &path)
{
	return(propagateInheritedPermissions(path, false));
}

ReturnValue RdsFileManager::propagateInheritedPermissions(const QString &path, bool forceinherit)
{
	ReturnValue ret = RdsUtils::getSystemPath(path);
	if (ret.isError()) return(ret);

	return(RdsNtAcl::propagateInheritedPermissions(ret.toString(), forceinherit));
}

void RdsFileManager::volumeAdded(QString id)
{
	volumeUpdated(id);
}

void RdsFileManager::volumeRemoved(QString id)
{
	volumeUpdated(id);
}

void RdsFileManager::volumeUpdated(QString id)
{
	Q_UNUSED(id);
	QStringList newlist = getMountedVolumes();

	foreach(QString name, newlist)
	{
		//Device was added
		if (!qxt_d().volumes.contains(name))
		{
			if (qxt_d().showvolumes)
			{
				addEntity("@VOLUME/" + name + "/");
			}

			if (qxt_d().showfilesystem)
			{
				removeEntity("/volumes/" + name + "/");
				addEntity("/volumes/" + name + "/");
			}
		}
	}

	foreach(QString name, qxt_d().volumes)
	{
		//Device was removed
		if (!newlist.contains(name))
		{
			if (qxt_d().showvolumes)
			{
				removeEntity("@VOLUME/" + name + "/");
			}

			if (qxt_d().showfilesystem)
			{
				removeEntity("/volumes/" + name + "/");
				addEntity("/volumes/" + name + "/");
			}
		}
	}

	qxt_d().volumes = newlist;
}

QStringList RdsFileManager::getMountedVolumes()
{
	///@todo Why does this function get called so much after a rename?
	QStringList list;

	QString rootvolume = "";
	RdsVolumeManager mgr;
	ReturnValue ret = mgr.rootVolume();
	if (!ret.isError())
	{
		ret = mgr.volume(ret.toString());
		if (!ret.isError())
		{
			RdsVolume volume;
			volume = ret;

			ret = volume.name();
			if (!ret.isError()) rootvolume = ret.toString();
			else
			{
				qWarning() << "Failed to get root volume name" << ret;
			}
		}
		else
		{
			qWarning() << "Failed to get root volume" << ret;
		}
	}
	else
	{
		qWarning() << "Failed to get root volume ID" << ret;
	}

	QDir dir("/volumes/");

	foreach(QString mnt, dir.entryList(QDir::Dirs))
	{
		if (mnt == "." || mnt == "..")
		{
			continue;
		}

		if (rdssettings()->value("volume-" + mnt + "/hidden").toBool()) continue;

		if (!isMounted("/volumes/" + mnt) && (mnt != rootvolume))
		{
			continue;
		}

		list << mnt;
	}

	return(list);
}

void RdsFileManagerPrivate::connectVolumeEvents(RdsFileManager *mgr)
{
	QMutexLocker lock(&volmutex);

	if (volmgr == NULL)
	{
		volmgr = new RdsVolumeManager();
		volmgr->init(); //enable events
		volumes = RdsFileManager::getMountedVolumes();

		QObject::connect(mgr, SIGNAL(destroyed()), volmgr, SLOT(deleteLater()), Qt::QueuedConnection);
		QObject::connect(volmgr, SIGNAL(entityAdded(QString)), mgr, SLOT(volumeAdded(QString)), Qt::QueuedConnection);
		QObject::connect(volmgr, SIGNAL(entityRemoved(QString)), mgr, SLOT(volumeRemoved(QString)), Qt::QueuedConnection);
		QObject::connect(volmgr, SIGNAL(entityUpdated(QString)), mgr, SLOT(volumeUpdated(QString)), Qt::QueuedConnection);
	}
}
