/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#ifndef RDSDNSZONE_P_H
#define RDSDNSZONE_P_H

#include <QxtPimpl>
#include <QSharedData>
#include <QStringList>
#include <QMutex>
#include <QDebug>
#include "rdsdnszone.h"
#include "rdsdnsrecord.h"

class RdsDnsZoneData;

/**
	@author Chris Vickery <chris@resara.com>
 */
class RdsDnsZonePrivate : public QxtPrivate<RdsDnsZone>
{
public:
	RdsDnsZonePrivate()
			: data(0)
	{ }
	static QMutex &dataMutex();
	static QHash<QString, RdsDnsZoneData* > &dataHash();
	QString domainName;
	RdsDnsZoneData* data;
};

class RdsDnsZoneData : public QObject, public QSharedData
{
	Q_OBJECT
	friend class RdsDnsZone;
public:
	RdsDnsZoneData()
			: valid(false),
			errorString("Zone data was not initialized"),
			refCount(1),
			mutex(QMutex::Recursive),
			manager(0)
	{ }
	~RdsDnsZoneData()
	{
		if (refCount > 0)
			qWarning() << this << "Deleting RdsDnsZoneData with ref greater than 0:" << refCount;
	}
#define COPY_FROM_OTHER(x) \
	if (x != other.x) \
	{ \
		emit valueChanged(#x); \
		x = other.x; \
	}
	RdsDnsZoneData& operator=(const RdsDnsZoneData& other)
	{
		COPY_FROM_OTHER(domainName);
		COPY_FROM_OTHER(primaryNameServer);
		COPY_FROM_OTHER(adminNameServer);
		COPY_FROM_OTHER(soaValues);
		COPY_FROM_OTHER(valid);
		COPY_FROM_OTHER(errorString);
		COPY_FROM_OTHER(soaComments);
		COPY_FROM_OTHER(soaTtl);
		COPY_FROM_OTHER(soaOrigin);
		COPY_FROM_OTHER(extraOrigins);
		RdsDnsRecordList list = records;
		RdsDnsRecordList otherList = other.records;
		records = other.records;
		foreach(RdsDnsRecord record, list)
		{
			if (otherList.contains(record))
			{
				otherList.removeAll(record);
				continue;
			}
			emit recordRemoved(record.key());
		}
		foreach(RdsDnsRecord record, otherList)
		{
			emit recordAdded(record.key());
		}
		return *this;
	}
#undef COPY_FROM_OTHER
	QMutex* getMutex() const
	{
		return const_cast<QMutex*>(&mutex);
	}

public slots:
	// This function tell the class that we're refing it... Always lock RdsDnsZonePrivate::dataMutex first
	void ref()
	{
		refCount++;
	}
	// This function tell the class that we're no longer refing it... Always lock RdsDnsZonePrivate::dataMutex first
	// Never do anything with the data object after calling this function, there is no way to know if it still exists...
	void deref()
	{
		if (--refCount == 0)
		{
			RdsDnsZonePrivate::dataHash().remove(domainName);
			deleteLater();
		}
	}

signals:
	void valueChanged(QString);
	void recordChanged(QString);
	void recordAdded(QString);
	void recordRemoved(QString);
	void removed();

public: //Data
	bool valid;
	QStringList soaComments;
	QString domainName;
	RdsDnsRecordList records;
	QString primaryNameServer;
	QString adminNameServer;
	QStringList soaValues;
	QString soaOrigin;
	QString soaTtl;
	QString errorString;
	int refCount;
	QMutex mutex;
	RdsDnsManager* manager;
	QStringList extraOrigins;
};

#endif
