/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "config.h"
#include <QCoreApplication>
#include <QString>
#include <QSettings>
#include <QDebug>
#include <QFileInfo>
#include <QDateTime>
#include <AuthToken>

#include <RdsClient>
#include <RdsDaemonManager>

#include "qxtcommandoptions.h"
#include "rdsprovisionator.h"
#include <iostream>

#define CHECK_ERR if(ret.isError()) \
	{ \
		qCritical() << ret; \
		return(1); \
	}

bool getValue(const QString &key, QVariantMap &settings, const QString &m = QString())
{
	if (key.size() < 1)
	{
		qCritical() << "Internal error (null key passed to getValue)";
		return false;
	}
	QString msg = m;
	if (msg.isEmpty())
	{
		msg = key;
		msg[0] = msg[0].toUpper();
	}
	QString input;
	QTextStream stream(stdin);
	QVariant::Type vtype = settings[key].type();
	while (true)
	{
		switch (vtype)
		{
			case QVariant::StringList:
				std::cout << qPrintable(msg + " [" + settings[key].toStringList().join(", ") + "]: ") << std::flush;
				break;
			case QVariant::Bool:
				std::cout << qPrintable(msg + " (" + QString(settings[key].toBool() ? "Y/n" : "y/N") + "): ") << std::flush;
				break;
			default:
				std::cout << qPrintable(msg + " [" + settings[key].toString() + "]: ") << std::flush;
				break;
		}
		input = stream.readLine();
		if (input.isNull())
		{
			qCritical() << "EOF received";
			return false;
		}

		if (!input.isEmpty())
		{
			if (vtype == QVariant::Bool)
			{
				input = input.toLower();
				if (input == "y" ||
				        input == "yes" ||
				        input == "true")
				{
					settings[key] = true;
					break;
				}
				else if (input == "no" ||
				         input == "n" ||
				         input == "false")
				{
					settings[key] = false;
					break;
				}
				continue;
			}
			settings[key] = input;
			break;
		}
		else if (!settings[key].isNull())
			break;
		else
			continue;
	}
	return true;
}

int main(int argc, char *argv[])
{
	QCoreApplication app(argc, argv);
	QxtCommandOptions options;
	bool socket = true;
	
	options.add("conf", "Use config file", QxtCommandOptions::Required);
	options.add("force", "Force a server to provision");
	options.alias("force", "f");
	options.add("help", "Show this help message");
	options.alias("help", "h");

	options.addSection("Connection options");
	options.add("host", "The hostname or IP to connect to", QxtCommandOptions::Required);
	options.add("port", "Port to connect to the RDS server", QxtCommandOptions::Required);
	options.add("service", "A custom service name to connect to", QxtCommandOptions::Required);
	options.add("user", "Username to authenticate with", QxtCommandOptions::Required);
	options.add("pass", "Password to authenticate with", QxtCommandOptions::Required);

	options.parse(argc, argv);
	options.parse(QCoreApplication::arguments());

	if (options.count("help") || options.showUnrecognizedWarning())
	{
		options.showUsage();
		return -1;
	}

	QString service = getenv("RDS_SERVICE");
	if (options.count("service"))
		service = options.value("service").toString();
	if (service == "")
		service = "Rds";

	QString host = getenv("RDS_SERVER");
	if (options.count("host"))
		host = options.value("host").toString();
	if (host == "")
		host = "localhost";
	else
		socket = false;
	
	int port = 0;
	QString tmp = getenv("RDS_PORT");
	if (tmp.toInt() != 0) port = tmp.toInt();
	if (options.count("port"))
		port = options.value("port").toInt();
	if(port == 0)
		port = 10123;
	else
		socket = false;

	QSettings s(RDS_CONF_PATH + "/rds.conf", QSettings::IniFormat);
	QString user = s.value("adminuser").toString();
	if (options.count("user"))
		user = options.value("user").toString();
	QString pass = s.value("adminpw").toString();
	if (options.count("pass"))
		pass = options.value("pass").toString();

	RdsClient client;
	QString url;
	if(socket)
		url = QString("socket:///var/run/rdsd.socket:%3").arg(service);
	else
		url = QString("tcp://%1:%2/%3").arg(host).arg(port).arg(service);
	QtRpc::AuthToken token(user, pass);
	ReturnValue ret = client.connect(url, token);
	if (ret.isError())
	{
		qCritical() << "Failed to connect to server:" << ret.errString();
		return(1);
	}

	RdsClient::setGlobalInstance(&client);
	QVariantMap usersettings;


	bool force = (options.count("force"));

	ret = rdsClient()->isProvisioned();
	CHECK_ERR;

	if (ret.toBool() & !force)
	{
		qCritical() << "This server is already provisioned. Use -f to override";
		return(1);
	}

	ret = rdsClient()->provisionator();
	CHECK_ERR;

	RdsProvisionator provisionator;
	provisionator = ret;

	ret = provisionator.getDefaultSettings();
	CHECK_ERR;

	QVariantMap settings = ret.toMap();

	if (options.count("conf"))
	{
		QString filename = options.value("conf").toString();
		if (!QFileInfo(filename).exists())
		{
			qCritical() << filename << "does not exist.";
			return(1);
		}

		QSettings cfg2(filename, QSettings::IniFormat);
		cfg2.beginGroup("Provision");
		foreach(QString key, cfg2.childKeys())
		{
			usersettings[key] = cfg2.value(key);
		}
	}
	else
	{
		QString input;
		qDebug() << "RDS Provision tool interactive mode";
		qDebug() << "Please put in each value and hit enter, or enter a blank string to use the default option";
		qDebug();

		getValue("RDS_SAMBA_PATH", settings, "Samba base path");
		settings["RDS_SAMBA_SYSVOL"] = settings["RDS_SAMBA_PATH"].toString() + "/var/locks/sysvol/";
		settings["RDS_SAMBA_CONFIGFILE"] = settings["RDS_SAMBA_PATH"].toString() + "/etc/smb.conf";
		settings["SAMBA_NET_PATH"] = settings["RDS_SAMBA_PATH"].toString() + "/bin/net";

		getValue("RDS_SAMBA_SYSVOL", settings, "Samba sysvol path");
		getValue("RDS_SAMBA_CONFIGFILE", settings, "Samba configuration file path");
		getValue("SAMBA_NET_PATH", settings, "Samba net path");
		getValue("RDS_SAMBA_PRIVATE", settings, "Samba private path");
		getValue("RDS_SAMBA_PROVISION", settings, "Samba provision path");

		settings["provisiondhcp"] = false;
		settings["existing_samba"] = false;
		getValue("existing_samba", settings, "Integrate into existing Samba4 domain?");
		if (settings["existing_samba"].toBool())
		{
			QVariantMap map, tmpmap;
			tmpmap["RDS_SAMBA_PATH"] = settings["RDS_SAMBA_PATH"];
			tmpmap["RDS_SAMBA_SYSVOL"] = settings["RDS_SAMBA_SYSVOL"];
			tmpmap["RDS_SAMBA_CONFIGFILE"] = settings["RDS_SAMBA_CONFIGFILE"];
			tmpmap["SAMBA_NET_PATH"] = settings["SAMBA_NET_PATH"];
			tmpmap["RDS_SAMBA_PRIVATE"] = settings["RDS_SAMBA_PRIVATE"];
			tmpmap["RDS_SAMBA_PROVISION"] = settings["RDS_SAMBA_PROVISION"];


			map["Paths"] = tmpmap;
			tmpmap = QVariantMap();
			getValue("realm", settings);
			getValue("domain", settings);
			ReturnValue ret = provisionator.writeRawConf(map);
			if (ret.isError())
			{
				qCritical() << ret;
				return 1;
			}
			ret = provisionator.provisionExistingDomain(settings);
			if (ret.isError())
			{
				qCritical() << ret;
				return 1;
			}
			return 0;
		}

		getValue("hostname", settings);
		getValue("realm", settings);
		getValue("domain", settings);
		getValue("ip", settings);
		getValue("netmask", settings);
		getValue("gateway", settings);
		getValue("provisiondhcp", settings);
		if (settings["provisiondhcp"].toBool())
		{
			getValue("dhcpnetmask", settings);
			getValue("dhcpstart", settings);
			getValue("dhcpend", settings);
		}
		getValue("adminpw", settings);
		getValue("dns", settings);

		settings["provisiondhcp"] = settings["provisiondhcp"].toBool();
		settings["dns"] = settings["dns"].toString().split(",", QString::SkipEmptyParts);
	}

	QString oldip = settings.value("ip").toString();

	foreach(QString key, usersettings.keys())
	{
		settings[key] = usersettings[key];
	}

	qDebug() << "Settings:";
	foreach(QString key, settings.keys())
	{
		if (settings[key].canConvert<QStringList>())
			qDebug() << qPrintable(key + " = " + settings[key].toStringList().join(", "));
		else
			qDebug() << qPrintable(key + " = " + settings[key].toString());
	}

	if (!settings.contains("ip"))
		return(1);

	QString ip = settings.value("ip").toString();
	bool netconfigured = settings.value("netconfigured").toBool();
	settings["realm"] = settings.value("realm").toString().toUpper();

	//If the network is not configured, or the current IP is different than the desired IP, reconfigure the network
	if (!netconfigured || ((oldip != "") && (oldip != ip)))
	{
		qDebug() << " - Reconfiguring the network.";
		ret = provisionator.setNetworkSettings(&provisionator , "", settings);
		sleep(5);
		rdsClient()->disconnect();
		provisionator.asyncConnect(user, pass, ip, host, port, service, settings);
	}
	else
	{
		ret = provisionator.provision(settings);
		CHECK_ERR;
	}

	return(app.exec());
}
