/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
***************************************************************************/

#include "rdssambaldapsession.h"
#include <QFileInfo>
#include <QTime>
#include <QProcess>
#include <QTemporaryFile>
#include <QHostInfo>
#include <RdsSettings>
#include <RdsUtils>
#include <RdsUser>
#include "config.h"
#include "logmanager.h"

RdsSambaLdapSession::RdsSambaLdapSession(QObject* parent)
		: RdsLdapSession(parent)
{
	_failednum = 0;
}

RdsSambaLdapSession::RdsSambaLdapSession(QString uri, QObject *parent)
		: RdsLdapSession(uri, parent)
{
	_failednum = 0;
}

RdsSambaLdapSession::~RdsSambaLdapSession()
{

}

ReturnValue RdsSambaLdapSession::kerberosBind()
{
	setenv("KRB5CCNAME", "/tmp/rds.creds", 1);

	if ((!QFileInfo(RDS_SAMBA_PRIVATE + "/rds.keytab").exists()) || (_failednum > 5))
	{
		_failednum = 0;
		ReturnValue ret = configureLdap();
		if (ret.isError()) return(ret);
	}

	ReturnValue ret = RdsUtils::runCommand("kinit", QStringList() << "-t" << RDS_SAMBA_PRIVATE + "/rds.keytab" << "-k" << rdssettings()->value("adminuser").toString());
	if (ret.isError())
	{
		_failednum++;
		return(ret);
	}

	ret = RdsLdapSession::kerberosBind();

	if (!ret.isError())
		_failednum = 0;
	else
		_failednum++;

	return(ret);
}

ReturnValue RdsSambaLdapSession::configureLdap()
{
	qInfo() << "Creating Admin User...";

	QString olduser = rdssettings()->value("adminuser").toString();
	
	if(olduser != "")
	{
		//qDebug() << "CMD:" << SAMBA_LDBDEL_PATH << "-H" << RDS_SAMBA_PRIVATE + "/sam.ldb" <<  
			QString("CN=%1,CN=Users,%2").arg(olduser).arg(RdsUtils::baseDn());
			if (QProcess::execute(SAMBA_LDBDEL_PATH, QStringList() << "-H" << RDS_SAMBA_PRIVATE + "/sam.ldb" <<  
				QString("CN=%1,CN=Users,%2").arg(olduser).arg(RdsUtils::baseDn())) != 0)
		{
			qWarning() << "Warning: Failed to delete old admin user.";
		}
	}
	
	srand(QTime::currentTime().msec());
	int rndint = rand();
	QString adminuser = QString("rdsadmin%1").arg(rndint, 0, 36);
	QString adminpw;
	for (int i = 0; i < 26; i++)
	{
		adminpw += (char)((65 + rand() % 26) ^(rand() % 2 ? 32 : 0));
	}

	//Add the rds admin user
	if (QProcess::execute(SAMBA_NET_PATH, QStringList() << "user" << "add" << adminuser << adminpw) != 0)
		return ReturnValue(2, "Failed to create RDS admin user");

	rdssettings()->setValue("adminuser", adminuser);

	//Add the rds admin user to the domain admins group
	if (QProcess::execute(SAMBA_NET_PATH, QStringList() << "group"  << "addmembers" << "Domain Admins" << adminuser) != 0)
		return ReturnValue(2, "Failed to add RDS admin user to Domain Admins");


	QFile file(RDS_DATA_PATH + "/secrets.ldiff");
	if (!file.open(QFile::ReadOnly))
	{
		return(ReturnValue(2, "Failed to open secrets template file: " + RDS_DATA_PATH + "/secrets.ldiff"));
	}

	QString tmp = file.readAll();
	file.close();

	tmp.replace("%USERNAME%", adminuser);
	tmp.replace("%REALM%", RdsUtils::realm());
	tmp.replace("%REALM_LOWER%", RdsUtils::realm().toLower());
	tmp.replace("%FQDN%", QHostInfo::localHostName() + "." + RdsUtils::realm().toLower());
	tmp.replace("%SECRET%", adminpw.toAscii());

	QTemporaryFile tmpsecret;
	if (!tmpsecret.open())
	{
		return(ReturnValue(2, "Failed to open secrets file: " + tmpsecret.fileName()));
	}
	tmpsecret.write(tmp.toAscii());
	tmpsecret.flush();

	QProcess p;
	p.setProcessChannelMode(QProcess::MergedChannels);
	p.start(SAMBA_LDBADD_PATH, QStringList() << "-H" << RDS_SAMBA_PRIVATE + "/secrets.ldb" << tmpsecret.fileName());
	p.waitForFinished(30000);

	tmpsecret.close();

	if (p.exitCode() == 0)
	{
		return(true);
	}
	else if (p.exitCode() == 68)
	{
		QFile file(RDS_DATA_PATH + "/secrets-edit.ldiff");
		if (!file.open(QFile::ReadOnly))
		{
			return(ReturnValue(2, "Failed to open secrets template file: " + RDS_DATA_PATH + "/secrets-edit.ldiff"));
		}

		QString tmp = file.readAll();
		file.close();

		tmp.replace("%USERNAME%", adminuser);
		tmp.replace("%SECRET%", adminpw.toAscii());

		QTemporaryFile tmpsecret;
		if (!tmpsecret.open())
		{
			return(ReturnValue(2, "Failed to open secrets file: " + tmpsecret.fileName()));
		}
		tmpsecret.write(tmp.toAscii());
		tmpsecret.flush();

		QProcess p;
		p.start(SAMBA_LDBEDIT_PATH, QStringList() << "-H" << RDS_SAMBA_PRIVATE + "/secrets.ldb" << tmpsecret.fileName());
		p.waitForFinished(30000);

		tmpsecret.close();

		if (p.exitCode() == 0)
		{
			return(true);
		}
		else
		{
			qCritical() << "Failed to update admin password in secrets file";
			qCritical() << p.readAll();
			return(ReturnValue(p.exitCode(), "Failed to update admin password in secrets file"));
		}
	}
	else
	{
		qCritical() << "Failed to add admin user to secrets file";
		qCritical() << p.readAll();
		return(ReturnValue(p.exitCode(), "Failed to add admin user to secrets file"));
	}

}

RdsSambaLdapSession& RdsSambaLdapSession::operator=(const RdsLdapSession & other)
{
	*(RdsLdapSession *)this = other;
	return(*this);
}

