/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "config.h"
#include "rdsprovisionthread.h"
#include "rdsprovisionthread_p.h"

#include <QFile>
#include <QDateTime>
#include <QProcess>
#include <QHostAddress>
#include <QHostInfo>
#include <QMutex>
#include <QMutexLocker>

#include <RdsDaemonManager>
#include <RdsDnsManager>
#include <RdsDhcpManager>
#include <RdsDhcpValues>
#include <RdsDhcpSubnet>
#include <RdsUtils>
#include <RdsSambaConfigParser>
#include <RdsSettings>
#include <RdsVolumeManager>
#include <RdsVolume>
#include <RdsNetworkSettings>
#include <RdsUser>
#include "main.h" // <-- What?
#include <unistd.h>

#define KRB5_TEMPLATE "[libdefaults]\n\
	default_realm = %REALM%\n\
	dns_lookup_realm = false\n\
	dns_lookup_kdc = false\n\
	ticket_lifetime = 24h\n\
	forwardable = yes\n\
	\n\
	[realms]\n\
	%REALM% = {\n\
	kdc = %FQDN%:88\n\
	admin_server = %FQDN%:749\n\
	default_domain = %REALM_LOWER%\n\
	}\n\
	\n\
	[domain_realm]\n\
	.%REALM_LOWER% = %REALM%\n\
	%REALM_LOWER% = %REALM%\n"

QMutex RdsProvisionThreadPrivate::provisionmutex(QMutex::Recursive);

RdsProvisionThread::RdsProvisionThread(QObject *parent)
		: QThread(parent)
{
}

RdsProvisionThread::RdsProvisionThread(const QVariantMap &settings, QObject *parent)
		: QThread(parent)
{
	qxt_d().settings = settings;
}

RdsProvisionThread::~RdsProvisionThread()
{
	quit();
	wait();
}

ReturnValue RdsProvisionThread::start(RdsProvisionator* service, const QVariantMap &settings)
{
	if (!RdsProvisionThreadPrivate::provisionmutex.tryLock())
	{
		return(ReturnValue(1, "This server is currently being provisioned"));
	}

	qxt_d().service = service;
	qxt_d().settings = settings;
	start();
	RdsProvisionThreadPrivate::provisionmutex.unlock();

	return(true);
}

void RdsProvisionThread::run()
{
	ReturnValue ret;

	if (!qxt_d().settings.contains("dns"))
	{
		qxt_d().completed(ReturnValue(1, "DNS servers were not provided"));
		return;
	}
	if (!qxt_d().settings.contains("domain"))
	{
		qxt_d().completed(ReturnValue(1, "The domain name was not provided"));
		return;
	}
	if (!qxt_d().settings.contains("gateway"))
	{
		qxt_d().completed(ReturnValue(1, "The gateway was not provided"));
		return;
	}
	if (!qxt_d().settings.contains("hostname"))
	{
		qxt_d().completed(ReturnValue(1, "The hostname was not provided"));
		return;
	}
	if (!qxt_d().settings.contains("ip"))
	{
		qxt_d().completed(ReturnValue(1, "The IP address was not provided"));
		return;
	}
	if (!qxt_d().settings.contains("realm"))
	{
		qxt_d().completed(ReturnValue(1, "The realm was not provided"));
		return;
	}
	

	//Remove any old settings
	RdsUtils::setFileContents(RDS_CONF_PATH + "/rds.conf", "");
	rdssettings()->clear();
	
	//define these early so DNS saving works
	QString domain = qxt_d().settings.value("domain").toString().toUpper();
	QString realm = qxt_d().settings.value("realm").toString().toUpper();
	QStringList parts = realm.toLower().split(".");
	QString basedn = "DC=" + parts.join(",DC=");
	
	rdssettings()->setValue("domain", domain);
	rdssettings()->setValue("realm", realm);
	rdssettings()->setValue("basedn", basedn);

	qxt_d().sendMajorProgress("Setting Date and Time");
	ret = setDateTime();
	if (ret.isError() && (ret.errNumber() != 1)) //Error numbers of 1 indicate that date and time info isn't present
	{
		qxt_d().completed(ret);
		return;
	}

	qxt_d().sendMajorProgress("Settings Hostname");
	ret = provisionHostname();
	if (ret.isError())
	{
		qxt_d().completed(ret);
		return;
	}

	qxt_d().sendMajorProgress("Provisioning Kerberos");
	ret = provisionKerberos();
	if (ret.isError())
	{
		qxt_d().completed(ret);
		return;
	}

	qxt_d().sendMajorProgress("Preparing DNS");
	ret = prepareDns();
	if (ret.isError())
	{
		qxt_d().completed(ret);
		return;
	}

	qxt_d().sendMajorProgress("Provisioning Samba");
	ret = provisionSamba();
	if (ret.isError())
	{
		qxt_d().completed(ret);
		return;
	}

	qxt_d().sendMajorProgress("Configuring DNS");
	ret = provisionDns();
	if (ret.isError())
	{
		qxt_d().completed(ret);
		return;
	}

	qxt_d().sendMajorProgress("Configuring DHCP");
	ret = provisionDhcp();
	if (ret.isError())
	{
		qxt_d().completed(ret);
		return;
	}

	qxt_d().sendMajorProgress("Configuring Samba");
	ret = provisionSambaConfig();
	if (ret.isError())
	{
		qxt_d().completed(ret);
		return;
	}

	qxt_d().sendMajorProgress("Completing Configuration");
	ret = provisionRds();
	if (ret.isError())
	{
		qxt_d().completed(ret);
		return;
	}

	qxt_d().completed(true);
}

ReturnValue RdsProvisionThread::setDateTime()
{
	if (!qxt_d().settings.contains("datetime"))
		return ReturnValue(1, "The date and time were not supplied.");
	if (!qxt_d().settings.contains("timezone"))
		return ReturnValue(1, "The timezone was not supplied.");

	QDateTime datetime = qxt_d().settings["datetime"].toDateTime();
	QString timezone = qxt_d().settings["timezone"].toString();

	//Because QFile::copy sucks
	if (QProcess::execute("cp", QStringList() << QString(RDS_TIMEZONE_FOLDER).arg(timezone) << "/etc/localtime") != 0)
		return ReturnValue(2, "Failed to copy timezone data.");

	//Another timezone file
	QFile tzfile(RDS_TIMEZONE);
	if (!tzfile.open(QFile::WriteOnly | QFile::Truncate))
		return ReturnValue(2, "Failed to open timezone file for writing.");

	tzfile.write(timezone.toAscii());
	tzfile.close();

	//set the date and time
	//qDebug() << "Setting date and time:" << "date" << "-s" << datetime.toString("MM/dd/yyyy hh:mm:ss");
	if (QProcess::execute("date", QStringList() << "-s" << datetime.toString("MM/dd/yyyy hh:mm:ss")) != 0)
		return ReturnValue(2, "Failed to set date and time");

	//sync the hardware clock to the system clock, and set it to UTC
	if (QProcess::execute("hwclock", QStringList() << "--systohc" << "--utc") != 0)
		qWarning() <<  "Failed to set the hardware clock";

	return(true);
}

ReturnValue RdsProvisionThread::provisionKerberos()
{
	if (!qxt_d().settings.contains("hostname"))
		return ReturnValue(1, "The server's hostname was not filled in on the settings");
	if (!qxt_d().settings.contains("realm"))
		return ReturnValue(1, "The domain was not filled in on the settings");

	QString realm = qxt_d().settings.value("realm").toString();
	QString hostname  = qxt_d().settings.value("hostname").toString();
	QString fqdn = hostname + "." + realm.toLower();
	QString krb5 = KRB5_TEMPLATE;

	krb5 = krb5.replace("%REALM%", realm);
	krb5 = krb5.replace("%REALM_LOWER%", realm.toLower());
	krb5 = krb5.replace("%FQDN%", fqdn);

	QFile file(RDS_KRB5);
	if (!file.open(QFile::WriteOnly | QFile::Truncate))
		return ReturnValue(2, QString("Failed to open ") + RDS_KRB5 + " for writing.");

	file.write(krb5.toAscii());
	file.close();

	return(true);
}

ReturnValue RdsProvisionThread::provisionSamba()
{
	if (!qxt_d().settings.contains("adminpw"))
		return ReturnValue(1, "The admin password was not supplied.");
	if (!qxt_d().settings.contains("realm"))
		return ReturnValue(1, "The realm was not supplied.");
	if (!qxt_d().settings.contains("domain"))
		return ReturnValue(1, "The domain was not supplied.");

	QString adminpw = qxt_d().settings.value("adminpw").toString();

	QProcess p;
	p.setProcessChannelMode(QProcess::MergedChannels);

	QProcessEnvironment env = QProcessEnvironment::systemEnvironment();
	foreach(QString key, qxt_d().settings.keys())
	{
		env.insert("PROVISION_" + key.toUpper(), qxt_d().settings[key].toString());
	}

	env.insert("RDS_SAMBA_PATH", RDS_SAMBA_PATH);
	env.insert("RDS_SAMBA_PRIVATE", RDS_SAMBA_PRIVATE);
	env.insert("RDS_SAMBA_CONFIGFILE", RDS_SAMBA_CONFIGFILE);
	env.insert("RDS_SAMBA_SYSVOL", RDS_SAMBA_SYSVOL);
	env.insert("RDS_SAMBA_PROVISION", RDS_SAMBA_PROVISION);

	p.setProcessEnvironment(env);
	p.start(RDS_DATA_PATH + "/provision");

	p.waitForStarted(-1);

	QString line;
	while (p.state() == QProcess::Running)
	{
		p.waitForReadyRead(10);
		line = p.readLine();
		line = line.replace("\n", "");
		if (line == "") continue;
		line = line.replace(adminpw, "*******");
		qxt_d().sendMinorProgress(line);
		qDebug() << line;
	}

	p.waitForFinished(-1);

	if (p.exitCode() != 0)
	{
		return ReturnValue(2, "Failed to provision samba.");
	}

	return(true);
}

ReturnValue RdsProvisionThread::prepareDns()
{
	if (!qxt_d().settings.contains("realm"))
		return ReturnValue(1, "The domain was not filled in on the settings");
	if (!qxt_d().settings.contains("ip"))
		return ReturnValue(1, "The IP address was not filled in on the settings");
	if (!qxt_d().settings.contains("dns"))
		return ReturnValue(1, "No DNS servers provided");

	if (QProcess::execute("which", QStringList() << "named") != 0)
		return(ReturnValue(1, "DNS server not installed"));

	QString domain = qxt_d().settings.value("realm").toString().toLower();
	QString ip = qxt_d().settings.value("ip").toString();
	QStringList dns = qxt_d().settings.value("dns").toStringList();

	RdsDnsManager mgr;
	mgr.reload();

	//Remove the old zone if it exists
	mgr.removeZone(domain + ".");

	return(mgr.save());
}

ReturnValue RdsProvisionThread::provisionDns()
{
	if (!qxt_d().settings.contains("realm"))
		return ReturnValue(1, "The domain was not filled in on the settings");
	if (!qxt_d().settings.contains("ip"))
		return ReturnValue(1, "The IP address was not filled in on the settings");
	if (!qxt_d().settings.contains("dns"))
		return ReturnValue(1, "No DNS servers provided");

	if (QProcess::execute("which", QStringList() << "named") != 0)
		return(ReturnValue(1, "DNS server not installed"));

	QString domain = qxt_d().settings.value("realm").toString().toLower();
	QString ip = qxt_d().settings.value("ip").toString();
	QStringList dns = qxt_d().settings.value("dns").toStringList();

	ReturnValue ret;
	if (!QFile::exists(DNS_DEFAULT_FILE))
	{
		return ReturnValue(1, "Failed to find the DNS default configuration file");
	}
	QFile file(DNS_DEFAULT_FILE);
	if (!file.open(QFile::ReadOnly))
	{
		return ReturnValue(1, "Failed to open the DNS default configuration file");
	}

	RdsDnsManager mgr;
	mgr.reload();

	if (!mgr.parse(file.readAll()))
	{
		mgr.reload();
		return ReturnValue(1, "Failed to parse the default DHCP settings");
	}

	ret = mgr.zone(domain + ".");
	if (ret.isError())
	{
		mgr.reload();
		return ret;
	}
	RdsDnsZone zone;
	zone = ret;

	ret = mgr.setForwarders(dns);
	if (ret.isError())
	{
		return ret;
	}

	ret = zone.save();
	if (ret.isError()) return(ret);
	ret = mgr.save();
	if (ret.isError()) return(ret);
	ret = mgr.restartService();
	if (ret.isError()) return(ret);

	//set up resolv.conf
	QFile rsfile("/etc/resolv.conf");
	if (!rsfile.open(QFile::WriteOnly | QFile::Truncate))
		return ReturnValue(2, "Failed to open /etc/resolv.conf file for writing.");

	rsfile.write(QString("domain %1\n").arg(domain).toAscii());
	rsfile.write(QString("search %1\n").arg(domain).toAscii());
	rsfile.write(QString("nameserver 127.0.0.1\n").toAscii());
	foreach(QString server, dns)
	{
		rsfile.write(QString("nameserver %1\n").arg(server).toAscii());
	}
	rsfile.close();

// 	return RdsUtils::runCommand("/usr/local/samba/sbin/samba_dnsupdate", QStringList() << "--verbose");
	return true;
}

ReturnValue RdsProvisionThread::provisionDhcp()
{
	//if provisiondhcp isn't provided, or if its false, just don't do it
	if (!qxt_d().settings.contains("provisiondhcp"))
		return(true);
	if (!qxt_d().settings["provisiondhcp"].toBool())
		return(true);
	if (!qxt_d().settings.contains("realm"))
		return ReturnValue(1, "The domain was not filled in on the settings");
	if (!qxt_d().settings.contains("ip"))
		return ReturnValue(1, "The IP was not filled in on the settings");
	if (!qxt_d().settings.contains("gateway"))
		return ReturnValue(1, "The gateway was not filled in on the settings");
	if (!qxt_d().settings.contains("dhcpnetmask"))
		return ReturnValue(1, "The DHCP netmask was not filled in on the settings");
	if (!qxt_d().settings.contains("dhcpstart"))
		return ReturnValue(1, "The DHCP start address was not filled in on the settings");
	if (!qxt_d().settings.contains("dhcpend"))
		return ReturnValue(1, "The DHCP end address was not filled in on the settings");

	if (QProcess::execute("which", QStringList() << "dhcpd3") != 0)
		return(ReturnValue(1, "DHCP server not installed"));

	//Erase the file
	RdsUtils::setFileContents(RDS_DHCP_CONFIGFILE, "");

	QString domain = qxt_d().settings.value("realm").toString().toLower();
	QString ip = qxt_d().settings["ip"].toString();
	QString gateway = qxt_d().settings["gateway"].toString();
	QString netmask = qxt_d().settings["dhcpnetmask"].toString();
	QString start = qxt_d().settings["dhcpstart"].toString();
	QString end = qxt_d().settings["dhcpend"].toString();
	QHostAddress startaddr(start);
	QHostAddress endaddr(end);
	QHostAddress netmaskaddr(netmask);
	QHostAddress networkaddr(startaddr.toIPv4Address() & netmaskaddr.toIPv4Address());

	RdsDhcpManager mgr;

	ReturnValue ret = mgr.values();
	if (ret.isError()) return(ret);

	RdsDhcpValues values = ret;
	values.setValue("ddns-update-style", "none");
	values.setValue("default-lease-time", "600");
	values.setValue("max-lease-time", "7200");
	values.setValue("authoritative", "");
	values.setOption("domain-name", "\"" + domain + "\"");
	values.setOption("domain-name-servers", ip);
	values.setOption("routers", gateway);

	ret = values.addSubnet("Default");
	if (ret.isError()) return(ret);

	RdsDhcpSubnet subnet = ret;
	subnet.setAddress(networkaddr);
	subnet.setNetmask(netmaskaddr);
	subnet.setRanges(QList<QPair<QHostAddress, QHostAddress> >() << QPair<QHostAddress, QHostAddress>(startaddr, endaddr));

	ret = mgr.save();
	if (ret.isError()) return(ret);
	ret = mgr.restartService();
	if (ret.isError()) return(ret);
	return true;
}

ReturnValue RdsProvisionThread::provisionHostname()
{
	if (!qxt_d().settings.contains("hostname"))
		return ReturnValue(1, "The server's hostname was not filled in on the settings");
	if (!qxt_d().settings.contains("ip"))
		return ReturnValue(1, "The server's IP address was not filled in on the settings");
	if (!qxt_d().settings.contains("realm"))
		return ReturnValue(1, "The domain was not filled in on the settings");

	QString realm = qxt_d().settings.value("realm").toString().toLower();
	QString hostname  = qxt_d().settings.value("hostname").toString();
	QString ip = qxt_d().settings.value("ip").toString();
	QString fqdn = hostname + "." + realm;

	//save hostname
	QFile hnfile(RDS_HOSTNAME);
	if (!hnfile.open(QFile::WriteOnly | QFile::Truncate))
		return ReturnValue(2, "Failed to open hostname file for writing.");

	hnfile.write(hostname.toAscii());
	hnfile.close();

	if (QProcess::execute("hostname", QStringList() << hostname) != 0)
		return ReturnValue(2, "Failed to set hostname");

	QFile hosts(RDS_HOSTS);
	if (!hosts.open(QFile::ReadOnly))
		return ReturnValue(2, QString("Failed to open ") + RDS_HOSTS + " for reading.");

	QStringList lines;
	bool foundip = false;
	while (!hosts.atEnd())
	{
		QString line = hosts.readLine();
		if (line.endsWith("\n")) line = line.mid(0, line.size() - 1);
		QStringList parts = line.split(QRegExp("[\\s]+"), QString::SkipEmptyParts);
		QStringList oldparts = parts; //keep an old list around so we can check for changes

		if (parts.size() == 0)
		{
			lines << line;
			continue;
		}

		QHostAddress address;
		if (!address.setAddress(parts[0]))
		{
			lines << line;
			continue;
		}

		if (parts.size() == 1)
		{
			continue;
		}

		//Remove instances of realm, hostname, and fqdn, so we don't have bad host entries
		int index = -1;
		if ((index = parts.indexOf(fqdn)) != -1) parts.removeAt(index);
		if ((index = parts.indexOf(hostname)) != -1) parts.removeAt(index);
		if ((index = parts.indexOf(realm)) != -1) parts.removeAt(index);

		if (parts[0] == ip)
		{
			parts.insert(1, realm);
			parts.insert(1, hostname);
			parts.insert(1, fqdn);
			foundip = true;
		}

		if (oldparts != parts)
		{
			if (parts.size() == 1) continue;
			lines << parts.join(" ");
		}
		else
		{
			lines << line;
		}
	}

	if (!foundip)
	{
		lines << ip + " " + fqdn + " " + hostname + " " + realm;
	}

	hosts.close();

	if (!hosts.open(QFile::WriteOnly | QFile::Truncate))
		return ReturnValue(2, QString("Failed to open ") + RDS_HOSTS + " for writing.");

	foreach(QString line, lines)
	{
		line = line + "\n";
		hosts.write(line.toAscii());
	}

	hosts.close();

	setenv("HOSTNAME",qPrintable(hostname),1);
	
	return(true);
}

ReturnValue RdsProvisionThread::provisionSambaConfig()
{
	RdsSambaConfigParser parser;
	parser.reload();
	
	RdsNetworkSettings net;
	ReturnValue ret = net.getDefaultInterface();
	ret = parser.setValue("global", "interfaces", ret.toString());
	if (ret.isError()) return(ret);

	ret = parser.setValue("sysvol", "browsable", "no");
        if (ret.isError()) return(ret);

	ret = parser.setValue("netlogon", "browsable", "no");
        if (ret.isError()) return(ret);

	
	parser.save();
	ret = parser.restartService();
	if (ret.isError()) return(ret);

	//Give samba time to restart
	///TODO find a better way to do this
	sleep(10);

	return(true);
}

ReturnValue RdsProvisionThread::provisionRds()
{
	if (!qxt_d().settings.contains("domain"))
		return ReturnValue(1, "The domain was not filled in on the settings");
	if (!qxt_d().settings.contains("realm"))
		return ReturnValue(1, "The realm was not filled in on the settings");

	QString domain = qxt_d().settings.value("domain").toString().toUpper();
	QString realm = qxt_d().settings.value("realm").toString().toUpper();
	QStringList parts = realm.toLower().split(".");
	QString basedn = "DC=" + parts.join(",DC=");

	rdssettings()->setValue("domain", domain);
	rdssettings()->setValue("realm", realm);
	rdssettings()->setValue("basedn", basedn);

	system("restart avahi-daemon"); //doesn't matter if this fails
	
	//Register services and create global LDAP session
	registerServices();

	//Add the root device as "Hard Disk"
	RdsVolumeManager mgr;
	ReturnValue ret = mgr.rootVolume();
	if (!ret.isError())
	{
		ret = mgr.volume(ret.toString());
		if (ret.isError()) return(ret);

		RdsVolume volume;
		volume = ret;

		ret = volume.id();
		if (ret.isError()) return(ret);

		ret = mgr.addVolume(ret.toString(), "Hard Disk");
		if (ret.isError()) return(ret);
		
		ret = volume.mount();
		if (ret.isError()) return(ret);
	}

	//Make sure the RDS admin user's password doesn't go bad
	RdsUser user(QString("CN=%1,CN=Users,%2").arg(rdssettings()->value("adminuser").toString()).arg(basedn));
	ret = user.setFlag(RdsUser::PasswordNeverExpires);
	if (ret.isError()) return(ret);
	ret = user.setFlag(RdsUser::CannotChangePassword);
	if (ret.isError()) return(ret);

	RdsUser user2(QString("CN=Administrator,CN=Users,%2").arg(basedn));
	user2.setLogonScript("launchapp.bat"); //set logon script, this is not a ciritcal error
	if (ret.isError()) qWarning() << "Failed to set administrator logon script:" << ret;
	ret = user2.setFlag(RdsUser::PasswordNeverExpires);
	if (ret.isError()) return(ret);

	RdsSambaConfigParser smb;
	ret = smb.setValue("global", "server string", QHostInfo::localHostName());
	if (ret.isError()) return(ret);

	rdssettings()->setValue("updates/dns", 1);
	
	ret = provisionNtpd();
	if (ret.isError()) return(ret);
	
	ret = smb.save();
	if (ret.isError()) return(ret);

	ret = RdsDaemonManager().restartService("Samba");
	if (ret.isError()) return(ret);

	//Done!
	rdssettings()->setValue("provisioned", true);
	rdssettings()->sync(); //make sure the settings are written to disk

	// Now we generate a password hash and assign it to "resara", which will just error if the user doesn't exist so we don't care.
	QString salt = "$1$";
	qsrand(time(NULL));
	for (int i = 0; i < 2; ++i)
	{
		int c = (qrand() % 0xFEFFFF) + 0x10000;
		salt += QByteArray::fromHex(QString::number(c, 16).toLocal8Bit()).toBase64();
	}
	salt = salt.replace("+", ".");
	salt += "$";
	QString password = crypt(qPrintable(qxt_d().settings.value("adminpw").toString()), qPrintable(salt));
	QProcess::execute("usermod", QStringList() << "-p" << password << "resara");

	return(true);
}

void RdsProvisionThreadPrivate::sendMinorProgress(const QString &str)
{
	QMetaObject::invokeMethod(service, "minorProgress", Qt::QueuedConnection, Q_ARG(QString, str));
}

void RdsProvisionThreadPrivate::sendMajorProgress(const QString &str)
{
	QMetaObject::invokeMethod(service, "majorProgress", Qt::QueuedConnection, Q_ARG(QString, str));
}

void RdsProvisionThreadPrivate::completed(const ReturnValue &ret)
{
	QMetaObject::invokeMethod(service, "completed", Qt::QueuedConnection, Q_ARG(ReturnValue, ret));
}

ReturnValue RdsProvisionThread::setSettings(const QVariantMap &settings)
{
	qxt_d().settings = settings;
	return(true);
}

ReturnValue RdsProvisionThread::provisionNtpd()
{
	QString text;
	bool founda = false, foundb=false;
	QFile file("/etc/ntp.conf");
	if(!file.open(QFile::ReadOnly))
	{
		return(ReturnValue("Failed to open /etc/ntp.conf"));
	}
	
	while(!file.atEnd())
	{
		QString line = file.readLine();
		QString tmp = line.replace("\n","").trimmed().simplified();
		
		//get rid of duplicate lines
		if(founda && tmp == "restrict default mssntp") continue;
		
		if(tmp == "restrict default mssntp") founda=true;
		if(tmp == "ntpsigndsocket " + RDS_SAMBA_RUN + "ntp_signd/") foundb=true;
		else if(tmp.startsWith("ntpsigndsocket")) continue;
		
		text += line + "\n";
	}
	
	if(!founda || !foundb)
	{
		file.close();
		if(!file.open(QFile::WriteOnly | QFile::Truncate))
		{
			return(ReturnValue("Failed to open /etc/ntp.conf for writing"));
		}
		
		file.write(text.toAscii());
		if(!founda) file.write("restrict default mssntp\n");
		if(!foundb) file.write(QString("ntpsigndsocket " + RDS_SAMBA_PATH + "var/run/ntp_signd/\n").toAscii());
		file.close();
		
		if(system("/etc/init.d/ntp restart") != 0)
		{
			return(ReturnValue(1,"Failed to restart ntpd."));
		}
	}
	
	return(true);
}

