/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "config.h"
#include "rdsnetworksettingsthread.h"
#include "rdsprovisionator.h"
#include "rdsprovisionthread_p.h"
#include <QDebug>
#include <QMutexLocker>
#include <QHostInfo>
#include <QStringList>
#include <QFile>
#include <RdsUtils>
#include <RdsDaemonManager>
#include <RdsDnsManager>
#include <RdsDnsRecord>
#include "main.h"

RdsNetworkSettingsThread::RdsNetworkSettingsThread(RdsProvisionator *parent)
		: QThread(parent),
		_service(parent)
{
}

RdsNetworkSettingsThread::~RdsNetworkSettingsThread()
{
	quit();
	wait();
}

ReturnValue RdsNetworkSettingsThread::start(const QVariantMap &settings)
{
	if (!RdsProvisionThreadPrivate::provisionmutex.tryLock())
	{
		return(ReturnValue(1, "This server is currently being provisioned"));
	}
	RdsProvisionThreadPrivate::provisionmutex.unlock();

	_settings = settings;

	connect(this, SIGNAL(finished()), this, SLOT(deleteLater()));
	start();

	return(true);
}

void RdsNetworkSettingsThread::run()
{
	QMutexLocker lock(&RdsProvisionThreadPrivate::provisionmutex);
	ReturnValue ret;

	if (!_settings.contains("dns"))
	{
		completed(ReturnValue(1, "DNS servers were not provided"));
		return;
	}
	if (!_settings.contains("ip"))
	{
		completed(ReturnValue(1, "The IP address was not provided"));
		return;
	}
	if (!_settings.contains("oldip"))
	{
		completed(ReturnValue(1, "The old IP address was not provided"));
		return;
	}

	sendMajorProgress("Updating hosts file");
	ret = updateHostsFile();
	if (ret.isError())
	{
		completed(ret);
		return;
	}

	sendMajorProgress("Updating resolv.conf");
	ret = updateResolveFile();
	if (ret.isError())
	{
		completed(ret);
		return;
	}

	sendMajorProgress("Updating DNS");
	ret = updateDns();
	if (ret.isError())
	{
		completed(ret);
		return;
	}

	sendMajorProgress("Restarting Samba");
	ret = restartSamba();
	if (ret.isError())
	{
		completed(ret);
		return;
	}

	sendMajorProgress("Publishing Services");
	ret = restartSamba();
	if (ret.isError())
	{
		completed(ret);
		return;
	}

	completed(true);
}

ReturnValue RdsNetworkSettingsThread::updateHostsFile()
{
	return(updateHostsFile(_settings));
}

ReturnValue RdsNetworkSettingsThread::updateHostsFile(QVariantMap settings)
{
	if (!settings.contains("ip"))
		return ReturnValue(1, "The server's IP address was not filled in on the settings");

	QString realm = RdsUtils::realm().toLower();
	QString hostname  = QHostInfo().localHostName();
	QString ip = settings.value("ip").toString();
	QString fqdn = hostname + "." + realm;

	QFile hosts(RDS_HOSTS);
	if (!hosts.open(QFile::ReadOnly))
		return ReturnValue(2, QString("Failed to open ") + RDS_HOSTS + " for reading.");

	QStringList lines;
	bool foundip = false;
	while (!hosts.atEnd())
	{
		QString line = hosts.readLine();
		if (line.endsWith("\n")) line = line.mid(0, line.size() - 1);
		QStringList parts = line.split(QRegExp("[\\s]+"), QString::SkipEmptyParts);
		QStringList oldparts = parts; //keep an old list around so we can check for changes

		if (parts.size() == 0)
		{
			lines << line;
			continue;
		}

		QHostAddress address;
		if (!address.setAddress(parts[0]))
		{
			lines << line;
			continue;
		}

		if (parts.size() == 1)
			continue;

		//Remove instances of realm, hostname, and fqdn, so we don't have bad host entries
		int index = -1;
		if ((index = parts.indexOf(fqdn)) != -1) parts.removeAt(index);
		if ((index = parts.indexOf(hostname)) != -1) parts.removeAt(index);
		if ((index = parts.indexOf(realm)) != -1) parts.removeAt(index);

		if (parts[0] == ip)
		{
			parts.insert(1, realm);
			parts.insert(1, hostname);
			parts.insert(1, fqdn);
			foundip = true;
		}

		if (oldparts != parts)
		{
			if(parts.size() == 1) continue;
			lines << parts.join(" ");
		}
		else
		{
			lines << line;
		}
	}

	if (!foundip)
	{
		lines << ip + " " + hostname + " " + fqdn + " " + realm;
	}

	hosts.close();

	if (!hosts.open(QFile::WriteOnly | QFile::Truncate))
		return ReturnValue(2, QString("Failed to open ") + RDS_HOSTS + " for writing.");

	foreach(QString line, lines)
	{
		line = line + "\n";
		hosts.write(line.toAscii());
	}

	hosts.close();

	return(true);
}

ReturnValue RdsNetworkSettingsThread::updateResolveFile()
{
	if (!_settings.contains("dns"))
		return ReturnValue(1, "No DNS servers provided");

	QStringList dns = _settings["dns"].toStringList();
	QString domain = RdsUtils::realm().toLower();

	//set up resolv.conf
	QFile rsfile("/etc/resolv.conf");
	if (!rsfile.open(QFile::WriteOnly | QFile::Truncate))
		return ReturnValue(2, "Failed to open /etc/resolv.conf file for writing.");

	rsfile.write(QString("domain %1\n").arg(domain).toAscii());
	rsfile.write(QString("search %1\n").arg(domain).toAscii());
	rsfile.write(QString("nameserver 127.0.0.1\n").toAscii());
	foreach(QString server, dns)
	{
		rsfile.write(QString("nameserver %1\n").arg(server).toAscii());
	}
	rsfile.close();

	return true;
}

ReturnValue RdsNetworkSettingsThread::updateDns()
{
	if (!_settings.contains("dns"))
		return ReturnValue(1, "No DNS servers provided");

	QStringList dnsservers = _settings["dns"].toStringList();
	QString domain = RdsUtils::realm().toLower();

	RdsDnsManager dns;
	dns.setForwarders(dnsservers);

	ReturnValue ret = dns.zone(domain + ".");
	if (ret.isError()) return(ret);

	RdsDnsZone zone;
	zone = ret;

	ret = zone.listRecordsByType(RdsDnsZone::A);
	if (ret.isError())
		return ret;

	RdsDnsRecordList list = ret.value<RdsDnsRecordList>();
	foreach(RdsDnsRecord record, list)
	{
		if (record.value() == _settings["oldip"] ||
		        record.key() == "gc._msdcs" ||
		        record.key() == QHostInfo::localHostName() ||
		        record.key().isEmpty())
		{
			record.setValue(_settings["ip"].toString());
			ret = zone.addRecord(record);
			if (ret.isError())
				return ret;
		}
	}

	ret = zone.save();
	if (ret.isError())
		return ret;

	ret = dns.save();
	if (ret.isError())
		return ret;

	return(dns.restartService());
}

ReturnValue RdsNetworkSettingsThread::publishServices()
{
	republishServices();
	return(true);
}


ReturnValue RdsNetworkSettingsThread::restartSamba()
{
	return(RdsDaemonManager().restartService("Samba"));
}

void RdsNetworkSettingsThread::sendMinorProgress(const QString &str)
{
	QMetaObject::invokeMethod(_service, "minorProgress", Qt::QueuedConnection, Q_ARG(QString, str));
}

void RdsNetworkSettingsThread::sendMajorProgress(const QString &str)
{
	QMetaObject::invokeMethod(_service, "majorProgress", Qt::QueuedConnection, Q_ARG(QString, str));
}

void RdsNetworkSettingsThread::completed(const ReturnValue &ret)
{
	QMetaObject::invokeMethod(_service, "completed", Qt::QueuedConnection, Q_ARG(ReturnValue, ret));
}

