/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/


#include "storageeditorwidget.h"
#include <RdsVolume>
#include <QMessageBox>
#include <QDebug>

StorageEditorWidget::StorageEditorWidget(RdsVolumeManager *manager, QWidget* parent)
		: RdsEntityWidget(manager, parent), Ui::StorageEditWidget()
{
	setupUi(this);
	_manager = manager;

	setApplyButton(ApplyButton);
	setDiscardButton(DiscardButton);

	//General
	addWidget("name", Name, NoMulti, "^[a-zA-Z0-9 \\-_]*$", "The name may only contain letters, numbers, underscore(_), and hyphen(-).");
	addWidget("serialnumber", SerialNumber, NoMulti);
	addWidget("model", Model, NoMulti);
	addWidget("type", Type, NoMulti);
	addWidget("fs", Filesystem, NoMulti);

	QObject::connect(manager, SIGNAL(entityRenamed(QString, QString)), this, SLOT(entityRenamed(QString, QString)));
}

StorageEditorWidget::~StorageEditorWidget()
{
}

void StorageEditorWidget::entityRenamed(QString oldid, QString newid)
{
	Q_UNUSED(oldid);
	Q_UNUSED(newid);
}

#define GET_DATA(name,function) ret = function; \
	if(ret.isError() && !ret.errString().endsWith("attribute does not exist")) err = ret; \
	fields[name] = ret

ReturnValue StorageEditorWidget::getData()
{
	QVariantMap fields;

	if (input() == "") return(ReturnValue(1, "Invalid Volume"));

	ReturnValue err = true;
	ReturnValue ret;
	
	ret = _manager->volume(input());
	if(ret.isError())
	{
		qWarning() << "Error:" << ret << input();
		err = ret;
	}
	
	RdsVolume volume;
	volume = ret;

	GET_DATA("name", volume.name());
	GET_DATA("serialnumber", volume.serialNumber());
	GET_DATA("model", volume.model());
	GET_DATA("type", volume.type());
	GET_DATA("fs", volume.fsType());

	if (err.isError())
	{
		QMessageBox msg(QMessageBox::Warning, "Error", "There was an error opening this volume: " + ret.errString(), QMessageBox::Ok);
		QAbstractButton *editbutton = msg.addButton("Edit Anyway", QMessageBox::RejectRole);
		msg.exec();
		if (msg.clickedButton() == editbutton)
		{
			if (QMessageBox::warning(this, "Warning", "Editing in this state may be dangerous. Continue?", QMessageBox::Yes | QMessageBox::No, QMessageBox::No) == QMessageBox::Yes)
			{
				return(fields);
			}
		}
		return(err);
	}


	return(fields);
}

#define SET_FIELD(field,function) if (name == field) \
	{ \
		ret = function; \
		if (ret.isError()) err = ret; \
	}

#define SET_FIELD_STRING(field,function) \
	SET_FIELD(field,volume.function(value.toString()));

bool StorageEditorWidget::setData(QString id, QVariantMap fields)
{
	ReturnValue err = true;
	ReturnValue ret;
	
	ret = _manager->volume(id);
	if(ret.isError())
	{
		QMessageBox::critical(this,"Error","Failed to get volume object: " + ret.errString());
		return(false);
	}
	
	RdsVolume volume;
	volume = ret;

	foreach(QString name, fields.keys())
	{
		QVariant value = fields[name];

		SET_FIELD_STRING("name",setName);
		_manager->updateEntity(id);
	}

	if (err.isError())
	{
		qWarning() << "Failed to save volume:" << id << err.errString();
		return(false);
	}
	else return(true);
}

void StorageEditorWidget::entityUpdated(QString id)
{
	Q_UNUSED(id);
}

