/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/


#include "shareeditwidget.h"
#include <QMessageBox>
#include <QDebug>
#include <RdsShare>
#include <RdsUtils>
#include <RdsDaemonManager>
#include <RdsFileManager>
#include "browsedialog.h"
#include "addsharedialog.h"
#include "config.h"

ShareEditWidget::ShareEditWidget(RdsShareManager *manager, QWidget* parent)
		: RdsEntityWidget(manager, parent), Ui::ShareEditWidget()
{
	setupUi(this);
	BrowseButton->setIcon(QPixmap(findRdsIcon("icons/16x16/folder.png")));
	setApplyButton(ApplyButton);
	setDiscardButton(DiscardButton);

	addWidget("name", Name, NoMulti, "^[a-zA-Z0-9 \\-_]+$", "The name may only contain letters, numbers, underscore(_), and hypthen(-).");
	addWidget("description", Description);
	addWidget("path", Path);
	addWidget("readonly", ReadOnly);
	addWidget("hidden", Hidden);
	addWidget("acl", AclWidget, "acl", SIGNAL(changed()), NoMulti, QStringList());
	addWidget("mapstring", DriveMap, "mapstring", SIGNAL(changed()), NoMulti, QStringList());

	QObject::connect(manager, SIGNAL(entityRenamed(QString, QString)), this, SLOT(entityRenamed(QString, QString)));

	_sharemgr = manager;
}

ShareEditWidget::~ShareEditWidget()
{
}

void ShareEditWidget::entityRenamed(QString oldid, QString newid)
{
	if (oldid != input()) return;

	QStringList outlist = outputs();
	for (int i = 0; i < outlist.size(); i++)
	{
		if (outlist[i] == input()) outlist[i] = newid;
	}

	setInput(newid);
	setOutputs(outlist);

	setField("name", RdsUtils::getShareName(newid));
}

#define GET_DATA(name,function) ret = function; \
	if(ret.isError() && !ret.errString().endsWith("attribute does not exist")\
	        && (ret.errString() != "Operation not supported") && (ret.errString() != "No data available")){err = ret;} \
	fields[name] = ret

ReturnValue ShareEditWidget::getData()
{
	QVariantMap fields;

	if (input() == "") return(ReturnValue(1, "Invalid Share"));

	ReturnValue err = true;
	ReturnValue ret = _sharemgr->share(RdsUtils::getShareName(input()).toString());
	if (ret.isError()) return(ret);
	RdsShare share = ret;

	GET_DATA("name", share.name());
	GET_DATA("description", share.description());
	GET_DATA("path", share.path());
	GET_DATA("readonly", share.readOnly());
	GET_DATA("hidden", share.hidden());
	GET_DATA("mapstring", share.mapString());

	RdsFileManager manager;
	ret = manager.init();
	if (ret.isError()) return(ret);

	AclWidget->setPath(fields["path"].toString());
	ret = manager.ntPermissions(fields["path"].toString());
	if (ret.isError())
	{
		AclWidget->setEnabled(false);
		setFieldEnabled("acl", false);
		fields["acl"] = QVariant::fromValue(RdsNtAcl());
	}
	else
	{
		fields["acl"] = ret;
	}

	if (err.isError())
	{
		QMessageBox msg(QMessageBox::Warning, "Error", "There was an error opening this share: " + ret.errString(), QMessageBox::Ok);
		QAbstractButton *editbutton = msg.addButton("Edit Anyway", QMessageBox::RejectRole);
		msg.exec();
		if (msg.clickedButton() == editbutton)
		{
			if (QMessageBox::warning(this, "Warning", "Editing in this state may be dangerous. Continue?", QMessageBox::Yes | QMessageBox::No, QMessageBox::No) == QMessageBox::Yes)
			{
				return(fields);
			}
		}
		return(err);
	}

	return(fields);
}

#define SET_FIELD(field,function) if (name == field) \
	{ \
		ret = function; \
		if (ret.isError()) err = ret; \
	}

#define SET_FIELD_STRING(field,function) \
	SET_FIELD(field,share.function(value.toString()));
#define SET_FIELD_BOOL(field,function) \
	SET_FIELD(field,share.function(value.toBool()));

bool ShareEditWidget::setData(QString id, QVariantMap fields)
{
	ReturnValue err = true;
	ReturnValue ret = _sharemgr->share(RdsUtils::getShareName(id).toString());
	if (ret.isError()) return(false);
	RdsShare share = ret;

	foreach(QString name, fields.keys())
	{
		QVariant value = fields[name];

		RdsFileManager fmgr;
		ret = fmgr.init();
		if (ret.isError()) qWarning() << "Failed to get file manager:" << ret;

		if (name == "path")
		{
			AddShareDialog::checkValidVolume(value.toString());
			//create the directory in case it we reinitialized the volume
			ret = fmgr.createDirectory(value.toString());
			if (ret.isError()) qWarning() << "Failed to create share path:" << ret;

		}

		SET_FIELD_STRING("description", setDescription);
		SET_FIELD_STRING("path", setPath);
		SET_FIELD_STRING("mapstring", setMapString);
		SET_FIELD_BOOL("readonly", setReadOnly);
		SET_FIELD_BOOL("hidden", setHidden);

		if(name == "acl")
		{
			RdsNtAcl acl = value.value<RdsNtAcl>();
			ret = fmgr.setNtPermissions(Path->text(), acl);
			if (ret.isError())
			{
				err = ret;
			}
		}
		if (name == "path")
		{
			manager()->removeEntity(id);
			manager()->addEntity(id);
		}
		if (name == "name")
		{
			//qDebug() << "setting name" << value;
			ret = share.setName(value.toString());
			if (ret.isError())
			{
				err = ret;
			}
			else
			{
				QString name = ret.toString();
				name = "@SHARE/" + name + "/";

				QStringList outlist = outputs();
				for (int i = 0; i < outlist.size(); i++)
				{
					if (outlist[i] == id) outlist[i] = name;
				}

				setInput(name);
				setOutputs(outlist);

				manager()->renameEntity(id, name);
			}
		}
	}

	if (err.isError())
	{
		qWarning() << "Failed to save share:" << id << err.errString();
		return(false);
	}
	else return(true);
}

void ShareEditWidget::entityUpdated(QString id)
{
	if (id != input()) return;

	if (unsavedChanges())
	{
		if (QMessageBox::question(this, "Share Changed",
		                          "Another user has made changes to the item you are editing. Would you like to overwrite your changes?",
		                          QMessageBox::Yes | QMessageBox::No, QMessageBox::No) == QMessageBox::Yes)
		{
			RdsEntityWidget::entityUpdated(id);
		}
	}
	else
	{
		RdsEntityWidget::entityUpdated(id);
	}
}

bool ShareEditWidget::endSetData()
{
	ReturnValue ret = _sharemgr->save();
	if (ret.isError())
	{
		qWarning() << "Failed to save changes:" << ret;
		return(false);
	}

	RdsDaemonManager mgr;
	ret = mgr.init();

	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to get RdsDaemonManager service: " + ret.errString());
		return(false);
	}

	ret = mgr.reloadService("Samba");

	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to restart samba: " + ret.errString());
		return(false);
	}

	return(true);
}

void ShareEditWidget::on_BrowseButton_clicked()
{
	BrowseDialog dialog(BrowseDialog::Volumes | BrowseDialog::FileSystem);

	if (dialog.exec())
	{
		if (dialog.path() == "") return;
		ReturnValue ret = RdsUtils::getSystemPath(dialog.path());

		if (ret.isError())
		{
			QMessageBox::critical(this, "Error", "Failed to look up path: " + ret.errString());
			return;
		}

		Path->setText(ret.toString());
		fieldChanged(Path);
	}
}

