/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdsstoragemodel.h"
#include <QPixmap>
#include <QDebug>
#include <QBrush>
#include <QColor>
#include "config.h"

RdsStorageModel::RdsStorageModel(RdsVolumeManager *manager, QObject *parent)
		: RdsEntityModel(manager, 5, parent)
{
	_manager = manager;
	setCachedPixmap("Hard Disk", QPixmap(findRdsIcon("./icons/16x16/hdd.png")));
	setCachedPixmap("CDROM", QPixmap(findRdsIcon("./icons/16x16/cdrom.png")));
	setCachedPixmap("USB Disk", QPixmap(findRdsIcon("./icons/16x16/usbdisk.png")));
}


RdsStorageModel::~RdsStorageModel()
{
}

QVariant RdsStorageModel::userData(const QModelIndex &index, int role, RdsEntityModel::Cache *cache) const
{
	if ((role == Qt::DecorationRole) && (index.column() == 0))
	{
		if (cache->metadata().contains("type"))
		{
			return(cachedPixmap(cache->metadata()["type"].toString()));
		}
	}
	else if (role == Qt::DisplayRole)
	{
		switch (index.column())
		{
			case 0:
				return(cache->name());
			case 1:
				if (cache->metadata().contains("type")) return(cache->metadata()["type"]);
				break;
			case 2:
				if (cache->metadata().contains("statustext")) return(cache->metadata()["statustext"]);
				break;
			case 3:
				if (cache->metadata().contains("size"))
					return(toHumanReadableFormat(cache->metadata()["size"].toLongLong()));
				break;
			case 4:
				if ((cache->metadata().contains("status")) && (cache->metadata()["status"].toString().toLower() == "disconnected"))
				{
					return("Unavailable");
				}
				if (cache->metadata().contains("free"))
					return(toHumanReadableFormat(cache->metadata()["free"].toLongLong()));
				break;
			default:
				return QVariant();
		}
	}
	else if (role == Qt::ForegroundRole)
	{
		if (!cache->metadata().contains("status")) return(QVariant());
		QString status = cache->metadata()["status"].toString();

		if (index.column() == 2)
		{
			if (status.toLower() == "normal")
			{
				return(QBrush(QColor(0, 180, 0)));
			}
			else if (status.toLower() == "warning")
			{
				return(QBrush(QColor(220, 120, 0)));
			}
			else if (status.toLower() == "error")
			{
				return(QBrush(QColor(180, 0, 0)));
			}
			else if (status.toLower() == "disconnected")
			{
				return(QBrush(QColor(180, 180, 180)));
			}
		}
		else if (status.toLower() == "disconnected")
		{
			return(QBrush(QColor(180, 180, 180)));
		}
	}
	if (role == Qt::ToolTipRole)
	{
		if (cache->metadata().contains("statusdetails"))
		{
			return(cache->metadata()["statusdetails"]);
		}
	}

	return(QVariant());
}

QVariant RdsStorageModel::userHeaderData(int section, int role) const
{
	if (role == Qt::DisplayRole)
	{
		switch (section)
		{
			case 0:
				return("Name");
			case 1:
				return("Type");
			case 2:
				return("Status");
			case 3:
				return("Size");
			case 4:
				return("Free Space");
		}
	}

	return(QVariant());
}

Qt::ItemFlags RdsStorageModel::flags(const QModelIndex &index) const
{
	if (index.isValid())
	{
		Qt::ItemFlags f = Qt::ItemIsDropEnabled | Qt::ItemIsDragEnabled | Qt::ItemIsEnabled | Qt::ItemIsSelectable;

		return(f);
	}
	else
		return Qt::ItemIsDropEnabled;
}

bool RdsStorageModel::setData(const QModelIndex &index, const QVariant &value, int role, Cache *cache)
{
	Q_UNUSED(index);
	Q_UNUSED(value);
	Q_UNUSED(role);
	Q_UNUSED(cache);
	return(false);
}

void RdsStorageModel::move(Cache *entity, Cache *parent)
{
	Q_UNUSED(entity);
	Q_UNUSED(parent);
	return;
}

QString RdsStorageModel::toHumanReadableFormat(qlonglong size) const
{
	double tmp = size;
	int i = 0;

	while (tmp >= 1)
	{
		tmp /= 1024;
		if (tmp >= 1) i++;
	}

	QString suffix;
	switch (i)
	{
		case 0:
			suffix = "Bytes";
			break;
		case 1:
			suffix = "Kb";
			break;
		case 2:
			suffix = "Mb";
			break;
		case 3:
			suffix = "Gb";
			break;
		case 4:
			suffix = "Tb";
			break;
		case 5:
			suffix = "Pb";
			break;
		default:
			return("Really Big");
	}

	double result = (double)tmp * 1000;
	QString tmpstring = QString("%1").arg(result, 0, 'f', 2);
	//remove trailing zeros if any
	if ((tmpstring.indexOf(".") != -1) && (tmpstring.endsWith("0"))) tmpstring.truncate(tmpstring.length() - 1);
	if ((tmpstring.indexOf(".") != -1) && (tmpstring.endsWith("0"))) tmpstring.truncate(tmpstring.length() - 1);
	if ((tmpstring.indexOf(".") != -1) && (tmpstring.endsWith("."))) tmpstring.truncate(tmpstring.length() - 1);

	return(QString("%1 %2").arg(tmpstring).arg(suffix));
}
