/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdsentitymodel.h"
#include "rdsentitymodel_p.h"
#include <QDebug>
#include <assert.h>
#include <QStringList>
#include <QMimeData>
#include <QTimer>
#include <QTreeView>
#include <QPainter>
#include <QMetaObject>
#include "config.h"

RdsEntityModel::Cache::Cache()
{
	qxt_d().parent = NULL;
}

RdsEntityModel::Cache::Cache(RdsEntityModel::Cache *cache)
{
	qxt_d().parent = cache;
}

RdsEntityModel::Cache::Cache(const RdsEntity &entity, RdsEntityModel::Cache *cache)
{
	qxt_d().parent = cache;
	qxt_d().name = entity.name();
	qxt_d().id = entity.id();
	qxt_d().type = entity.type();
	qxt_d().visible = entity.visible();
	qxt_d().metadata = entity.metadata();
	qxt_d().group = entity.isGroup();
	qxt_d().hasmore = entity.hasMore();
}

RdsEntityModel::Cache &RdsEntityModel::Cache::operator =(const RdsEntity & entity)
{
	qxt_d().name = entity.name();
	qxt_d().id = entity.id();
	qxt_d().type = entity.type();
	qxt_d().visible = entity.visible();
	qxt_d().metadata = entity.metadata();
	qxt_d().group = entity.isGroup();
	qxt_d().hasmore = entity.hasMore();

	return(*this);
}

QString RdsEntityModel::Cache::name() const
{
	return(qxt_d().name);
}

QString RdsEntityModel::Cache::id() const
{
	return(qxt_d().id);
}

QString RdsEntityModel::Cache::type() const
{
	return(qxt_d().type);
}

bool RdsEntityModel::Cache::visible() const
{
	return(qxt_d().visible);
}

QVariantMap &RdsEntityModel::Cache::metadata()
{
	return(qxt_d().metadata);
}

QList<RdsEntityModel::Cache *> &RdsEntityModel::Cache::children()
{
	return(qxt_d().children);
}

RdsEntityModel::Cache *RdsEntityModel::Cache::parent()
{
	return(qxt_d().parent);
}

bool RdsEntityModel::Cache::isGroup()
{
	return(qxt_d().group);
}

bool RdsEntityModel::Cache::hasMore()
{
	return(qxt_d().hasmore);
}

void RdsEntityModel::Cache::setHasMore(bool more)
{
	qxt_d().hasmore = more;
}

void RdsEntityModel::Cache::setName(QString name)
{
	qxt_d().name = name;
}


RdsEntityModel::RdsEntityModel(RdsEntityManager *manager, int columns, QObject *parent)
		: QAbstractItemModel(parent)
{
	qxt_d().manager = manager;
	qxt_d().columns = columns;
	qxt_d().loadingframe = 0;
	qxt_d().loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-0.png"));
	qxt_d().loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-1.png"));
	qxt_d().loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-2.png"));
	qxt_d().loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-3.png"));
	qxt_d().loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-4.png"));
	qxt_d().loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-5.png"));
	qxt_d().loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-6.png"));
	qxt_d().loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-7.png"));
	qxt_d().loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-8.png"));
	qxt_d().loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-9.png"));
	qxt_d().loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-10.png"));
	qxt_d().loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-11.png"));
	qxt_d().editicon =  QPixmap(findRdsIcon("./icons/16x16/pencil.png"));
	
	QTimer *timer = new QTimer(this);
	QObject::connect(timer, SIGNAL(timeout()), this, SLOT(onTimerUpdate()));
	timer->start(75);

	RdsEntity ent;
	ent.setId("loading");
	ent.setName("Loading...");
	ent.setType("loading");
	ent.setVisible(true);
	qxt_d().root = new Cache(ent);
	qxt_d().constrainselection = true;

	ReturnValue ret = manager->listEntities(this, SLOT(entitiesReturned(uint, ReturnValue)), "");
	if (ret.isError()) entitiesReturned(0, ret);
	
	ret = manager->editingList();
	if(ret.isError())
	{
		qWarning() << "Failed to get editing list:" << ret;
	}
	else
	{
		qxt_d().editlist = ret.toStringList();
	}
	
	//qDebug() << "Editing:" << qxt_d().editlist;
}


RdsEntityModel::~RdsEntityModel()
{
}

int RdsEntityModel::rowCount(const QModelIndex &parent) const
{
	if (parent.column() != -1 && parent.column() != 0) return(0);
	if (!parent.isValid())
	{
		return(qxt_d().root->children().count());
	}
	else
	{
		Cache *cache = (Cache *)parent.internalPointer();
		return(cache->children().count());
	}
}

QModelIndex RdsEntityModel::index(int row, int column, const QModelIndex &parent) const
{
	if (!hasIndex(row, column, parent)) return(QModelIndex());
	Cache *cache = NULL;
	if (parent.isValid())
	{
		cache = (Cache *)parent.internalPointer();
		assert(cache != NULL);
	}
	else
	{
		cache = qxt_d().root;
	}

	if (row >= cache->children().size())
	{
		return(QModelIndex());
	}

	return(createIndex(row, column, cache->children()[row]));
}

QModelIndex RdsEntityModel::parent(const QModelIndex &index) const
{
	if (!index.isValid()) return(QModelIndex());
	Cache *cache = (Cache *)index.internalPointer();

	if (cache->parent() == NULL) return(QModelIndex());
	Cache *parent = cache->parent();

	//look up the index of the parent in its parents list of children
	if (parent->parent() == NULL) return(QModelIndex()); //return(createIndex(0,0,qxt_d().root));
	int row = parent->parent()->children().indexOf(parent);

	return(createIndex(row, 0, parent));
}

Qt::ItemFlags RdsEntityModel::flags(const QModelIndex &index) const
{
	if (index.isValid())
	{
		Cache *cache = (Cache *)index.internalPointer();
		Qt::ItemFlags f = Qt::ItemIsDropEnabled | Qt::ItemIsDragEnabled | Qt::ItemIsEnabled | Qt::ItemIsSelectable;
		f |= flags(index, cache);

		return(f);
	}
	else
		return Qt::ItemIsDropEnabled;
}

Qt::ItemFlags RdsEntityModel::flags(const QModelIndex &index, Cache *cache) const
{
	Q_UNUSED(index);
	Q_UNUSED(cache);
	return(0);
}

QVariant RdsEntityModel::data(const QModelIndex &index, int role) const
{
	if (!index.isValid())
		return QVariant();

	Cache *cache = (Cache *)index.internalPointer();
	if (cache->type() == "loading")
	{
		if (index.column() != 0) return(QVariant());
		if (role == Qt::DisplayRole || role == Qt::EditRole)
		{
			return("Loading...");
		}
		else if (role == Qt::DecorationRole)
		{
			return(qxt_d().loadingframes[qxt_d().loadingframe]);
		}
	}
	else
	{
		if (role == Qt::DisplayRole || role == Qt::EditRole)
		{
			QVariant tmp = userData(index, role, cache);
			if (tmp.isNull() && (index.column() == 0))
				return(cache->name());
			else
				return(tmp);
		}
		else
		{
			if ((role == Qt::DecorationRole) && (index.column() == 0) && (qxt_d().editlist.contains(cache->id())))
			{
				QPixmap base(16, 16);
				base.fill(QColor::fromRgba(0));
				QPixmap orig = userData(index, role, cache).value<QPixmap>();

				QPainter p;
				p.begin(&base);

				p.drawPixmap(QRect(0, 0, 16, 16), orig, QRect(0, 0, 16, 16));
				p.drawPixmap(QRect(0, 0, 16, 16), qxt_d().editicon, QRect(0, 0, 16, 16));

				p.end();

				return(base);
			}
			else
			{
				return(userData(index, role, cache));
			}
		}
	}

	return QVariant();
}

bool RdsEntityModel::setData(const QModelIndex &index, const QVariant &value, int role)
{
	if (!index.isValid()) return(false);
	Cache *cache = (Cache *)index.internalPointer();
	return(setData(index, value, role, cache));
}

bool RdsEntityModel::setData(const QModelIndex &index, const QVariant &value, int role, Cache *cache)
{
	Q_UNUSED(index);
	Q_UNUSED(value);
	Q_UNUSED(role);
	Q_UNUSED(cache);
	return(false);
}

QVariant RdsEntityModel::userData(const QModelIndex &index, int role, RdsEntityModel::Cache *cache) const
{
	Q_UNUSED(index);
	Q_UNUSED(role);
	Q_UNUSED(cache);
	return(QVariant());
}

QVariant RdsEntityModel::headerData(int section, Qt::Orientation orientation, int role) const
{
	if (section >= qxt_d().columns) return(QVariant());
	if (orientation != Qt::Horizontal) return(QVariant());
	return(userHeaderData(section, role));
}

QVariant RdsEntityModel::userHeaderData(int section, int role) const
{
	Q_UNUSED(section);
	Q_UNUSED(role);
	return(QVariant());
}

int RdsEntityModel::columnCount(const QModelIndex &parent) const
{
	Q_UNUSED(parent);
	return(qxt_d().columns);
}

void RdsEntityModel::loadEntity(RdsEntity entity, Cache *cache)
{
	//qDebug() << "Loading Entity" << entity.name();
	foreach(RdsEntity childentity, entity.children())
	{
		//qDebug() << "ENTITY:" << childentity.id() << childentity.parent();
		Cache *child = new Cache(childentity, cache);
		qxt_d().idmap[childentity.id()] = child;
		cache->children() << child;
		loadEntity(childentity, child);
	}
}

void RdsEntityModel::entitiesReturned(uint id, ReturnValue ret)
{
	Q_UNUSED(id);
	if (ret.isError())
	{
		//do something here
		return;
	}

	delete qxt_d().root;

	RdsEntity entity = ret.value<RdsEntity>();

	beginResetModel();
	qxt_d().root = new Cache(entity);
	qxt_d().idmap[entity.id()] = qxt_d().root;
	loadEntity(entity, qxt_d().root = qxt_d().root);
	//qDebug() << "ENTITY:" << entity.id() << entity.parent();
	endResetModel();

	QObject::connect(qxt_d().manager, SIGNAL(entityAdded(QString)), this, SLOT(entityAdded(QString)));
	QObject::connect(qxt_d().manager, SIGNAL(entityRemoved(QString)), this, SLOT(entityRemoved(QString)));
	QObject::connect(qxt_d().manager, SIGNAL(entityUpdated(QString)), this, SLOT(entityUpdated(QString)));
	QObject::connect(qxt_d().manager, SIGNAL(entityMetadataUpdated(QString,QVariantMap)), this, SLOT(entityMetadataUpdated(QString,QVariantMap)));
	QObject::connect(qxt_d().manager, SIGNAL(entityRenamed(QString, QString)), this, SLOT(entityRenamed(QString, QString)));
	QObject::connect(qxt_d().manager, SIGNAL(editingStarted(QString)), this, SLOT(editingStarted(QString)));
	QObject::connect(qxt_d().manager, SIGNAL(editingStopped(QString)), this, SLOT(editingStopped(QString)));
	
	emit(loaded());
}

RdsEntityModel::Cache *RdsEntityModel::getCache(const QModelIndex &index) const
{
	if (index.model() != this) return(NULL);
	return((Cache *)index.internalPointer());
}

RdsEntityModel::Cache *RdsEntityModel::getCache(QString dn) const
{
	if (!qxt_d().idmap.contains(dn)) return(NULL);
	return(qxt_d().idmap.value(dn));
}

void RdsEntityModel::entityAdded(QString id)
{
	if (qxt_d().idmap.contains(id))
	{
		//qDebug() << "Skipping Add:" << id;
		return;
	}

	ReturnValue ret = manager()->listEntities(id);
	if (ret.isError())
	{
		qWarning() << "Failed to list entity" << id << ":" << ret.errString();
		return;
	}

	if (qxt_d().idmap.contains(id))
	{
		deleteCache(qxt_d().idmap.value(id));
	}

	RdsEntity entity = ret.value<RdsEntity>();
	//qDebug() << entity.parent();
	if (!qxt_d().idmap.contains(entity.parent()))
	{
		qWarning() << "Failed to find parent:" << entity.parent() << "for entity" << entity.id() << "Cannot add.";
		return;
	}

	Cache *parent = qxt_d().idmap.value(entity.parent());
	if (parent->hasMore()) return; //If we have to load more data for this later, don't add anything now

	int row = parent->children().size();
	if (row < 0) row = 0;
	QModelIndex parentindex = indexFromCache(parent);
	if (parent == qxt_d().root) parentindex = QModelIndex();
	beginInsertRows(parentindex, row, row);
	Cache *cache = new Cache(entity, parent);
	parent->children() << cache;
	qxt_d().idmap[id] = cache;
	if (!cache->hasMore())
	{
		loadEntity(entity, cache);
	}
	endInsertRows();
}

void RdsEntityModel::entityRemoved(QString id, bool events)
{
	//qDebug() << "Removing:" << id << events;
	if (!qxt_d().idmap.contains(id)) return; //we don't have a record of it, so don't delete it!

	Cache *cache = qxt_d().idmap.value(id);
	if (cache->parent() == NULL) return;

	//qDebug() << "Actually Removing:" << id;

	//qDebug() << cache << cache->parent() << cache->parent()->children();
	int row = cache->parent()->children().indexOf(cache);
	//qDebug() << "Removing Rows:" << indexFromCache(cache->parent());
	QModelIndex parentindex = indexFromCache(cache->parent());
	if (cache->parent() == qxt_d().root) parentindex = QModelIndex();
	beginRemoveRows(parentindex, row, row);
	//if (cache->parent() != NULL) cache->parent()->children().removeAll(cache);
	deleteCache(cache);
	endRemoveRows();
}

void RdsEntityModel::entityRenamed(QString oldentity, QString newentity)
{
	//qDebug() << "Renaming:" << oldentity << newentity;
	if (!qxt_d().idmap.contains(oldentity)) return; //we don't have a record of it, so don't rename it!

	Cache *cache = qxt_d().idmap.value(oldentity);
	if (cache->parent() == NULL) return;
	QModelIndex parent = indexFromCache(cache);
	if (cache == qxt_d().root) parent = QModelIndex();

	if (cache->children().size() > 0 && !cache->hasMore())
	{
		beginRemoveRows(parent, 0, cache->children().size() - 1);

		foreach(Cache *child, cache->children())
		{
			deleteCache(child);
		}

		cache->children().clear();

		endRemoveRows();
		qxt_d().view->collapse(qxt_d().sort->mapFromSource(parent));
	}

	ReturnValue ret = manager()->listEntities(newentity);
	if (ret.isError())
	{
		qWarning() << "Failed to list entity" << newentity << ":" << ret.errString();
		return;
	}

	RdsEntity tmp = ret.value<RdsEntity>();
	qxt_d().idmap.remove(oldentity);
	qxt_d().idmap[newentity] = cache;

	if (tmp.children().size() > 0 && !cache->hasMore())
	{
		beginInsertRows(parent, 0, tmp.children().size() - 1);
		*cache = tmp;
		loadEntity(tmp, cache);
		endInsertRows();
	}
	else
	{
		*cache = tmp;
	}

	if (qxt_d().input == oldentity) qxt_d().input = newentity;
	dataChanged(indexFromCache(cache, 0), indexFromCache(cache, qxt_d().columns - 1));
}

void RdsEntityModel::entityUpdated(QString id)
{
	if (!qxt_d().idmap.contains(id)) return;

	ReturnValue ret = qxt_d().manager->listEntities(id);
	if (ret.isError())
	{
		qWarning() << "Failed to update entity:" << ret.errString();
		return;
	}

	RdsEntity entity = ret.value<RdsEntity>();
	Cache *cache = qxt_d().idmap.value(id);
	bool hasmore = cache->hasMore();
	*cache = entity;
	cache->setHasMore(hasmore);

	dataChanged(indexFromCache(cache, 0), indexFromCache(cache, qxt_d().columns - 1));

	emit(entityUpdateFinished(id));
}

void RdsEntityModel::entityMetadataUpdated(QString id, QVariantMap metadata)
{
	if (!qxt_d().idmap.contains(id)) return;
	
	Cache *cache = qxt_d().idmap.value(id);
	
	if(metadata.size() != 0)
	{
		foreach(QString key, metadata.keys())
		{
			cache->metadata()[key] = metadata[key];
		}
	}
	else
	{
		ReturnValue ret = qxt_d().manager->listEntities(id);
		if (ret.isError())
		{
			qWarning() << "Failed to update entity:" << ret.errString();
			return;
		}
		
		RdsEntity entity = ret.value<RdsEntity>();
		bool hasmore = cache->hasMore();
		*cache = entity;
		cache->setHasMore(hasmore);
	}
	
	dataChanged(indexFromCache(cache, 0), indexFromCache(cache, qxt_d().columns - 1));
	
	emit(entityMetadataUpdateFinished(id));
}


void RdsEntityModel::editingStarted(QString id)
{
	if (!qxt_d().idmap.contains(id)) return;

	Cache *cache = qxt_d().idmap.value(id);
	qxt_d().editlist << id;

	dataChanged(indexFromCache(cache, 0), indexFromCache(cache, 0));
}

void RdsEntityModel::editingStopped(QString id)
{
	if (!qxt_d().idmap.contains(id)) return;

	Cache *cache = qxt_d().idmap.value(id);
	qxt_d().editlist.removeAll(id);

	dataChanged(indexFromCache(cache, 0), indexFromCache(cache, qxt_d().columns - 1));
}


RdsEntityManager *RdsEntityModel::manager()
{
	return(qxt_d().manager);
}

void RdsEntityModel::deleteCache(Cache *cache)
{
	if (cache == NULL) return;
	if (cache->parent() != NULL) cache->parent()->children().removeAll(cache);

	foreach(Cache *child, cache->children())
	{
		deleteCache(child);
	}
	//invalidateIndexes(cache);
	qxt_d().idmap.remove(cache->id());
	qxt_d().loadingcaches.remove(qxt_d().loadingcaches.key(cache));
	qxt_d().loadingindexes.remove(cache);
	delete cache;
}

QModelIndex RdsEntityModel::indexFromCache(Cache *cache, int column)
{
	if (cache == NULL) return(QModelIndex());

	int row = 0;
	if (cache->parent() != NULL)
	{
		row = cache->parent()->children().indexOf(cache);
	}

	return(createIndex(row, column, cache));
}

bool RdsEntityModel::dropMimeData(const QMimeData * mime, Qt::DropAction action, int row, int column, const QModelIndex & parentindex)
{
	Q_UNUSED(action);
	Q_UNUSED(row);
	Q_UNUSED(column);
	Cache *parent = NULL;
	if (!parentindex.isValid())
		parent = qxt_d().root;
	else
		parent = (Cache *)parentindex.internalPointer();

	QByteArray data = mime->data("application/rds.entityid");
	QDataStream stream(&data, QIODevice::ReadOnly);
	QStringList ids;
	stream >> ids;

	qxt_d().selection->clear();
	emit(inputChanged("", ""));

	//qDebug() << "Dropping:" << ids;
	foreach(QString id, ids)
	{
		if (!qxt_d().idmap.contains(id)) continue;
		if (parent->id().startsWith(qxt_d().idmap.value(id)->id())) continue;
		move(qxt_d().idmap.value(id), parent);
	}

	return(true);
}

QMimeData * RdsEntityModel::mimeData(const QModelIndexList & indexes) const
{
	QStringList ids;
	QString text;
	foreach(QModelIndex index, indexes)
	{
		if (index.column() != 0) continue;
		Cache *cache = (Cache *)index.internalPointer();
		ids << cache->id();
		text += cache->id() + "\n";
	}

	QByteArray data;
	QDataStream stream(&data, QIODevice::WriteOnly);
	stream << ids;
	QMimeData *mime = new QMimeData();
	mime->setData("application/rds.entityid", data);
	mime->setText(text);
	return(mime);
}

QStringList RdsEntityModel::mimeTypes() const
{
	return(QStringList() << "application/rds.entityid");
}

void RdsEntityModel::move(Cache *entity, Cache *parent)
{
	Q_UNUSED(entity);
	Q_UNUSED(parent);
	return;
}

void RdsEntityModel::invalidateIndexes(Cache *cache)
{
	foreach(QModelIndex index, persistentIndexList())
	{
		if (index.internalPointer() == cache)
		{
			changePersistentIndex(index, QModelIndex());
		}
	}
}

void RdsEntityModel::setupSelection(RdsEntitySortModel *sort, QTreeView *view)
{
	QObject::connect(view->selectionModel(), SIGNAL(selectionChanged(const QItemSelection&, const QItemSelection&)),
	                 this, SLOT(outputsChanged()));
	qxt_d().sort = sort;
	qxt_d().selection = view->selectionModel();
	qxt_d().view = view;
}

void RdsEntityModel::outputsChanged()
{
	QStringList outputlist;
	foreach(QModelIndex index, qxt_d().selection->selectedRows())
	{
		RdsEntityModel::Cache *cache = (RdsEntityModel::Cache *)qxt_d().sort->mapToSource(index).internalPointer();
		if (cache == NULL) continue;
		if ((cache->type() != qxt_d().inputtype) && qxt_d().constrainselection) continue; //if its not the same type of entity, don't include it!

		outputlist << cache->id();
	}

	if (!outputlist.contains(qxt_d().input))
	{
		RdsEntityModel::Cache *cache = (RdsEntityModel::Cache *)qxt_d().sort->mapToSource(qxt_d().selection->currentIndex()).internalPointer();
		if (cache != NULL)
		{
			qxt_d().input = cache->id();
			qxt_d().inputtype = cache->type();
			emit(inputChanged(cache->id(), cache->type()));
		}
	}

	if (outputlist.size() == 0) outputlist << qxt_d().input;

	qxt_d().outputs = outputlist;
	emit(outputsChanged(outputlist));
}

QString RdsEntityModel::input()
{
	return(qxt_d().input);
}

QStringList RdsEntityModel::outputs()
{
	return(qxt_d().outputs);
}

void RdsEntityModel::setConstrainSelection(bool constrain)
{
	qxt_d().constrainselection = constrain;
}

bool RdsEntityModel::hasChildren(const QModelIndex &parent) const
{
	if (!parent.isValid()) return(true);
	Cache *cache = (Cache *)parent.internalPointer();
	if (cache == NULL) return(false);

	if (cache->children().size() > 0) return(true);
	if (cache->hasMore()) return(true);
	return(false);
}

bool RdsEntityModel::canFetchMore(const QModelIndex &parent) const
{
	if (!parent.isValid()) return(false);
	Cache *cache = (Cache *)parent.internalPointer();
	if (cache == NULL) return(false);

	return(cache->hasMore());
}

void RdsEntityModel::fetchMore(const QModelIndex &parent)
{
	if (!parent.isValid()) return;
	Cache *cache = (Cache *)parent.internalPointer();
	if (cache == NULL) return;

	if (qxt_d().loadingcaches.values().contains(cache)) return;

	cache->setHasMore(false);

	ReturnValue ret = qxt_d().manager->listEntities(this, SLOT(dataReceived(uint, ReturnValue)), cache->id(), true);
	if (ret.isError()) return;

	uint id = ret.toInt();
	qxt_d().loadingcaches[id] = cache;

	RdsEntity entity;
	entity.setId("loading-" + cache->id());
	entity.setName("Loading...");
	entity.setType("loading");
	entity.setVisible(true);
	entity.setParent(cache->id());

	Cache *child = new Cache(entity, cache);
	qxt_d().idmap[child->id()] = child;

	qxt_d().loadingindexes[cache] = child;
	beginInsertRows(parent, 0, 0);
	cache->children() << child;
	endInsertRows();
}

void RdsEntityModel::dataReceived(uint id, ReturnValue ret)
{
	if (!qxt_d().loadingcaches.contains(id)) return;
	Cache *cache = qxt_d().loadingcaches[id];
	Cache *loadingcache = qxt_d().loadingindexes[cache];

	qxt_d().loadingcaches.remove(id);
	qxt_d().loadingindexes.remove(cache);

	//if the entity was renamed, don't bother loading
	if (!qxt_d().idmap.contains(cache->id())) return;

	if (ret.isError())
	{
		qWarning() << "Failed to fetch data:" << ret.errString();
		return;
	}

	QModelIndex parent = indexFromCache(cache);;
	//if (cache->parent() != NULL) parent = indexFromCache(cache->parent());

	RdsEntity tmp = ret.value<RdsEntity>();

	if (tmp.children().size() > 0)
	{
		beginInsertRows(parent, 1, tmp.children().size());
		*cache = tmp;
		loadEntity(tmp, cache);
		endInsertRows();


		if (loadingcache != NULL)
		{
			QMetaObject::invokeMethod(this, "entityRemoved", Qt::QueuedConnection, Q_ARG(QString, loadingcache->id()), Q_ARG(bool, false));
		}
	}
	else
	{
		*cache = tmp;
		dataChanged(parent, parent);


		if (loadingcache != NULL)
		{
			QMetaObject::invokeMethod(this, "entityRemoved", Qt::QueuedConnection, Q_ARG(QString, loadingcache->id()), Q_ARG(bool, false));
		}
	}
}

void RdsEntityModel::setCachedPixmap(QString name, const QPixmap &pixmap)
{
	qxt_d().pixmaps[name] = pixmap;
}

QPixmap RdsEntityModel::cachedPixmap(QString name) const
{
	if (qxt_d().pixmaps.contains(name))
		return(qxt_d().pixmaps[name]);
	else
		return(QPixmap());
}

void RdsEntityModel::onTimerUpdate()
{
	if (qxt_d().loadingframe == 11) qxt_d().loadingframe = 0;
	else qxt_d().loadingframe++;

	foreach(Cache *cache, qxt_d().loadingindexes)
	{
		QModelIndex index = indexFromCache(cache);
		dataChanged(index, index);
	}
}

void RdsEntityModel::expandEntity(QString entity)
{
	//check if the entity is already loaded
	if (qxt_d().idmap.contains(entity))
	{
		finishExpanding(entity);
	}
	else //if not load it
	{
		QList<int> keys = qxt_d().entitymap.keys();
		int id = 0;
		while (true)
		{
			if (!keys.contains(id)) break;
			id++;
		}

		qxt_d().entitymap[id] = entity;
		QMetaObject::invokeMethod(this, "loadEntity(uint, ReturnValue)", Qt::QueuedConnection, Q_ARG(uint, id), Q_ARG(ReturnValue, ReturnValue(QVariant())));
	}
}

void RdsEntityModel::loadEntity(uint id, ReturnValue ret)
{
	QString entity;
	int entid;

	if (ret.isError())
	{
		qWarning() << "Failed to fetch entity:" << ret.toString();
		return;
	}

	if (ret.isNull())
	{
		entity = qxt_d().entitymap[id];
		entid = id;
	}
	else
	{
		if (qxt_d().asyncmap.contains(id))
		{
			entid = qxt_d().asyncmap[id];
			qxt_d().asyncmap.remove(id);

			RdsEntity ent = ret.value<RdsEntity>();
			qxt_d().entitylist.prepend(ent);

			entity = ent.parent();
		}
		else
		{
			qWarning() << "Failed to find async ID";
			return;
		}
	}

	ret = manager()->listEntities(this, SLOT(loadEntity(uint, ReturnValue)), entity, false);
	if (ret.isError())
	{
		qWarning() << "Async call failed:" << ret.toString();
		return;
	}

	uint newid = ret.toInt();
	qxt_d().asyncmap[newid] = entid;
}

void RdsEntityModel::finishExpanding(QString entity)
{
	QModelIndex index = qxt_d().sort->mapFromSource(indexFromCache(qxt_d().idmap.value(entity)));

	while (index.isValid())
	{
		qxt_d().view->setExpanded(index, true);
		index = qxt_d().sort->parent(index);
	}

	//get the final index again
	index = qxt_d().sort->mapFromSource(indexFromCache(qxt_d().idmap.value(entity)));

	qxt_d().view->selectionModel()->clear();
	qxt_d().view->selectionModel()->setCurrentIndex(index, QItemSelectionModel::Select);
	qxt_d().view->selectionModel()->select(index, QItemSelectionModel::Select);
}

void RdsEntityModel::reload()
{
	beginResetModel();
	deleteCache(qxt_d().root);

	RdsEntity ent;
	ent.setId("loading");
	ent.setName("Loading...");
	ent.setType("loading");
	ent.setVisible(true);
	qxt_d().root = new Cache(ent);

	ReturnValue ret = qxt_d().manager->listEntities(this, SLOT(entitiesReturned(uint, ReturnValue)), "");
	if (ret.isError()) entitiesReturned(0, ret);

	endResetModel();
}



