/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdsdhcpmodel.h"
#include <QMessageBox>
#include <QDebug>
#include <RdsShare>
#include <RdsDhcpValues>
#include <RdsDaemonManager>
#include "config.h"

RdsDhcpModel::RdsDhcpModel(RdsDhcpManager *manager, QObject *parent)
		: RdsEntityModel(manager, 2, parent)
{
	_manager = manager;
	setCachedPixmap("group", QPixmap(findRdsIcon("./icons/16x16/folder.png")));
	setCachedPixmap("sharednetwork", QPixmap(findRdsIcon("./icons/16x16/folder_red.png")));
	setCachedPixmap("subnet", QPixmap(findRdsIcon("./icons/16x16/network.png")));
	setCachedPixmap("host", QPixmap(findRdsIcon("./icons/16x16/network_local.png")));
	setCachedPixmap("manager", QPixmap(findRdsIcon("./icons/16x16/server.png")));
}


RdsDhcpModel::~RdsDhcpModel()
{

}

QVariant RdsDhcpModel::userData(const QModelIndex &index, int role, RdsEntityModel::Cache *cache) const
{
	Q_UNUSED(index);

	
	if ((role == Qt::DecorationRole) && (index.column() == 0))
	{
		return(cachedPixmap(cache->type()));
	}
	if ((role == Qt::DisplayRole) && (index.column() == 1))
	{
		if(cache->type() == "manager") return("");
		if(cache->type() == "group") return("Group");
		if(cache->type() == "sharednetwork") return("Shared Network");
		if(cache->type() == "subnet") return("Subnet");
		if(cache->type() == "host") return("Host");
	}

	return(QVariant());
}

QVariant RdsDhcpModel::userHeaderData(int section, int role) const
{
	if (role == Qt::DisplayRole)
	{
		if (section == 0)
		{
			return("Name");
		}
		else 	if (section == 1)
		{
			return("Type");
		}
	}

	return(QVariant());
}

Qt::ItemFlags RdsDhcpModel::flags(const QModelIndex &index, Cache *cache) const
{
	Q_UNUSED(index);
	
	if(cache->type() == "manager") return(0);
	else if(index.column() == 0) return(Qt::ItemIsEditable);
	else return(0);
}

bool RdsDhcpModel::setData(const QModelIndex &index, const QVariant &value, int role, Cache *cache)
{
	Q_UNUSED(index);
	if (cache->name() == value.toString()) return(false); //if the new name is the same, we didn't change anything
	if (value.toString() == "") return(false);
	if(role != Qt::EditRole) return(false);
	
	if(cache->type() == "manager") return(false);
	else
	{
		if(cache->type() == "host")
		{
			if(!QRegExp("^[A-Za-z][A-Za-z0-9\\-]*$").exactMatch(value.toString()))
			{
				QMessageBox::warning(NULL, "Error", "The host name must start with a letter and contain only letters, numbers, and the - character.");
				return(false);
			}
		}
		else if(value.toString().contains('/'))
		{
			QMessageBox::critical(NULL, "Error", "The / character is not allowd.");
			return(false);
		}
		
		ReturnValue ret = _manager->values(cache->id());
		if(ret.isError())
		{
			QMessageBox::critical(NULL, "Error", "Failed to get values object: " + ret.errString());
			return(false);
		}
		
		RdsDhcpValues values;
		values = ret;
		
		ret = values.setName(value.toString());
		if(ret.isError())
		{
			QMessageBox::critical(NULL, "Error", "Failed to set name: " + ret.errString());
			return(false);
		}
		
		int index = cache->id().lastIndexOf("/");
		QString tmp = cache->id().left(index + 2);
		tmp += value.toString();
		
		_manager->renameEntity(cache->id(), tmp);
		
		save();
		
		return(true);
	}

	return(false);
}

void RdsDhcpModel::move(Cache *entity, Cache *parent)
{
	if(entity->type() == "manager") return;
	if(parent->type() == "host") return;
	if(parent->id() == "root") return;

	
	QString newid = parent->id() + entity->id().mid(entity->id().lastIndexOf('/'));
	
	if((entity->type() == "subnet") && (parent->type() == "subnet"))
	{
		QMessageBox::critical(NULL, "Error", "You may not move a subnet into another subnet.");
		return;
	}
	
	ReturnValue ret = _manager->move(entity->id(), newid);
	if(ret.isError())
	{
		QMessageBox::critical(NULL, "Error", "Failed to move: " + ret.errString());
		return;
	}
	
	_manager->removeEntity(entity->id());
	_manager->addEntity(newid);
	
	save();
	
	return;
}

bool RdsDhcpModel::save()
{
	ReturnValue ret = _manager->save();
	if (ret.isError())
	{
		QMessageBox::critical(NULL, "Error", "Failed to save changes: " + ret.errString());
		return(false);
	}
	
	RdsDaemonManager mgr;
	ret = mgr.init();
	
	if (ret.isError())
	{
		QMessageBox::critical(NULL, "Error", "Failed to get RdsDaemonManager service: " + ret.errString());
		return(false);
	}
	
	ret = mgr.restartService("Dhcp");
	
	if (ret.isError())
	{
		QMessageBox::critical(NULL, "Error", "Failed to restart the DHCP server: " + ret.errString());
		return(false);
	}
	
	return(true);
}

