/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/


#include "permissionswidget.h"
#include <QDebug>
#include <QMessageBox>
#include <RdsAdObject>
#include <RdsUtils>
#include <QDir>
#include <RdsFileManager>
#include "selectiondialog.h"
#include "aceflagswidget.h"
#include "config.h"

PermissionsWidget::PermissionsWidget(QWidget* parent, Qt::WFlags fl)
		: QWidget(parent, fl), Ui::PermissionsWidget()
{
	setupUi(this);
	AddAce->setIcon(QPixmap(findRdsIcon("icons/32x32/add.png")));
	RemoveAce->setIcon(QPixmap(findRdsIcon("icons/32x32/remove.png")));

	label->hide();
	OwnerLabel->hide();
	OwnerButton->hide();
	InheritCheck->hide();
	ForceInheritCheck->hide();

	AclView->setHeaderHidden(true);
	AclView->header()->setStretchLastSection(true);
	_aclmodel = new AclModel(this);
	_aclsort = new QSortFilterProxyModel(this);
	_aclsort->setSourceModel(_aclmodel);
	AclView->setModel(_aclsort);

	_acemodel = new AceModel(this);
	_acemodel->setAdvanced(false);
	AceView->setModel(_acemodel);

	QObject::connect(_acemodel, SIGNAL(flagsChanged(RdsAce::AccessFlags, RdsAce::AccessFlags, RdsAce::Flags, RdsAce::Flags)),
	                 this, SLOT(flagsChanged(RdsAce::AccessFlags, RdsAce::AccessFlags, RdsAce::Flags, RdsAce::Flags)));
	QObject::connect(AclView->selectionModel(), SIGNAL(currentChanged(QModelIndex, QModelIndex)),
	                 this, SLOT(aclClicked(QModelIndex)));

	aclClicked(QModelIndex());
}

PermissionsWidget::~PermissionsWidget()
{
}

void PermissionsWidget::setPath(const QString &path)
{
	_path = path;
}

RdsNtAcl PermissionsWidget::acl() const
{
	return(_acl);
}

void PermissionsWidget::setAcl(const RdsNtAcl &acl)
{
	_acl = acl;
	reload();
}

bool PermissionsWidget::forceInherit()
{
	return(ForceInheritCheck->isChecked());
}

void PermissionsWidget::reload()
{
	_aclmodel->setAcl(&_acl);
	displayOwner(_acl.owner());
	InheritCheck->setChecked(((_acl.daclFlags() & RdsNtAcl::Protected) == 0));
	_aclsort->sort(0);
	aclClicked(QModelIndex());
}

void PermissionsWidget::on_AdvancedButton_toggled(bool checked)
{
	AclView->header()->setStretchLastSection(!checked);
	if (checked) AclView->header()->resizeSection(0, 150);
	AclView->setHeaderHidden(!checked);
	_aclmodel->setAdvanced(checked);
	_acemodel->setAdvanced(checked);
	AceView->setRootIsDecorated(false);
	AceView->expandAll();
	aclClicked(QModelIndex());
	if(checked)
	{
		AdvancedButton->setText("&Simple");
	}
	else
	{
		AdvancedButton->setText("Ad&vanced");
	}
}

void PermissionsWidget::on_AddAce_clicked()
{
	SelectionDialog dialog(SelectionDialog::User | SelectionDialog::Group | SelectionDialog::Special , this);
	AceFlagsWidget widget(&dialog);
	dialog.AddWidget(&widget);

	if (!AdvancedButton->isChecked())
		widget.hide();

	if (!dialog.exec())
	{
		return;
	}

	RdsAdObject object(dialog.selectedItem());
	ReturnValue ret = object.sid();

	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to look up SID.");
		return;
	}

	RdsSid sid = ret.value<RdsSid>();

	AclModel::Acl acl;
	acl.sid = sid;
	acl.inherited = false;
	acl.flags = RdsAce::FileInherit | RdsAce::FolderInherit;
	if (AdvancedButton->isChecked()) acl.flags = widget.flags();
	acl.allow = true;
	acl.type = AclModel::Unknown;

	ret = object.readAttribute("cn");
	LdapValues attr = ret.value<LdapValues>();
	if (attr.size() > 0)
	{
		acl.name = attr[0];

		ret = object.readAttribute("objectClass");
		if (!ret.isError())
		{
			LdapValues attr = ret.value<LdapValues>();
			//qDebug() << acl.name << attr;
			if (attr.contains(QString("user").toAscii()))
			{
				acl.type = AclModel::User;
				//qDebug() << "User:" << acl.name;
			}
			else if (attr.contains(QString("group").toAscii()))
			{
				acl.type = AclModel::Group;
				//qDebug() << "Group:" << acl.name;
			}
			else if (attr.contains(QString("foreignSecurityPrincipal").toAscii()))
			{
				acl.type = AclModel::Group;
				//qDebug() << "Group:" << acl.name;
			}
			else
			{
				acl.type = AclModel::Group;
			}
		}
	}

	bool inherited = false;
	foreach(RdsAce ace, _acl.dacl())
	{
		if ((ace.sid() == sid) && (ace.flags() == (RdsAce::FileInherit | RdsAce::FolderInherit)))
		{
			QMessageBox::critical(this, "Error", "There is already a permission for " + acl.name + ".");
			selectAcl(sid, acl.flags);
			return;
		}
		else if ((ace.sid() == sid) && (ace.flags() == (RdsAce::FileInherit | RdsAce::FolderInherit | RdsAce::Inherited)))
		{
			inherited = true;
		}
	}

	//if there is an inherited permissions for this user, don't add it to the list
	if (!inherited)
	{
		_aclmodel->addAcl(acl);
		_aclsort->sort(0);
	}
	writePermissions(RdsAce::CreateChild | RdsAce::ListObject |  RdsAce::SelfWrite | RdsAce::ReadControl, 0, sid,
	                 0, acl.flags);


	reload();
	selectAcl(sid, acl.flags);
}

void PermissionsWidget::on_RemoveAce_clicked()
{
	foreach(QModelIndex index, AclView->selectionModel()->selectedRows())
	{
		AclModel::Acl acl = _aclmodel->getAclFromIndex(_aclsort->mapToSource(index));
		if (acl.sid == RdsSid()) return;

		if (AdvancedButton->isChecked())
		{
			if ((acl.primaryace.flags() & RdsAce::Inherited) == RdsAce::Inherited)
			{
				QMessageBox::warning(this, "Error", "The permission you have selected is inherited, and cannot be removed.");
				continue;
			}

			if (!_acl.dacl().removeAll(acl.primaryace)) return;
		}
		else
		{
			bool found = false;
			foreach(RdsAce ace, _acl.dacl())
			{
				if ((ace.sid() == acl.sid) && ((ace.flags() & RdsAce::Inherited) != RdsAce::Inherited))
				{
					found = true;
					_acl.dacl().removeAll(ace);
				}
			}

			if (!found)
			{
				QMessageBox::warning(this, "Error", "The permission you have selected is inherited, and cannot be removed.");
				continue;
			}

			foreach(RdsAce ace, _acl.dacl())
			{
				if ((ace.sid() == acl.sid) && ((ace.flags() & RdsAce::Inherited) == RdsAce::Inherited))
				{
					QMessageBox::warning(this, "Error", "The permission you have removed is also inherited. It will continue to be shown on the list");
					continue;
				}
			}
		}

		_aclmodel->removeAcl(acl);
	}
	changed();
	_aclsort->sort(0);
}

void PermissionsWidget::displayOwner(RdsSid sid)
{
	ReturnValue ret = RdsUtils::getObjectBySid(sid);
	if (!ret.isError())
	{
		RdsAdObject object(ret.toString());
		ret = object.readAttribute("cn");
		if (!ret.isError())
		{
			LdapValues attr = ret.value<LdapValues>();
			if (attr.size() > 0)
			{
				OwnerLabel->setText(attr[0]);
				return;
			}
		}
	}

	OwnerLabel->setText(sid.toString());
}

void PermissionsWidget::on_OwnerButton_clicked()
{
	SelectionDialog dialog(SelectionDialog::User, this);
	if (!dialog.exec())
	{
		return;
	}

	RdsAdObject object(dialog.selectedItem());
	ReturnValue ret = object.sid();

	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to look up SID.");
		return;
	}

	RdsSid sid = ret.value<RdsSid>();
	_acl.setOwner(sid);
	displayOwner(sid);
	changed();
}

void PermissionsWidget::on_InheritCheck_clicked(bool inherit)
{
	if (inherit)
	{
		ReturnValue ret = RdsUtils::getSystemPath(_path);
		if (ret.isError())
		{
			QMessageBox::critical(this, "Error", "Failed to get path: " + ret.errString());
			InheritCheck->setChecked(false);
			return;
		}

		QDir dir(ret.toString());
		dir.cdUp();

		RdsFileManager manager;
		ret = manager.init();
		if (ret.isError())
		{
			QMessageBox::critical(this, "Error", "Failed to get file manager: " + ret.errString());
			InheritCheck->setChecked(false);
			return;
		}

		ret = manager.ntPermissions(dir.path());
		if (ret.isError())
		{
			QMessageBox::critical(this, "Error", "Failed to get parent permissions: " + ret.errString());
			InheritCheck->setChecked(false);
			return;
		}
		RdsNtAcl parentperm = ret.value<RdsNtAcl>();

		ret = manager.isFile(_path);
		if (ret.isError())
		{
			QMessageBox::critical(this, "Error", "Failed to get file type: " + ret.errString());
			InheritCheck->setChecked(false);
			return;
		}

		bool isfile = ret.toBool();

		if (isfile)
		{
			//Add ACEs from the parent that have the FileInherit flag
			foreach(RdsAce ace, parentperm.dacl())
			{
				if ((ace.flags() & RdsAce::FileInherit) == RdsAce::FileInherit)
				{
					//qDebug() << "Adding file ACE:" << ace;
					ace.setFlags(ace.flags() | RdsAce::Inherited);
					_acl.dacl() << ace;
				}
			}
		}
		else
		{
			//Add ACEs from the parent that have the FolderInherit flag
			foreach(RdsAce ace, parentperm.dacl())
			{
				if ((ace.flags() & RdsAce::FolderInherit) == RdsAce::FolderInherit)
				{
					//qDebug() << "Adding folder ACE:" << ace;
					ace.setFlags(ace.flags() | RdsAce::Inherited);
					_acl.dacl() << ace;
				}
			}
		}

		_acl.setDaclFlags(0);
		reload();
		changed();
	}
	else
	{
		bool found = false;

		//search for inherited permissions
		foreach(RdsAce ace, _acl.dacl())
		{
			if ((ace.flags() & RdsAce::Inherited) != 0) found = true;
		}

		//If we found some inherited permissinos we should ask if we should copy or remove them
		if (found)
		{
			QMessageBox box(QMessageBox::Question, "Question",
			                "Do you wish to copy or remove permissions inherited from the parent folder?"
			                , QMessageBox::NoButton, this, 0);

			QAbstractButton *copy = box.addButton("Copy", QMessageBox::AcceptRole);
			box.addButton("Remove", QMessageBox::AcceptRole);
			box.addButton(QMessageBox::Cancel);
			box.setDefaultButton(QMessageBox::Cancel);

			if (box.exec() == QMessageBox::Cancel)
			{
				InheritCheck->setChecked(true); //reset check value
				return;
			}
			bool docopy = (box.clickedButton() == copy);

			int i = 0; //no for loop because this program is non-euclidean
			foreach(RdsAce ace, _acl.dacl())
			{
				if ((ace.flags() & RdsAce::Inherited) != 0)
				{
					if (docopy)
					{
						//just rmove the inherited flag
						_acl.dacl()[i].setFlags(ace.flags() & (~RdsAce::Inherited));
					}
					else
					{
						//remove it
						_acl.dacl().removeAll(ace);
					}
				}
				i++; //really, its not a for loop
			}
		}

		_acl.setDaclFlags(RdsNtAcl::Protected);
		reload();
		changed();
	}
}

void PermissionsWidget::aclClicked(const QModelIndex& index)
{
	if (index.isValid())
	{
		AceView->setEnabled(true);
	}
	else
	{
		AceView->setEnabled(false);
		return;
	}

	AclModel::Acl tmp = _aclmodel->getAclFromIndex(_aclsort->mapToSource(index));
	MergedPermissions perms;

	if (AdvancedButton->isChecked())
	{
		perms = readAdvancedPermissions(tmp.sid, tmp.flags, tmp.inherited);
	}
	else
	{
		perms = readSimplePermissions(tmp.sid);
	}

	//qDebug() << "perms" << perms.allow << perms.deny << perms.inheritedAllow << perms.inheritedDeny;

	RdsAce::Flags flags = tmp.flags;
	if (!AdvancedButton->isChecked()) flags = RdsAce::FileInherit | RdsAce::FolderInherit;

	_acemodel->setFlags(perms.allow, perms.deny, perms.inheritedAllow, perms.inheritedDeny, perms.specialAllow, perms.specialDeny,  flags);
	if (AdvancedButton->isChecked())
	{
		AceView->setRootIsDecorated(false);
		AceView->expandAll();
	}
}

PermissionsWidget::MergedPermissions::MergedPermissions()
		: specialAllow(false),
		specialDeny(false)
{
}

PermissionsWidget::MergedPermissions PermissionsWidget::readSimplePermissions(const RdsSid &sid)
{
	PermissionsWidget::MergedPermissions mergedPermissions;
	PermissionsWidget::MergedPermissions thisFolder;
	PermissionsWidget::MergedPermissions subFolders;
	PermissionsWidget::MergedPermissions subFiles;

	foreach(RdsAce ace, _acl.dacl())
	{
		if (ace.sid() != sid)
			continue;

		if (ace.flags().testFlag(RdsAce::Inherited))
		{
			if (ace.type() == RdsAce::Allow)
			{
				mergedPermissions.inheritedAllow |= ace.access();
				if (ace.flags().testFlag(RdsAce::FileInherit))
				{
					subFiles.inheritedAllow |= ace.access();
				}
				if (ace.flags().testFlag(RdsAce::FolderInherit))
				{
					subFolders.inheritedAllow |= ace.access();
				}
				if (!ace.flags().testFlag(RdsAce::InheritOnly))
				{
					thisFolder.inheritedAllow |= ace.access();
				}
			}
			if (ace.type() == RdsAce::Deny)
			{
				mergedPermissions.inheritedDeny |= ace.access();
				if (ace.flags().testFlag(RdsAce::FileInherit))
				{
					subFiles.inheritedDeny |= ace.access();
				}
				if (ace.flags().testFlag(RdsAce::FolderInherit))
				{
					subFolders.inheritedDeny |= ace.access();
				}
				if (!ace.flags().testFlag(RdsAce::InheritOnly))
				{
					thisFolder.inheritedDeny |= ace.access();
				}
			}
		}
		else
		{
			if (ace.type() == RdsAce::Allow)
			{
				mergedPermissions.allow |= ace.access();
				if (ace.flags().testFlag(RdsAce::FileInherit))
				{
					subFiles.allow |= ace.access();
				}
				if (ace.flags().testFlag(RdsAce::FolderInherit))
				{
					subFolders.allow |= ace.access();
				}
				if (!ace.flags().testFlag(RdsAce::InheritOnly))
				{
					thisFolder.allow |= ace.access();
				}
			}
			if (ace.type() == RdsAce::Deny)
			{
				mergedPermissions.deny |= ace.access();
				if (ace.flags().testFlag(RdsAce::FileInherit))
				{
					subFiles.deny |= ace.access();
				}
				if (ace.flags().testFlag(RdsAce::FolderInherit))
				{
					subFolders.deny |= ace.access();
				}
				if (!ace.flags().testFlag(RdsAce::InheritOnly))
				{
					thisFolder.deny |= ace.access();
				}
			}
		}
	}

	PermissionsWidget::MergedPermissions ret;

	ret.allow = thisFolder.allow & subFolders.allow & subFiles.allow;
	ret.deny = thisFolder.deny & subFolders.deny & subFiles.deny;
	ret.inheritedAllow = thisFolder.inheritedAllow & subFolders.inheritedAllow & subFiles.inheritedAllow;
	ret.inheritedDeny = thisFolder.inheritedDeny & subFolders.inheritedDeny & subFiles.inheritedDeny;

	int fakeExecute = (1 << 25);

	if (ret.allow.testFlag(RdsAce::ExecuteFile))
		ret.allow |= (RdsAce::AccessFlag)fakeExecute;
	else if (thisFolder.allow.testFlag(RdsAce::ExecuteFile) && subFolders.allow.testFlag(RdsAce::ExecuteFile))
		ret.allow |= RdsAce::ExecuteFile;

	if (ret.deny.testFlag(RdsAce::ExecuteFile))
		ret.deny |= (RdsAce::AccessFlag)fakeExecute;
	else if (thisFolder.deny.testFlag(RdsAce::ExecuteFile) && subFolders.deny.testFlag(RdsAce::ExecuteFile))
		ret.deny |= RdsAce::ExecuteFile;

	if (ret.inheritedAllow.testFlag(RdsAce::ExecuteFile))
		ret.inheritedAllow |= (RdsAce::AccessFlag)fakeExecute;
	else if (thisFolder.inheritedAllow.testFlag(RdsAce::ExecuteFile) && subFolders.inheritedAllow.testFlag(RdsAce::ExecuteFile))
		ret.inheritedAllow |= RdsAce::ExecuteFile;

	if (ret.inheritedDeny.testFlag(RdsAce::ExecuteFile))
		ret.inheritedDeny |= (RdsAce::AccessFlag)fakeExecute;
	else if (thisFolder.inheritedDeny.testFlag(RdsAce::ExecuteFile) && subFolders.inheritedDeny.testFlag(RdsAce::ExecuteFile))
		ret.inheritedDeny |= RdsAce::ExecuteFile;

	mergedPermissions.allow &= (~ret.allow);
	mergedPermissions.deny &= (~ret.deny);
	mergedPermissions.inheritedAllow &= (~ret.inheritedAllow);
	mergedPermissions.inheritedDeny &= (~ret.inheritedDeny);

	if (mergedPermissions.allow != 0 || mergedPermissions.inheritedAllow != 0)
		ret.specialAllow = true;

	if (mergedPermissions.deny != 0 || mergedPermissions.inheritedDeny != 0)
		ret.specialDeny = true;

	return ret;
}

PermissionsWidget::MergedPermissions PermissionsWidget::readAdvancedPermissions(const RdsSid &sid, const RdsAce::Flags &inheritence, bool inherited)
{
	MergedPermissions ret;
	foreach(RdsAce ace, _acl.dacl())
	{
		if (ace.sid() != sid)
			continue;
		if (inherited != ace.flags().testFlag(RdsAce::Inherited))
			continue;
		RdsAce::Flags fullInheritence = (
		                                    RdsAce::InheritOnly |
		                                    RdsAce::NoPropogate |
		                                    RdsAce::FileInherit |
		                                    RdsAce::ContainerInherit |
		                                    RdsAce::FolderInherit);

		if (((ace.flags() & fullInheritence) != inheritence))
		{
			continue;
		}
		if (ace.type() == RdsAce::Allow)
		{
			ret.allow |= ace.access();
		}
		else if (ace.type() == RdsAce::Deny)
		{
			ret.deny |= ace.access();
		}
	}
	return ret;
}

void PermissionsWidget::flagsChanged(RdsAce::AccessFlags allow, RdsAce::AccessFlags deny, RdsAce::Flags flags, RdsAce::Flags lastflags)
{
	QModelIndex index = _aclsort->mapToSource(AclView->selectionModel()->currentIndex());
	if (!index.isValid()) return;

	const int Execute = 1 << 25;
	bool execspecial = false;
	AclModel::Acl tmp = _aclmodel->getAclFromIndex(index);

	if (!AdvancedButton->isChecked())
	{
		if (((allow & Execute) == 0) && ((allow & RdsAce::WriteProperty) != 0))
		{
			execspecial = true;
		}

		if (((deny & Execute) == 0) && ((deny & RdsAce::WriteProperty) != 0))
		{
			execspecial = true;
		}
	}

	allow &= ~Execute;
	deny &= ~Execute;

	if (execspecial)
	{
		if (!writePermissions(allow & (~ RdsAce::WriteProperty), deny & (~ RdsAce::WriteProperty),
		                      tmp.sid, lastflags, flags & RdsAce::FileInherit))
		{
			qCritical() << "Failed to set flags while setting special case for executable, file inherit.";
		}
		if (!writePermissions(allow, deny, tmp.sid, lastflags, flags & (~RdsAce::FileInherit)))
		{
			qCritical() << "Failed to set flags while setting special case for executable, NOT file inherit.";
		}
	}
	else
	{
		if (!writePermissions(allow, deny, tmp.sid, lastflags, flags))
		{
			qCritical() << "Failed to set flags!";
		}
	}
}

bool PermissionsWidget::writePermissions(const RdsAce::AccessFlags &in_allow, const RdsAce::AccessFlags &in_deny, const RdsSid &sid, const RdsAce::Flags &old_inheritence, const RdsAce::Flags &new_inheritence)
{
	RdsAce::AccessFlags allow = in_allow;
	RdsAce::AccessFlags deny = in_deny;

	bool wroteAllow = false;
	bool wroteDeny = false;

// 	qDebug();
// 	qDebug() << "I am applying these:";
// 	qDebug() << "Allow:" << QString::number(allow, 16);
// 	qDebug() << "deny:" << QString::number(deny, 16);
// 	qDebug() << "These are my old inheritence values:";
// 	qDebug() << "Subfiles:" << (old_inheritence.testFlag(RdsAce::FileInherit));
// 	qDebug() << "Subfolders:" << (old_inheritence.testFlag(RdsAce::FolderInherit));
// 	qDebug() << "This Folder:" << (!old_inheritence.testFlag(RdsAce::NoPropogate));
// 	qDebug() << "These are my new inheritence values:";
// 	qDebug() << "Subfiles:" << (new_inheritence.testFlag(RdsAce::FileInherit));
// 	qDebug() << "Subfolders:" << (new_inheritence.testFlag(RdsAce::FolderInherit));
// 	qDebug() << "This Folder:" << (!new_inheritence.testFlag(RdsAce::NoPropogate));

	QList<RdsAce> dacls = _acl.dacl();
	for (QList<RdsAce>::iterator i = dacls.begin(); i != dacls.end(); ++i)
	{
		if (i->sid() != sid)
			continue;
		if (i->flags().testFlag(RdsAce::Inherited))
			continue;
		if (old_inheritence != i->flags())
			continue;
		if (i->type() == RdsAce::Allow && !wroteAllow)
		{
			if (allow == 0)
			{
				i = dacls.erase(i);
				i--;
				continue;
			}
			else
			{
				i->setAccess(allow);
				i->setFlags(new_inheritence);
			}
			wroteAllow = true;
		}
		else if (i->type() == RdsAce::Deny && !wroteDeny)
		{
			if (deny == 0)
			{
				i = dacls.erase(i);
				i--;
				continue;
			}
			else
			{
				i->setAccess(deny);
				i->setFlags(new_inheritence);
			}
			wroteDeny = true;
		}
	}
	if (!wroteAllow && allow != 0)
	{
		RdsAce newace;
		newace.setSid(sid);
		newace.setAccess(allow);
		newace.setType(RdsAce::Allow);
		newace.setFlags(new_inheritence);
		dacls.append(newace);
	}
	if (!wroteDeny && deny != 0)
	{
		RdsAce newace;
		newace.setSid(sid);
		newace.setAccess(deny);
		newace.setType(RdsAce::Deny);
		newace.setFlags(new_inheritence);
		dacls.append(newace);
	}
	PermissionsWidget::MergedPermissions mergedPermissions;
	PermissionsWidget::MergedPermissions thisFolder;
	PermissionsWidget::MergedPermissions subFolders;
	PermissionsWidget::MergedPermissions subFiles;

	for (QList<RdsAce>::iterator i = dacls.begin(); i != dacls.end();)
	{
		if (i->flags().testFlag(RdsAce::Inherited) || i->sid() != sid)
		{
			++i;
			continue;
		}

		if (i->type() == RdsAce::Allow)
		{
			mergedPermissions.allow |= i->access();
			if (i->flags().testFlag(RdsAce::FileInherit))
			{
				subFiles.allow |= i->access();
			}
			if (i->flags().testFlag(RdsAce::FolderInherit))
			{
				subFolders.allow |= i->access();
			}
			if (!i->flags().testFlag(RdsAce::InheritOnly))
			{
				thisFolder.allow |= i->access();
			}
		}
		if (i->type() == RdsAce::Deny)
		{
			mergedPermissions.deny |= i->access();
			if (i->flags().testFlag(RdsAce::FileInherit))
			{
				subFiles.deny |= i->access();
			}
			if (i->flags().testFlag(RdsAce::FolderInherit))
			{
				subFolders.deny |= i->access();
			}
			if (!i->flags().testFlag(RdsAce::InheritOnly))
			{
				thisFolder.deny |= i->access();
			}
		}
		i = dacls.erase(i);
	}

	// All permissions
	PermissionsWidget::MergedPermissions allPerms;
	allPerms.allow = thisFolder.allow & subFolders.allow & subFiles.allow;
	allPerms.deny = thisFolder.deny & subFolders.deny & subFiles.deny;

	// This folder & subfolders
	PermissionsWidget::MergedPermissions folderSubfolder;
	folderSubfolder.allow = thisFolder.allow & subFolders.allow & (~allPerms.allow);
	folderSubfolder.deny = thisFolder.deny & subFolders.deny & (~allPerms.deny);

	// this folder & subfiles
	PermissionsWidget::MergedPermissions folderSubfiles;
	folderSubfiles.allow = thisFolder.allow & subFiles.allow & (~allPerms.allow);
	folderSubfiles.deny = thisFolder.deny & subFiles.deny & (~allPerms.deny);

	// subfolder & subfiles
	PermissionsWidget::MergedPermissions subfilesSubfolder;
	subfilesSubfolder.allow = subFolders.allow & subFiles.allow & (~allPerms.allow);
	subfilesSubfolder.deny = subFolders.deny & subFiles.deny & (~allPerms.deny);

	thisFolder.allow &= (~(allPerms.allow | folderSubfolder.allow | folderSubfiles.allow));
	thisFolder.deny &= (~(allPerms.deny | folderSubfolder.deny | folderSubfiles.deny));

	subFolders.allow &= (~(allPerms.allow | folderSubfolder.allow | subfilesSubfolder.allow));
	subFolders.deny &= (~(allPerms.deny | folderSubfolder.deny | subfilesSubfolder.deny));

	subFiles.allow &= (~(allPerms.allow | subfilesSubfolder.allow | folderSubfiles.allow));
	subFiles.deny &= (~(allPerms.deny | subfilesSubfolder.deny | folderSubfiles.deny));

#define WRITE_OUT_PERMISSIONS(permType, inheritFlags) \
	if (permType.allow != 0 && permType.allow != 0x100000) \
	{ \
		RdsAce newace; \
		newace.setSid(sid); \
		newace.setAccess(permType.allow); \
		newace.setType(RdsAce::Allow); \
		newace.setFlags(inheritFlags); \
		dacls.append(newace); \
	} \
	if (permType.deny != 0 && permType.deny != 0x100000) \
	{ \
		RdsAce newace; \
		newace.setSid(sid); \
		newace.setAccess(permType.deny); \
		newace.setType(RdsAce::Deny); \
		newace.setFlags(inheritFlags); \
		dacls.append(newace); \
	}

	WRITE_OUT_PERMISSIONS(allPerms, (RdsAce::FileInherit | RdsAce::FolderInherit));
	WRITE_OUT_PERMISSIONS(folderSubfolder, (RdsAce::FolderInherit));
	WRITE_OUT_PERMISSIONS(folderSubfiles, (RdsAce::FileInherit));
	WRITE_OUT_PERMISSIONS(subfilesSubfolder, (RdsAce::InheritOnly | RdsAce::FileInherit | RdsAce::FolderInherit));
	WRITE_OUT_PERMISSIONS(subFolders, (RdsAce::InheritOnly | RdsAce::FolderInherit));
	WRITE_OUT_PERMISSIONS(subFiles, (RdsAce::InheritOnly | RdsAce::FileInherit));
	WRITE_OUT_PERMISSIONS(thisFolder, (RdsAce::NoPropogate));

	_acl.setDacl(dacls);
	emit changed();
	return true; // Why did I make this a bool? meh.
}

void PermissionsWidget::selectAcl(RdsSid sid, RdsAce::Flags flags)
{
	for (int i = 0; i < _aclmodel->rowCount(QModelIndex()); i++)
	{
		QModelIndex index = _aclmodel->index(i, 0, QModelIndex());
		AclModel::Acl acl = _aclmodel->getAclFromIndex(index);

		if (!AdvancedButton->isChecked())
		{
			if (acl.sid == sid)
			{
				AclView->selectionModel()->setCurrentIndex(_aclsort->mapFromSource(index),
				        QItemSelectionModel::Clear | QItemSelectionModel::Select | QItemSelectionModel::Rows);
			}
		}
		else
		{
			if ((acl.sid == sid) && (acl.flags == flags))
			{
				AclView->selectionModel()->setCurrentIndex(_aclsort->mapFromSource(index),
				        QItemSelectionModel::Clear | QItemSelectionModel::Select | QItemSelectionModel::Rows);
			}
		}
	}
}


