/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/

#include <QApplication>
#include <QDebug>
#include <QLabel>
#include <QFile>
#include <QSettings>
#include <QMessageBox>
#include <RdsClient>
#include <RdsLdapSession>
#include <RdsOrganizationalUnit>
#include <RdsUtils>
#include <RdsEntityManager>
#include <RdsUser>
#include <AuthToken>
#include "mainwindow.h"
#include "loginwindow.h"
#include "qxtcommandoptions.h"
#include "provisionwizard.h"
#include "main.h"
#ifdef Q_OS_WIN32
#include <windows.h>
#endif

QString ConnectionSettings::user = "";
QString ConnectionSettings::pass = "";
QString ConnectionSettings::host = "";
int ConnectionSettings::port = 0;
QString ConnectionSettings::service = "";
bool ConnectionSettings::ignoredisconnect = false;

int main(int argc, char *argv[])
{
	/*
	#ifdef Q_OS_WIN32
	SetErrorMode(SEM_NOALIGNMENTFAULTEXCEPT);
	quint32 fuOldErrorMode = SetErrorMode(0);
	Q_ASSERT(fuOldErrorMode == SEM_NOALIGNMENTFAULTEXCEPT);
	#endif
	*/
	QApplication app(argc, argv);

	QCoreApplication::setOrganizationName("Resara LLC");
	QCoreApplication::setOrganizationDomain("resara.com");
	QCoreApplication::setApplicationName("RDS Console");
	QxtCommandOptions options;
	options.add("auto-connect", "Do not bring up the login window");
	options.alias("auto-connect", "a");
	options.add("exit", "Exit after provisioning");
	options.alias("exit", "e");
	options.add("force-provision", "Provision an already provisioned server");
	options.alias("force-provision", "f");
	options.add("help", "Show this help message");
	options.alias("help", "h");

	options.addSection("Connection options");
	options.add("host", "The hostname or IP to connect to", QxtCommandOptions::Required);
	options.add("port", "Port to connect to the RDS server", QxtCommandOptions::Required);
	options.add("service", "A custom service name to connect to", QxtCommandOptions::Required);
	options.add("user", "Username to authenticate with", QxtCommandOptions::Required);
	options.add("pass", "Password to authenticate with", QxtCommandOptions::Required);

	options.parse(argc, argv);
	options.parse(QCoreApplication::arguments());

	if (options.count("help") || options.showUnrecognizedWarning())
	{
		options.showUsage();
		return -1;
	}

	QString service = getenv("RDS_SERVICE");
	if (options.count("service"))
		service = options.value("service").toString();
	if (service == "")
		service = "Rds";

	QString host = getenv("RDS_SERVER");
	if (options.count("host"))
		host = options.value("host").toString();

	int port = QString(getenv("RDS_PORT")).toInt();
	if (options.count("port"))
		port = options.value("port").toInt();

	QString user = "";
	if (options.count("user"))
		user = options.value("user").toString();
	QString pass = "";
	if (options.count("pass"))
		pass = options.value("pass").toString();

	RdsClient client;
	RdsClient::setGlobalInstance(&client);

	LoginWindow *login = NULL;

	if (options.count("auto-connect"))
	{
		if (port == 0)
			port = 10123;

		QString url = QString("tcp://%1:%2/%3").arg(host).arg(port).arg(service);
		QtRpc::AuthToken token(user, pass);
		ReturnValue ret = client.connect(url, token);
		if (ret.isError())
		{
			QMessageBox::critical(0, "Connection Error", ret.errString());
			return 1;
		}
	}
	else
	{
		login = new LoginWindow(&client, user, pass, host, port, service);
		if (!login->exec())
		{
			return(1);
		}

		user = login->user();
		pass = login->pass();
		host = login->host();
		port = login->port();
		service = login->service();
		login->setEnabled(false);
		login->show();
	}

	ConnectionSettings::user = user;
	ConnectionSettings::pass = pass;
	ConnectionSettings::host = host;
	ConnectionSettings::port = port;
	ConnectionSettings::service = service;

	if (!client.checkProtocolVersions())
	{
		QMessageBox::warning(NULL, "Warning", "The server you are connecting to is a different version than this client. This may cause unexpected behaviour.");
	}

	ReturnValue ret = client.isProvisioned();
	if (ret.isError())
	{
		QMessageBox::critical(NULL, "Error", ret.errString());
		return(1);
	}
	if (!ret.toBool() || options.count("force-provision"))
	{
		if (login != NULL)
		{
			delete login;
			login = NULL;
		}

		ProvisionWizard wizard(user, pass, host, port, service);
		if (!wizard.exec()) return(1);

		if (options.count("exit"))
		{
			return(0);
		}
	}

	ret = RdsUtils::init();
	if (ret.isError())
	{
		QMessageBox::critical(NULL, "Error", "Failed to fetch domain data:" + ret.errString());
		return(1);
	}

	ret = client.ldapSession();
	if (ret.isError())
	{
		QMessageBox::critical(NULL, "Failed to retrieve LDAP session.", ret.errString());
		return(1);
	}

	RdsLdapSession ldap;
	ldap = ret;
	RdsLdapSession::setGlobalSession(&ldap);

	ret = client.getService("UserGroupComputerManager");
	if (ret.isError())
	{
		QMessageBox::critical(NULL, "Failed", "Failed to get user manager: " + ret.errString());
		return(1);
	}

	if (login != NULL)
	{
		delete login;
		login = NULL;
	}
	MainWindow window;
	QCoreApplication::processEvents();

	return app.exec();
}
