/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "listwidget.h"
#include <QMessageBox>
#include <QDebug>
#include <QRegExp>
#include <QRegExpValidator>
#include "config.h"

bool validTest(QLineEdit *edit, QString error, QRegExpValidator *validator)
{
	QString text = edit->text();
	int size = edit->text().size();
	if ((validator != NULL) && (validator->validate(text, size) == QValidator::Acceptable))
		return(true);
	else
	{
		QMessageBox::critical(NULL, "Error", error);
		return(false);
	}
}

ListWidget::ListWidget(QWidget* parent, Qt::WFlags fl)
		: QWidget(parent, fl), Ui::ListWidget(), _strongvalidator(QRegExp(".*"), this)
{
	setupUi(this);
	AddButton->setIcon(QPixmap(findRdsIcon("icons/32x32/add.png")));
	RemoveButton->setIcon(QPixmap(findRdsIcon("icons/32x32/remove.png")));

	UpButton->setIcon(QPixmap(findRdsIcon("icons/32x32/up.png")));
	DownButton->setIcon(QPixmap(findRdsIcon("icons/32x32/down.png")));
	UpButton->hide();
	DownButton->hide();

	_seperator = QRegExp("[ \t\n]");
	_seperatortext = " ";

	QObject::connect(List, SIGNAL(itemChanged(QListWidgetItem*)), this, SIGNAL(changed()));
	QObject::connect(List->model(), SIGNAL(dataChanged(QModelIndex,QModelIndex)), this, SIGNAL(changed()));
	QObject::connect(List->itemDelegate(), SIGNAL(commitData(QWidget*)), this, SLOT(itemEdited()));
}

ListWidget::~ListWidget()
{
}

void ListWidget::on_AddButton_clicked()
{
	if (AddEdit->text().trimmed() == "") return;
	if (!validTest(AddEdit, _errtext, &_strongvalidator)) return;

	QListWidgetItem *item = new QListWidgetItem(_icon, AddEdit->text(), List);
	item->setFlags(item->flags() | Qt::ItemIsEditable);
	emit(changed());
	AddEdit->clear();
}

void ListWidget::on_RemoveButton_clicked()
{
	foreach(QListWidgetItem *item, List->selectedItems())
	{
		delete item;
	}
	emit(changed());
}

void ListWidget::on_UpButton_clicked()
{
	QList<int> rows;
	QList<QListWidgetItem*> selection = List->selectedItems();
	foreach(QListWidgetItem *item, selection)
	{
		rows << List->row(item);
	}
	qSort(rows);
	int topRow = 0;
	foreach(int row, rows)
	{
		if (row == topRow)
		{
			topRow++;
			continue;
		}
		QListWidgetItem *moveMe = List->takeItem(row);
		List->insertItem(row - 1, moveMe);
	}
	foreach(QListWidgetItem *item, selection)
	{
		List->setCurrentItem(item, QItemSelectionModel::Select);
	}
	emit(changed());
}

void ListWidget::on_DownButton_clicked()
{
	QList<int> rows;
	QList<QListWidgetItem*> selection = List->selectedItems();
	foreach(QListWidgetItem *item, selection)
	{
		rows << List->row(item);
	}
	qSort(rows.begin(), rows.end(), qGreater<int>());
	int bottomRow = List->count() - 1;
	foreach(int row, rows)
	{
		if (row == bottomRow)
		{
			bottomRow--;
			continue;
		}
		QListWidgetItem *moveMe = List->takeItem(row);
		List->insertItem(row + 1, moveMe);
	}
	foreach(QListWidgetItem *item, selection)
	{
		List->setCurrentItem(item, QItemSelectionModel::Select);
	}
	emit(changed());
}

QStringList ListWidget::list() const
{
	QStringList tmp;
	for (int i = 0; i < List->count(); i++)
	{
		if (!_strongvalidator.regExp().exactMatch(List->item(i)->text())) continue;
		tmp << List->item(i)->text();
	}

	return(tmp);
}

void ListWidget::setList(const QStringList& newlist)
{
	List->clear();

	foreach(QString text, newlist)
	{
		QListWidgetItem *item = new QListWidgetItem(_icon, text, List);
		item->setFlags(item->flags() | Qt::ItemIsEditable);
	}
}

QString ListWidget::listString() const
{
	return(list().join(_seperatortext));
}

void ListWidget::setListString(const QString& newlist)
{
	setList(newlist.split(_seperator, QString::SkipEmptyParts));
}

void ListWidget::setSeperator(const QRegExp& regex, QString text)
{
	_seperator = regex;
	_seperatortext = text;
}

void ListWidget::setWeakValidator(const QRegExp& regex)
{
	AddEdit->setValidator(new QRegExpValidator(regex, AddEdit));
}

void ListWidget::setStrongValidator(const QRegExp& regex)
{
	_strongvalidator.setRegExp(regex);
}

void ListWidget::setIcon(const QPixmap& icon)
{
	_icon = icon;
}

void ListWidget::setErrorText(const QString& errtext)
{
	_errtext = errtext;
}

void ListWidget::itemEdited()
{
        if(!_strongvalidator.regExp().exactMatch(List->currentItem()->text()))
        {
                QMessageBox::critical(this, "Error", _errtext);
                return;
        }
}

void ListWidget::setUpDownArrows(bool enabled)
{
	UpButton->setVisible(enabled);
	DownButton->setVisible(enabled);
}



