/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/


#include "dhcpconfiguredialog.h"
#include <RdsUtils>
#include <QMessageBox>
#include <QDebug>
#include <QRegExpValidator>
#include <RdsDaemonManager>
#include <RdsDhcpSubnet>
#include <RdsDhcpValues>
#include <RdsClient>
#include <rdsprovisionator.h>
#include <config.h>

#define IP_VALIDATOR QRegExp("^([0-9]{0,3}\\.){0,3}[0-9]{0,3}$")
#define IP_VALIDATOR_STRONG QRegExp("^(([0-9][0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))\\.(([0-9][0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))\\.(([0-9][0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))\\.(([0-9][0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))$")

DhcpConfigureDialog::DhcpConfigureDialog(RdsDhcpManager* manager, QWidget* parent, Qt::WFlags fl)
		: QDialog(parent, fl), Ui::DhcpConfigureDialog()
{
	setupUi(this);
	_manager = manager;
	_valid = true;
	
	StartEdit->setValidator(new QRegExpValidator(IP_VALIDATOR, this));
	EndEdit->setValidator(new QRegExpValidator(IP_VALIDATOR, this));
	
	ReturnValue ret = rdsClient()->provisionator();
	if(ret.isError())
	{
		qWarning() << "Failed to get provisionator object:" << ret;
		_valid = false;
		return;
	}
	
	RdsProvisionator *p = new RdsProvisionator(this);
	*p = ret;
	ret = p->getDefaultSettings();
	if(ret.isError())
	{
		qWarning() << "Failed to get network settings:" << ret;
		_valid = false;
		return;
	}
	
	QVariantMap settings = ret.toMap();
	QHostAddress netmask(settings.value("netmask").toString());
	QHostAddress ip(settings.value("ip").toString());
	QHostAddress network(netmask.toIPv4Address() & ip.toIPv4Address());
	
	QStringList parts = network.toString().split(".");
	if (parts.size() == 4)
	{
		parts[3] = "100";
		StartEdit->setText(parts.join("."));
		parts[3] = "200";
		EndEdit->setText(parts.join("."));
	}
	
	_settings = settings;
	_settings["domain"] = RdsUtils::realm().toLower();
	_settings["dhcpnetmask"] = netmask.toString();
	_provisionator = p;
}


DhcpConfigureDialog::~DhcpConfigureDialog()
{
}

bool DhcpConfigureDialog::isValid()
{
	return(_valid);
}


void DhcpConfigureDialog::accept()
{
	if(!QRegExp(IP_VALIDATOR_STRONG).exactMatch(StartEdit->text()))
	{
		QMessageBox::critical(this, "Error", "The start address must be a valid IP address");
		return;
	}
	
	if(!QRegExp(IP_VALIDATOR_STRONG).exactMatch(EndEdit->text()))
	{
		QMessageBox::critical(this, "Error", "The end address must be a valid IP address");
		return;
	}
	
	_settings["dhcpstart"] = StartEdit->text();
	_settings["dhcpend"] = EndEdit->text();
	
	ReturnValue ret = _provisionator->provisionDhcp(_settings);
	if(ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to configure the DHCP server: " + ret.errString());
		return;
	}
	
	//if we get here, everything worked
	QDialog::accept();
}
