/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdsclient.h"
#include "rdsclient_p.h"
#include <QStringList>
#include <QMutexLocker>
#include <QDebug>

RdsClient *RdsClientPrivate::instance = NULL;
QStringList RdsClientPrivate::serverprotocolversions = QStringList();

QTRPC_CLIENTPROXY_PIMPL_IMPLEMENT(RdsClient);

RdsClient::RdsClient(QObject *parent)
		: QtRpc::ClientProxy(parent)
{
	QXT_INIT_PRIVATE(RdsClient);
}

RdsClient::RdsClient(const RdsClient &other)
		: QtRpc::ClientProxy(other)
{
	QXT_INIT_PRIVATE(RdsClient);
}

RdsClient::~RdsClient()
{
}

RdsClient &RdsClient::operator=(const RdsClient & other)
{
	QtRpc::ClientProxy::operator=(other);
	return *this;
}

void RdsClient::setGlobalInstance(RdsClient *client)
{
	RdsClientPrivate::instance = client;
}

RdsClient *RdsClient::globalInstance()
{
	return(RdsClientPrivate::instance);
}

ReturnValue RdsClient::connect(QUrl url, QObject *obj, const char *slot, const QtRpc::AuthToken &auth)
{
	QMutexLocker locker(&qxt_d().signalMutex);
	QtRpc::AuthToken token = auth;
	token.clientData()["protocolversions"] = rdsProtocolVersions();

	// COPY AND PASTED FROM QTRPC
	if (url.scheme() == "socket")
	{
#ifdef Q_OS_WIN32
		return ReturnValue(1, "Sockets are not supported on windows");
#else
		// sockets have weird paths
		if (url.path().contains(':'))
		{
			token.clientData()["service"] = url.path().mid(url.path().indexOf(':') + 1);
			url.setPath(url.path().left(url.path().indexOf(':')));
		}
#endif
	}
	else
	{
		// > 1 because it includes the server:1000"/Service", so even if there's no service the length could be 1 for the /
		if (url.path().count() > 1)
			token.clientData()["service"] = url.path().mid(1); //cut out the /
		url.setPath(QString());
	}
	// END COPYPASTA

	if (!auth.clientContains("service"))
	{
		qWarning() << "No service name was passed to connect, because we intercept the return value we need one. Manually selecting Rds, which can be overwritten later.";
		token.clientData()["service"] = "Rds";
	}
	ReturnValue ret;
	if (obj)
	{
		RdsClientPrivate::ConnectionObject conobj;
		conobj.object = obj;
		conobj.slot = slot;
		ret = QtRpc::ClientProxy::connect(url, this, SLOT(rdsConnectionCompleted(uint, ReturnValue)), auth);
		if (ret.isError())
			return ret;
		qxt_d().connectionObjects.insert(ret.toUInt(), conobj);
	}
	else
	{
		ret = QtRpc::ClientProxy::connect(url, obj, slot, auth);
		if (ret.isError())
			return ret;
		qxt_d().serverprotocolversions = authToken().clientData()["auth_return"].toStringList();
	}

	return ret;
}

QStringList RdsClient::serverProtocolVersions()
{
	return(qxt_d().serverprotocolversions);
}

void RdsClient::rdsConnectionCompleted(uint id, ReturnValue ret)
{
	if (!ret.isError())
	{
		qxt_d().serverprotocolversions = authToken().clientData()["auth_return"].toStringList();
	}
	QMutexLocker locker(&qxt_d().signalMutex);
	if (!qxt_d().connectionObjects.contains(id))
	{
		qCritical() << "Received a connection return without an associated object for id" << id;
		return;
	}
	RdsClientPrivate::ConnectionObject obj = qxt_d().connectionObjects.take(id);
	if (obj.slot.contains("QtRpc::ReturnValue"))
	{
		QObject::connect(this, SIGNAL(rdsAsyncronousSignalerNamespace(uint, QtRpc::ReturnValue)), obj.object, qPrintable(obj.slot), Qt::DirectConnection);
		emit rdsAsyncronousSignalerNamespace(id, ret);
		QObject::disconnect(this, SIGNAL(rdsAsyncronousSignalerNamespace(uint, QtRpc::ReturnValue)), obj.object, qPrintable(obj.slot));
	}
	else
	{
		QObject::connect(this, SIGNAL(rdsAsyncronousSignaler(uint, ReturnValue)), obj.object, qPrintable(obj.slot), Qt::DirectConnection);
		emit rdsAsyncronousSignaler(id, ret);
		QObject::disconnect(this, SIGNAL(rdsAsyncronousSignaler(uint, ReturnValue)), obj.object, qPrintable(obj.slot));
	}
}

QString RdsClient::serverProtocolVersion(RdsProtocol p)
{
	if (p >= qxt_d().serverprotocolversions.size()) return("0.0");
	else return qxt_d().serverprotocolversions[p];
}

bool RdsClient::checkProtocolVersions()
{
	for (int i = 0; i < rdsProtocolVersions().size(); i++)
	{
		if (!checkProtocolVersion((RdsProtocol)i)) return(false);
	}
	return(true);
}

bool RdsClient::checkProtocolVersion(RdsProtocol p)
{
	QStringList parts1 = rdsProtocolVersion(p).split(".");
	QStringList parts2 = serverProtocolVersion(p).split(".");

	if (parts1.size() != 2) return(false);
	if (parts2.size() != 2) return(false);

	if (parts1[0] != parts2[0]) return(false);

	if (parts1[1].toInt() > parts2[1].toInt()) return(false);

	return(true);
}

