/****************************************************************************
**
** Copyright (C) 2012 Digia Plc and/or its subsidiary(-ies).
** Contact: http://www.qt-project.org/legal
**
** This file is part of the test suite of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and Digia.  For licensing terms and
** conditions see http://qt.digia.com/licensing.  For further information
** use the contact form at http://qt.digia.com/contact-us.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Digia gives you certain additional
** rights.  These rights are described in the Digia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
**
** $QT_END_LICENSE$
**
****************************************************************************/
#include <qtest.h>
#include <QtCore/qdebug.h>
#include <QtCore/qtimer.h>
#include <QtCore/qdir.h>
#include <QtCore/qfileinfo.h>
#include <QtQml/qjsengine.h>

#include <QtQml/qqmlcomponent.h>
#include <QtQml/qqmlengine.h>

#include <private/qquickworkerscript_p.h>
#include <private/qqmlengine_p.h>
#include "../../shared/util.h"

class tst_QQuickWorkerScript : public QQmlDataTest
{
    Q_OBJECT
public:
    tst_QQuickWorkerScript() {}
private slots:
    void source();
    void messaging();
    void messaging_data();
    void messaging_sendQObjectList();
    void messaging_sendJsObject();
    void messaging_sendExternalObject();
    void script_with_pragma();
    void script_included();
    void scriptError_onLoad();
    void scriptError_onCall();
    void stressDispose();

private:
    void waitForEchoMessage(QQuickWorkerScript *worker) {
        QEventLoop loop;
        QVERIFY(connect(worker, SIGNAL(done()), &loop, SLOT(quit())));
        QTimer timer;
        timer.setSingleShot(true);
        connect(&timer, SIGNAL(timeout()), &loop, SLOT(quit()));
        timer.start(10000);
        loop.exec();
        QVERIFY(timer.isActive());
    }

    QQmlEngine m_engine;
};

void tst_QQuickWorkerScript::source()
{
    QQmlComponent component(&m_engine, testFileUrl("worker.qml"));
    QQuickWorkerScript *worker = qobject_cast<QQuickWorkerScript*>(component.create());
    QVERIFY(worker != 0);
    const QMetaObject *mo = worker->metaObject();

    QVariant value(100);
    QVERIFY(QMetaObject::invokeMethod(worker, "testSend", Q_ARG(QVariant, value)));
    waitForEchoMessage(worker);
    QCOMPARE(mo->property(mo->indexOfProperty("response")).read(worker).value<QVariant>(), value);

    QUrl source = testFileUrl("script_fixed_return.js");
    worker->setSource(source);
    QCOMPARE(worker->source(), source);
    QVERIFY(QMetaObject::invokeMethod(worker, "testSend", Q_ARG(QVariant, value)));
    waitForEchoMessage(worker);
    QCOMPARE(mo->property(mo->indexOfProperty("response")).read(worker).value<QVariant>(), qVariantFromValue(QString("Hello_World")));

    qApp->processEvents();
    delete worker;
}

void tst_QQuickWorkerScript::messaging()
{
    QFETCH(QVariant, value);

    QQmlComponent component(&m_engine, testFileUrl("worker.qml"));
    QQuickWorkerScript *worker = qobject_cast<QQuickWorkerScript*>(component.create());
    QVERIFY(worker != 0);

    QVERIFY(QMetaObject::invokeMethod(worker, "testSend", Q_ARG(QVariant, value)));
    waitForEchoMessage(worker);

    const QMetaObject *mo = worker->metaObject();
    QCOMPARE(mo->property(mo->indexOfProperty("response")).read(worker).value<QVariant>(), value);

    qApp->processEvents();
    delete worker;
}

void tst_QQuickWorkerScript::messaging_data()
{
    QTest::addColumn<QVariant>("value");

    QTest::newRow("invalid") << QVariant();
    QTest::newRow("bool") << qVariantFromValue(true);
    QTest::newRow("int") << qVariantFromValue(1001);
    QTest::newRow("real") << qVariantFromValue(10334.375);
    QTest::newRow("string") << qVariantFromValue(QString("More cheeeese, Gromit!"));
    QTest::newRow("variant list") << qVariantFromValue((QVariantList() << "a" << "b" << "c"));
    QTest::newRow("date time") << qVariantFromValue(QDateTime::currentDateTime());
#ifndef QT_NO_REGEXP
    // QtScript's QScriptValue -> QRegExp uses RegExp2 pattern syntax
    QTest::newRow("regexp") << qVariantFromValue(QRegExp("^\\d\\d?$", Qt::CaseInsensitive, QRegExp::RegExp2));
#endif
}

void tst_QQuickWorkerScript::messaging_sendQObjectList()
{
    // Not allowed to send QObjects other than QQuickListModelWorkerAgent
    // instances. If objects are sent in a list, they will be sent as 'undefined'
    // js values.

    QQmlComponent component(&m_engine, testFileUrl("worker.qml"));
    QQuickWorkerScript *worker = qobject_cast<QQuickWorkerScript*>(component.create());
    QVERIFY(worker != 0);

    QVariantList objects;
    for (int i=0; i<3; i++)
        objects << qVariantFromValue(new QObject(this));

    QVERIFY(QMetaObject::invokeMethod(worker, "testSend", Q_ARG(QVariant, qVariantFromValue(objects))));
    waitForEchoMessage(worker);

    const QMetaObject *mo = worker->metaObject();
    QVariantList result = mo->property(mo->indexOfProperty("response")).read(worker).value<QVariantList>();
    QCOMPARE(result, (QVariantList() << QVariant() << QVariant() << QVariant()));

    qApp->processEvents();
    delete worker;
}

void tst_QQuickWorkerScript::messaging_sendJsObject()
{
    QQmlComponent component(&m_engine, testFileUrl("worker.qml"));
    QQuickWorkerScript *worker = qobject_cast<QQuickWorkerScript*>(component.create());
    QVERIFY(worker != 0);

    // Properties are in alphabetical order to enable string-based comparison after
    // QVariant roundtrip, since the properties will be stored in a QVariantMap.
    QString jsObject = "{'haste': 1125, 'name': 'zyz', 'spell power': 3101}";

    QVariantMap map;
    map.insert("haste", 1125);
    map.insert("name", "zyz");
    map.insert("spell power", 3101);

    QVERIFY(QMetaObject::invokeMethod(worker, "testSend", Q_ARG(QVariant, qVariantFromValue(map))));
    waitForEchoMessage(worker);

    QVariant result = qVariantFromValue(false);
    QVERIFY(QMetaObject::invokeMethod(worker, "compareLiteralResponse", Qt::DirectConnection, 
            Q_RETURN_ARG(QVariant, result), Q_ARG(QVariant, jsObject)));
    QVERIFY(result.toBool());

    qApp->processEvents();
    delete worker;
}

void tst_QQuickWorkerScript::messaging_sendExternalObject()
{
    QQmlComponent component(&m_engine, testFileUrl("externalObjectWorker.qml"));
    QObject *obj = component.create();
    QVERIFY(obj);
    QMetaObject::invokeMethod(obj, "testExternalObject");
    QTest::qWait(100); // shouldn't crash.
    delete obj;
}

void tst_QQuickWorkerScript::script_with_pragma()
{
    QVariant value(100);

    QQmlComponent component(&m_engine, testFileUrl("worker_pragma.qml"));
    QQuickWorkerScript *worker = qobject_cast<QQuickWorkerScript*>(component.create());
    QVERIFY(worker != 0);

    QVERIFY(QMetaObject::invokeMethod(worker, "testSend", Q_ARG(QVariant, value)));
    waitForEchoMessage(worker);

    const QMetaObject *mo = worker->metaObject();
    QCOMPARE(mo->property(mo->indexOfProperty("response")).read(worker).value<QVariant>(), value);

    qApp->processEvents();
    delete worker;
}

void tst_QQuickWorkerScript::script_included()
{
    QQmlComponent component(&m_engine, testFileUrl("worker_include.qml"));
    QQuickWorkerScript *worker = qobject_cast<QQuickWorkerScript*>(component.create());
    QVERIFY(worker != 0);

    QString value("Hello");

    QVERIFY(QMetaObject::invokeMethod(worker, "testSend", Q_ARG(QVariant, value)));
    waitForEchoMessage(worker);

    const QMetaObject *mo = worker->metaObject();
    QCOMPARE(mo->property(mo->indexOfProperty("response")).read(worker).toString(), value + " World");

    qApp->processEvents();
    delete worker;
}

static QString qquickworkerscript_lastWarning;
static void qquickworkerscript_warningsHandler(QtMsgType type, const QMessageLogContext &, const QString &msg)
{
    if (type == QtWarningMsg)
         qquickworkerscript_lastWarning = msg;
}

void tst_QQuickWorkerScript::scriptError_onLoad()
{
    QQmlComponent component(&m_engine, testFileUrl("worker_error_onLoad.qml"));

    QtMessageHandler previousMsgHandler = qInstallMessageHandler(qquickworkerscript_warningsHandler);
    QQuickWorkerScript *worker = qobject_cast<QQuickWorkerScript*>(component.create());
    QVERIFY(worker != 0);

    QTRY_COMPARE(qquickworkerscript_lastWarning,
            testFileUrl("script_error_onLoad.js").toString() + QLatin1String(":3: SyntaxError: Unexpected identifier"));

    qInstallMessageHandler(previousMsgHandler);
    qApp->processEvents();
    delete worker;
}

void tst_QQuickWorkerScript::scriptError_onCall()
{
    QQmlComponent component(&m_engine, testFileUrl("worker_error_onCall.qml"));
    QQuickWorkerScript *worker = qobject_cast<QQuickWorkerScript*>(component.create());
    QVERIFY(worker != 0);

    QtMessageHandler previousMsgHandler = qInstallMessageHandler(qquickworkerscript_warningsHandler);
    QVariant value;
    QVERIFY(QMetaObject::invokeMethod(worker, "testSend", Q_ARG(QVariant, value)));

    QTRY_COMPARE(qquickworkerscript_lastWarning,
            testFileUrl("script_error_onCall.js").toString() + QLatin1String(":4: ReferenceError: getData is not defined"));

    qInstallMessageHandler(previousMsgHandler);
    qApp->processEvents();
    delete worker;
}

// Rapidly create and destroy worker scripts to test resources are being disposed
// in the correct isolate
void tst_QQuickWorkerScript::stressDispose()
{
    for (int ii = 0; ii < 100; ++ii) {
        QQmlEngine engine;
        QQmlComponent component(&engine, testFileUrl("stressDispose.qml"));
        QObject *o = component.create();
        QVERIFY(o);
        delete o;
    }
}

QTEST_MAIN(tst_QQuickWorkerScript)

#include "tst_qquickworkerscript.moc"
