#!/usr/bin/env python

__author__ = "Adam Robbins-Pianka"
__copyright__ = "Copyright 2011, The QIIME Project"
__credits__ = ["Adam Robbins-Pianka"]
__license__ = "GPL"
__version__ = "1.5.0"
__maintainer__ = "Adam Robbins-Pianka"
__email__ = "adam.robbinspianka@colorado.edu"
__status__ = "Release"

from os.path import sep, split, splitext, exists
from shutil import rmtree

from cogent.util.unit_test import TestCase, main
from cogent.util.misc import remove_files, get_random_directory_name

from qiime.util import get_tmp_filename, create_dir

from qiime.convert_fastaqual_to_fastq import convert_fastq

class MakeFastqTests(TestCase):
    """ Unit tests for the fastaQualToFastq_script.py module """

    def setUp(self):
        self._files_to_remove = []

        self.qual_fp = get_tmp_filename(prefix='qual_', suffix='.qual')
        self.fasta_fp = get_tmp_filename(prefix='fasta_', suffix='.fna')

        qual_file = open(self.qual_fp, 'w')
        fasta_file = open(self.fasta_fp, 'w')

        qual_file.write(qual_test_string)
        qual_file.close()

        fasta_file.write(fasta_test_string)
        fasta_file.close()

        self.output_dir = get_random_directory_name(prefix = '/tmp/')
        self.output_dir += sep

        create_dir(self.output_dir)

        self._files_to_remove.append(self.qual_fp)
        self._files_to_remove.append(self.fasta_fp)

    def tearDown(self):
        if self._files_to_remove:
            remove_files(self._files_to_remove)
        if exists(self.output_dir):
            rmtree(self.output_dir)

    def test_default_settings(self):
        convert_fastq(self.fasta_fp, self.qual_fp,
         output_directory=self.output_dir)
        actual_output_file_path = self.output_dir + \
                splitext(split(self.fasta_fp)[1])[0] + '.fastq'
        actual_output_file = open(actual_output_file_path)
        actual_output = actual_output_file.read()
        actual_output_file.close()
        self._files_to_remove.append(actual_output_file_path)

        self.assertEquals(actual_output, expected_fastq_default_options)

    def test_full_fasta_headers(self):
        convert_fastq(self.fasta_fp, self.qual_fp, full_fasta_headers = True, \
                output_directory = self.output_dir)
        actual_output_file_path = self.output_dir + \
                splitext(split(self.fasta_fp)[1])[0] + '.fastq'
        actual_output_file = open(actual_output_file_path)
        actual_output = actual_output_file.read()
        actual_output_file.close()
        self._files_to_remove.append(actual_output_file_path)

        self.assertEquals(actual_output, expected_fastq_full_fasta_headers)

    def test_full_fasta_full_fastq(self):
        convert_fastq(self.fasta_fp, self.qual_fp, full_fasta_headers = True, \
                full_fastq = True, output_directory = self.output_dir)
        actual_output_file_path = self.output_dir + \
                splitext(split(self.fasta_fp)[1])[0] + '.fastq'
        actual_output_file = open(actual_output_file_path)
        actual_output = actual_output_file.read()
        actual_output_file.close()
        self._files_to_remove.append(actual_output_file_path)

        self.assertEquals(actual_output, expected_fastq_full_fasta_full_fastq)

    def test_multiple_output_files(self):
        convert_fastq(self.fasta_fp, self.qual_fp,
         multiple_output_files = True, output_directory = self.output_dir)
        sample_ids = [('PC.634', expected_fastq_634_default), 
                ('PC.354', expected_fastq_354_default), 
                ('PC.481', expected_fastq_481_default)]
        for sample_id, expected_output in sample_ids:
            actual_output_file_path = self.output_dir + \
                    splitext(split(self.fasta_fp)[1])[0] +'_%s.fastq' %\
                    sample_id
            actual_output_file = open(actual_output_file_path)
            actual_output = actual_output_file.read()
            actual_output_file.close()
            self.assertEquals(actual_output, expected_output) 
            self._files_to_remove.append(actual_output_file_path)
    
    def test_ascii_increment(self):
        self.assertRaises(ValueError, convert_fastq, self.fasta_fp,
         self.qual_fp, ascii_increment=140, output_directory = self.output_dir)
        self.assertRaises(ValueError, convert_fastq, self.fasta_fp,
         self.qual_fp, ascii_increment=10, output_directory = self.output_dir)
        


fasta_test_string = '''>PC.634_1 FLP3FBN01ELBSX orig_bc=GCAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=1
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTTACCCTCTCAGGCCGGCTACGCATCATCGCCTTGGTGGGCCGTTACCTCACCAACTAGCTAATGCGCCGCAGGTCCATCCATGTTCACGCCTTGATGGGCGCTTTAATATACTGAGCATGCGCTCTGTATACCTATCCGGTTTTAGCTACCGTTTCCAGCAGTTATCCCGGACACATGGGCTAGG
>PC.634_2 FLP3FBN01EG8AX orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCA
>PC.354_3 FLP3FBN01EEWKD orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTTAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.481_4 FLP3FBN01DEHK3 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTGAGCCGTTACCTCACCAACTATCTAATCAGACGCGAGCCCATCTTTCAGCGGATTGCTCCTTTGGTATTCCGGCGATGCCGCCAAAATCATTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCAGGTTGCTCACG
>PC.634_5 FLP3FBN01DGFYQ orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCTCACACCGATTAATC
'''

qual_test_string = '''>PC.634_1 FLP3FBN01ELBSX orig_bc=GCAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=1
40 40 39 39 38 40 40 40 40 40 40 40 37 37 37 37 37 35 35 35 37 37 37 37 37 35 35 35 31 31 23 23 23 31 21 21 21 35 35 37 37 37 36 36 36 36 36 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 28 28 28 36 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 35 32 32 32 32 35 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 32 32 32 36 37 35 32 32 32 32 32 32 32 32 36 37 37 37 37 36 36 31 31 32 32 36 36 36 36 36 36 36 36 36 36 36 28 27 27 27 26 26 26 30 29 30 29 24 24 24 21 15 15 13 13
>PC.634_2 FLP3FBN01EG8AX orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
24 29 33 33 39 39 39 40 39 39 39 40 37 37 37 37 37 37 37 37 37 37 37 32 32 20 20 20 20 20 35 35 37 37 37 37 37 37 37 36 36 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 36 36 36 37 37 37 37 37 36 36 36 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 33 28 28 28 28 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 33 33 33 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 28 28 28 37 28 28 28 37 37 37 37 37 36 36 36 36 36 28 26 26 26 26 28 36 36 36 36 36 36 36 37 38 38 38 38 38 37 37 37 37 37 31 31 31 31 31 31 31 31 31 31 31 31 30 22 22 22 25 25 31 31 31 31 31 31 31 25 25 25 25 25 28
>PC.354_3 FLP3FBN01EEWKD orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
34 34 36 37 36 37 37 37 37 37 37 37 37 37 37 37 36 28 28 28 36 36 37 37 35 35 35 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 32 32 32 35 35 37 35 32 32 32 37 37 37 37 37 37 36 36 36 36 36 36 36 36 37 37 37 37 37 37 37 35 35 35 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 35 35 35 37 37 37 37 37 37 37 37 37 37 36 36 36 37 36 35 35 35 37 28 28 28 32 35 37 37 37 36 36 36 37 37 37 37 37 37 35 35 35 35 35 37 37 37 37 36 36 36 37 28 28 28 28 35 36 37 37 37 37 37 37 37 37 37 37 36 33 33 32 31 36 36 33 33 27 27 27 36 31 25 25 25 32 36 36 36 36 36 36 36 36 36 36
>PC.481_4 FLP3FBN01DEHK3 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 33 33 33 36 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33 33 33 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 28 28 28 28 33 33 33 36 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 36 36 36 31 31 27 27 28 28 28 27 21 31 31 36 36 36 36 36 36 36 36 36 36 36 31 31 31 31 31 31 31
>PC.634_5 FLP3FBN01DGFYQ orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
17 21 20 20 20 31 31 33 35 35 35 35 35 33 33 33 35 31 31 31 35 35 35 35 35 35 35 31 31 31 33 35 35 35 35 35 35 35 35 35 35 31 31 31 26 26 26 26 35 35 35 35 35 35 35 33 31 31 31 35 35 35 35 35 35 35 35 35 35 35 35 35 35 31 31 31 35 35 35 33 33 33 33 33 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 30 26 26 26 30 33 35 35 35 35 35 35 35 35 33 33 33 35 33 27 27 25 25 25 27
'''

expected_fastq_default_options = '''@PC.634_1
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTTACCCTCTCAGGCCGGCTACGCATCATCGCCTTGGTGGGCCGTTACCTCACCAACTAGCTAATGCGCCGCAGGTCCATCCATGTTCACGCCTTGATGGGCGCTTTAATATACTGAGCATGCGCTCTGTATACCTATCCGGTTTTAGCTACCGTTTCCAGCAGTTATCCCGGACACATGGGCTAGG
+
IIHHGIIIIIIIFFFFFDDDFFFFFDDD@@888@666DDFFFEEEEEEFFFFFFFFFFFFFFFFFFFF===EEFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEEEFFFFFFFFFFFFEEEFFFFFFFFFFFFFFFFDAAAADFFFFFFFFFFFFFFFFFFEAAAEFDAAAAAAAAEFFFFEE@@AAEEEEEEEEEEE=<<<;;;?>?>999600..
@PC.634_2
TTGGACCGTGTCTCAGTTCCAATGTGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCA
+
9>BBHHHIHHHIFFFFFFFFFFFAA55555DDFFFFFFFEEEFFFFFFFFFFFFFFFFFFFFFFFFFFFFEEEEEEFFFFFEEEEFFFFFFFFFFFFFFFFEB====EFFFFFFFFFFFFFFFEBBBFFFFFFFFFFFFFFFFFFFFFEEEFFFFFFFFFFFFFFFFFFFF===F===FFFFFEEEEE=;;;;=EEEEEEEFGGGGGFFFFF@@@@@@@@@@@@?777::@@@@@@@:::::=
@PC.354_3
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTTAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
+
CCEFEFFFFFFFFFFFE===EEFFDDDFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFAAADDFDAAAFFFFFFEEEEEEEEFFFFFFFDDDFFFFFFFFFFFFFFEDDDFFFFFFFFFFEEEFEDDDF===ADFFFEEEFFFFFFDDDDDFFFFEEEF====DEFFFFFFFFFFEBBA@EEBB<<<E@:::AEEEEEEEEEE
@PC.481_4
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTGAGCCGTTACCTCACCAACTATCTAATCAGACGCGAGCCCATCTTTCAGCGGATTGCTCCTTTGGTATTCCGGCGATGCCGCCAAAATCATTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCAGGTTGCTCACG
+
IIIIIIIIIIIIFFFFFBBBEEFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFBBBBFFFFFFFFFFFFFFEE====BBBEEFFFFFFFFFFFFFFFFFFFFFFFFEEEEEE@@<<===<6@@EEEEEEEEEEE@@@@@@@
@PC.634_5
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCTCACACCGATTAATC
+
26555@@BDDDDDBBBD@@@DDDDDDD@@@BDDDDDDDDDD@@@;;;;DDDDDDDB@@@DDDDDDDDDDDDDD@@@DDDBBBBBDDDDDDDDDDDDDDDDD?;;;?BDDDDDDDDBBBDB<<:::<
'''

expected_fastq_full_fasta_headers = '''@PC.634_1 FLP3FBN01ELBSX orig_bc=GCAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=1
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTTACCCTCTCAGGCCGGCTACGCATCATCGCCTTGGTGGGCCGTTACCTCACCAACTAGCTAATGCGCCGCAGGTCCATCCATGTTCACGCCTTGATGGGCGCTTTAATATACTGAGCATGCGCTCTGTATACCTATCCGGTTTTAGCTACCGTTTCCAGCAGTTATCCCGGACACATGGGCTAGG
+
IIHHGIIIIIIIFFFFFDDDFFFFFDDD@@888@666DDFFFEEEEEEFFFFFFFFFFFFFFFFFFFF===EEFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEEEFFFFFFFFFFFFEEEFFFFFFFFFFFFFFFFDAAAADFFFFFFFFFFFFFFFFFFEAAAEFDAAAAAAAAEFFFFEE@@AAEEEEEEEEEEE=<<<;;;?>?>999600..
@PC.634_2 FLP3FBN01EG8AX orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCA
+
9>BBHHHIHHHIFFFFFFFFFFFAA55555DDFFFFFFFEEEFFFFFFFFFFFFFFFFFFFFFFFFFFFFEEEEEEFFFFFEEEEFFFFFFFFFFFFFFFFEB====EFFFFFFFFFFFFFFFEBBBFFFFFFFFFFFFFFFFFFFFFEEEFFFFFFFFFFFFFFFFFFFF===F===FFFFFEEEEE=;;;;=EEEEEEEFGGGGGFFFFF@@@@@@@@@@@@?777::@@@@@@@:::::=
@PC.354_3 FLP3FBN01EEWKD orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTTAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
+
CCEFEFFFFFFFFFFFE===EEFFDDDFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFAAADDFDAAAFFFFFFEEEEEEEEFFFFFFFDDDFFFFFFFFFFFFFFEDDDFFFFFFFFFFEEEFEDDDF===ADFFFEEEFFFFFFDDDDDFFFFEEEF====DEFFFFFFFFFFEBBA@EEBB<<<E@:::AEEEEEEEEEE
@PC.481_4 FLP3FBN01DEHK3 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTGAGCCGTTACCTCACCAACTATCTAATCAGACGCGAGCCCATCTTTCAGCGGATTGCTCCTTTGGTATTCCGGCGATGCCGCCAAAATCATTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCAGGTTGCTCACG
+
IIIIIIIIIIIIFFFFFBBBEEFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFBBBBFFFFFFFFFFFFFFEE====BBBEEFFFFFFFFFFFFFFFFFFFFFFFFEEEEEE@@<<===<6@@EEEEEEEEEEE@@@@@@@
@PC.634_5 FLP3FBN01DGFYQ orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCTCACACCGATTAATC
+
26555@@BDDDDDBBBD@@@DDDDDDD@@@BDDDDDDDDDD@@@;;;;DDDDDDDB@@@DDDDDDDDDDDDDD@@@DDDBBBBBDDDDDDDDDDDDDDDDD?;;;?BDDDDDDDDBBBDB<<:::<
'''

expected_fastq_full_fasta_full_fastq = '''@PC.634_1 FLP3FBN01ELBSX orig_bc=GCAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=1
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTTACCCTCTCAGGCCGGCTACGCATCATCGCCTTGGTGGGCCGTTACCTCACCAACTAGCTAATGCGCCGCAGGTCCATCCATGTTCACGCCTTGATGGGCGCTTTAATATACTGAGCATGCGCTCTGTATACCTATCCGGTTTTAGCTACCGTTTCCAGCAGTTATCCCGGACACATGGGCTAGG
+PC.634_1 FLP3FBN01ELBSX orig_bc=GCAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=1
IIHHGIIIIIIIFFFFFDDDFFFFFDDD@@888@666DDFFFEEEEEEFFFFFFFFFFFFFFFFFFFF===EEFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEEEFFFFFFFFFFFFEEEFFFFFFFFFFFFFFFFDAAAADFFFFFFFFFFFFFFFFFFEAAAEFDAAAAAAAAEFFFFEE@@AAEEEEEEEEEEE=<<<;;;?>?>999600..
@PC.634_2 FLP3FBN01EG8AX orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCA
+PC.634_2 FLP3FBN01EG8AX orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
9>BBHHHIHHHIFFFFFFFFFFFAA55555DDFFFFFFFEEEFFFFFFFFFFFFFFFFFFFFFFFFFFFFEEEEEEFFFFFEEEEFFFFFFFFFFFFFFFFEB====EFFFFFFFFFFFFFFFEBBBFFFFFFFFFFFFFFFFFFFFFEEEFFFFFFFFFFFFFFFFFFFF===F===FFFFFEEEEE=;;;;=EEEEEEEFGGGGGFFFFF@@@@@@@@@@@@?777::@@@@@@@:::::=
@PC.354_3 FLP3FBN01EEWKD orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTTAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
+PC.354_3 FLP3FBN01EEWKD orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
CCEFEFFFFFFFFFFFE===EEFFDDDFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFAAADDFDAAAFFFFFFEEEEEEEEFFFFFFFDDDFFFFFFFFFFFFFFEDDDFFFFFFFFFFEEEFEDDDF===ADFFFEEEFFFFFFDDDDDFFFFEEEF====DEFFFFFFFFFFEBBA@EEBB<<<E@:::AEEEEEEEEEE
@PC.481_4 FLP3FBN01DEHK3 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTGAGCCGTTACCTCACCAACTATCTAATCAGACGCGAGCCCATCTTTCAGCGGATTGCTCCTTTGGTATTCCGGCGATGCCGCCAAAATCATTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCAGGTTGCTCACG
+PC.481_4 FLP3FBN01DEHK3 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
IIIIIIIIIIIIFFFFFBBBEEFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFBBBBFFFFFFFFFFFFFFEE====BBBEEFFFFFFFFFFFFFFFFFFFFFFFFEEEEEE@@<<===<6@@EEEEEEEEEEE@@@@@@@
@PC.634_5 FLP3FBN01DGFYQ orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCTCACACCGATTAATC
+PC.634_5 FLP3FBN01DGFYQ orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
26555@@BDDDDDBBBD@@@DDDDDDD@@@BDDDDDDDDDD@@@;;;;DDDDDDDB@@@DDDDDDDDDDDDDD@@@DDDBBBBBDDDDDDDDDDDDDDDDD?;;;?BDDDDDDDDBBBDB<<:::<
'''

expected_fastq_354_default = '''@PC.354_3
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTTAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
+
CCEFEFFFFFFFFFFFE===EEFFDDDFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFAAADDFDAAAFFFFFFEEEEEEEEFFFFFFFDDDFFFFFFFFFFFFFFEDDDFFFFFFFFFFEEEFEDDDF===ADFFFEEEFFFFFFDDDDDFFFFEEEF====DEFFFFFFFFFFEBBA@EEBB<<<E@:::AEEEEEEEEEE
'''

expected_fastq_481_default = '''@PC.481_4
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTGAGCCGTTACCTCACCAACTATCTAATCAGACGCGAGCCCATCTTTCAGCGGATTGCTCCTTTGGTATTCCGGCGATGCCGCCAAAATCATTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCAGGTTGCTCACG
+
IIIIIIIIIIIIFFFFFBBBEEFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFBBBBFFFFFFFFFFFFFFEE====BBBEEFFFFFFFFFFFFFFFFFFFFFFFFEEEEEE@@<<===<6@@EEEEEEEEEEE@@@@@@@
'''

expected_fastq_634_default = '''@PC.634_1
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTTACCCTCTCAGGCCGGCTACGCATCATCGCCTTGGTGGGCCGTTACCTCACCAACTAGCTAATGCGCCGCAGGTCCATCCATGTTCACGCCTTGATGGGCGCTTTAATATACTGAGCATGCGCTCTGTATACCTATCCGGTTTTAGCTACCGTTTCCAGCAGTTATCCCGGACACATGGGCTAGG
+
IIHHGIIIIIIIFFFFFDDDFFFFFDDD@@888@666DDFFFEEEEEEFFFFFFFFFFFFFFFFFFFF===EEFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEEEFFFFFFFFFFFFEEEFFFFFFFFFFFFFFFFDAAAADFFFFFFFFFFFFFFFFFFEAAAEFDAAAAAAAAEFFFFEE@@AAEEEEEEEEEEE=<<<;;;?>?>999600..
@PC.634_2
TTGGACCGTGTCTCAGTTCCAATGTGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCA
+
9>BBHHHIHHHIFFFFFFFFFFFAA55555DDFFFFFFFEEEFFFFFFFFFFFFFFFFFFFFFFFFFFFFEEEEEEFFFFFEEEEFFFFFFFFFFFFFFFFEB====EFFFFFFFFFFFFFFFEBBBFFFFFFFFFFFFFFFFFFFFFEEEFFFFFFFFFFFFFFFFFFFF===F===FFFFFEEEEE=;;;;=EEEEEEEFGGGGGFFFFF@@@@@@@@@@@@?777::@@@@@@@:::::=
@PC.634_5
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCTCACACCGATTAATC
+
26555@@BDDDDDBBBD@@@DDDDDDD@@@BDDDDDDDDDD@@@;;;;DDDDDDDB@@@DDDDDDDDDDDDDD@@@DDDBBBBBDDDDDDDDDDDDDDDDD?;;;?BDDDDDDDDBBBDB<<:::<
'''

if __name__ == '__main__':
    main()
