#!/usr/bin/env python
# File created on 09 Feb 2010
from __future__ import division

__author__ = "Rob Knight"
__copyright__ = "Copyright 2011, The QIIME Project"
__credits__ = ["Rob Knight","Justin Kuczynski","Greg Caporaso"]
__license__ = "GPL"
__version__ = "1.5.0"
__maintainer__ = "Daniel McDonald"
__email__ = "wasade@gmail.com"
__status__ = "Release"
 

from qiime.util import parse_command_line_parameters
from qiime.util import make_option
from biom.parse import parse_biom_table
from qiime.parse import parse_taxonomy_to_otu_metadata
from qiime.format import format_biom_table

script_info={}
script_info['brief_description']="""Add taxa to OTU table"""
script_info['script_description']="""This script adds taxa to a biom-formatted OTU table."""
script_info['script_usage']=[]

script_info['script_usage'].append(("""Example:""","""Given an input otu table with no metadata (otu_table_no_tax.biom) and a tab-separated text file mapping OTU ids to taxonomic assignments and scores associated with those assignments (tax.txt), generate a new otu table that includes taxonomic assignments (otu_table_w_tax.biom).""","""%prog -i otu_table_no_tax.biom -o otu_table_w_tax.biom -t tax.txt"""))

script_info['script_usage'].append(("""Example:""","""Given an input otu table with no metadata (otu_table_no_tax.biom) and a tab-separated text file mapping OTU ids to taxonomic assignments and scores associated with those assignments (tax.txt), generate a new otu table that includes taxonomic assignments (otu_table_w_tax.biom) with alternate metadata identifiers.""","""%prog -i otu_table_no_tax.biom -o otu_table_w_alt_labeled_tax.biom -t tax.txt -l "Consensus Lineage,Score" """))

script_info['script_usage'].append(("""Example:""","""Given an input otu table with no metadata (otu_table_no_tax.biom) and a tab-separated text file mapping OTU ids to some value, generate a new otu table that includes that metadata category labeled as "Score" (otu_table_w_score.biom).""","""%prog -i otu_table_no_tax.biom -o otu_table_w_score.biom -t score_only.txt -l "Score" --all_strings"""))

script_info['output_description']="""An OTU table in biom format is written to the file specified as -o."""
script_info['required_options']=[\
    make_option('-i','--input_fp',type='existing_filepath',
                help='path to input otu table file in biom format'),
    make_option('-o','--output_fp',type='new_filepath',
                help='path to output file in biom format'),
    make_option('-t','--taxonomy_fp',type='existing_filepath',
                help='path to input taxonomy file (e.g., as generated by assign_taxonomy.py)'),
]

script_info['optional_options']=[
    make_option('-l','--labels',type='string',default='taxonomy,score',
                help='labels to be assigned to metadata in taxonomy_fp'),
    make_option('--all_strings',action='store_true',default=False,
                help='treat all metadata as strings, rather than casting to lists/floats (useful with --labels for adding arbitrary observation metadata) [default:%default]')]
script_info['version'] = __version__

def main():
    option_parser, opts, args = parse_command_line_parameters(**script_info)
    
    labels = opts.labels.split(',')
    if opts.all_strings:
        process_fs = [str] * len(labels)
        observation_metadata = parse_taxonomy_to_otu_metadata(\
                            open(opts.taxonomy_fp,'U'),labels=labels,process_fs=process_fs)
    else:
        observation_metadata = parse_taxonomy_to_otu_metadata(\
                            open(opts.taxonomy_fp,'U'),labels=labels)
    

    otu_table = parse_biom_table(open(opts.input_fp,'U'))
    
    if otu_table.ObservationMetadata != None:
        # if there is already metadata associated with the 
        # observations, confirm that none of the metadata names
        # are already present
        existing_keys = otu_table.ObservationMetadata[0].keys()
        for label in labels:
            if label in existing_keys:
                option_parser.error(\
                 "%s is already an observation metadata field." 
                 " Can't add it, so nothing is being added." % label)
    
    otu_table.addObservationMetadata(observation_metadata)
    
    output_f = open(opts.output_fp,'w')
    output_f.write(format_biom_table(otu_table))
    output_f.close()
    
    
    

if __name__ == "__main__":
    main()
