/***************************************************************************
 *   Copyright (C) 2007 by Anistratov Oleg                                 *
 *   ower@users.sourceforge.net                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation;                         *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 ***************************************************************************/

#include "filetransferwgt.h"

#include <QFileDialog>

FileTransferWgt::FileTransferWgt(const QString & filename_, bool receiving_, QWidget *parent)
 : QWidget(parent),
  m_id       (0),
  m_srcIp    (0),
  m_filename (filename_),
  m_finished (false),
  m_receiving(receiving_),
  m_accepted (false)
{
  m_grid          = new QGridLayout (this);
  m_cancelBtn     = new QPushButton (this);
  m_acceptBtn     = new QPushButton (this);
  m_rejectBtn     = new QPushButton (this);
  m_progress      = new QProgressBar(this);
  m_autoCloseChbx = new QCheckBox   (this);

  if(!m_receiving)
  {
    m_grid->addWidget(new QLabel(m_filename), 0, 0, 1, 2);
    m_grid->addWidget(m_progress            , 1, 0, 1, 2);
    m_grid->addWidget(m_autoCloseChbx       , 2, 0, 1, 2);
    m_grid->addWidget(m_cancelBtn           , 3, 0, 1, 2);
    m_acceptBtn->hide();
    m_rejectBtn->hide();
  }
  else
  {
    m_grid->addWidget(new QLabel(m_filename), 0, 0, 1, 2);
    m_grid->addWidget(m_progress            , 1, 0, 1, 2);
    m_grid->addWidget(m_acceptBtn           , 2, 0);
    m_grid->addWidget(m_rejectBtn           , 2, 1);
    m_cancelBtn    ->hide();
    m_autoCloseChbx->hide();
  }

  m_progress->setMinimumHeight(25);
  setMinimumWidth (300);

  m_progress->setTextVisible(true);
  m_progress->setValue(0);

  connect(m_cancelBtn, SIGNAL(clicked()), this, SLOT(slot_reject     ()));
  connect(m_acceptBtn, SIGNAL(clicked()), this, SLOT(slot_accept     ()));
  connect(m_rejectBtn, SIGNAL(clicked()), this, SLOT(slot_reject     ()));

  retranslate();

  show();
}
//\*****************************************************************************
FileTransferWgt::~FileTransferWgt()
{
}
//\*****************************************************************************
void FileTransferWgt::retranslate()
{
  m_cancelBtn     ->setText(tr("&Cancel"));
  m_acceptBtn     ->setText(tr("&Accept"));
  m_rejectBtn     ->setText(tr("&Reject"));
  m_autoCloseChbx ->setText(tr("Auto close when complete"));

  if(!m_receiving) setWindowTitle(QString(tr("Sending "  )) + m_filename);
  else             setWindowTitle(QString(tr("Receiving ")) + m_filename);
}
//\*****************************************************************************
void FileTransferWgt::slot_accept()
{
  qDebug("[FileTransferWgt::slot_accept]");

  QString filename = QFileDialog::getSaveFileName(
                    this,
                    tr("Choose a file to save"),
                    QString("/") + m_filename,
                    "");

  if(filename.isEmpty())
    return;

  m_filename = filename;
  m_grid     ->removeWidget(m_acceptBtn);
  m_grid     ->removeWidget(m_rejectBtn);
  m_acceptBtn->hide();
  m_rejectBtn->hide();
  m_grid     ->addWidget(m_autoCloseChbx, 2, 0, 1, 2);
  m_grid     ->addWidget(m_cancelBtn    , 3, 0, 1, 2);
  m_autoCloseChbx->show();
  m_cancelBtn    ->show();
  m_accepted = true;

  emit accepted(m_filename, m_id, m_srcIp);
}
//\*****************************************************************************
void FileTransferWgt::slot_reject()
{
  qDebug("[FileTransferWgt::slot_reject] ip = %lu", (unsigned long) m_srcIp);

  if(!m_receiving)
  {
    emit rejected(m_id, m_srcIp, 2);
    emit cancel  (m_id);
  }
  if(!m_finished)
    emit rejected(m_id, m_srcIp, m_accepted);

  slot_selfDestroy();
}
//\*****************************************************************************
void FileTransferWgt::slot_selfDestroy()
{
  hide();
  this->~FileTransferWgt();
}
//\*****************************************************************************
void FileTransferWgt::slot_setProgress(quint8 progress, quint16 id, const QHostAddress & addr)
{
  if(!m_finished && m_id == id && (!m_receiving || addr.toIPv4Address() == m_srcIp))
  {
    m_progress->setValue(100 - progress);

    if(progress == 0)
    {
      m_finished = true;

      if(m_autoCloseChbx->checkState() == Qt::Checked)
        slot_selfDestroy();
      else
      {
        m_progress->setFormat(tr("Completed (%p%)"));
        m_progress->repaint();
        m_cancelBtn->setText(tr("Close"));
      }
    }
  }
}
//\*****************************************************************************
void FileTransferWgt::slot_cancelledByReceiver(quint16 id)
{
  if(!m_finished && id == m_id)
  {
    m_progress->setFormat(tr("Failed: cancelled by receiver(%p%)"));
    m_progress->repaint();
    m_finished = true;
  }
}
//\*****************************************************************************
void FileTransferWgt::slot_cancelledBySender(quint16 id, quint64 ip)
{
  if(!m_finished && id == m_id && ip == m_srcIp && !m_finished)
  {
    m_progress->setFormat(tr("Failed: cancelled by sender (%p%)"));
    m_progress->repaint();
    m_finished = true;
  }
}
//\*****************************************************************************
void FileTransferWgt::slot_cannotSend(quint16 id)
{
  if(!m_finished && id == m_id)
  {
    m_progress->setFormat(tr("Failed: cannot send file"));
    m_progress->repaint();
    m_finished = true;
  }
}
//\*****************************************************************************
void FileTransferWgt::slot_rejectedByReceiver (quint16 id)
{
  if(!m_finished && id == m_id)
  {
    m_progress->setFormat(tr("Failed: rejected by receiver"));
    m_progress->repaint();
    m_finished = true;
  }

}
//\*****************************************************************************
void FileTransferWgt::slot_receivingTimeout(quint16 id, quint64 ip)
{
  if(!m_finished && id == m_id && ip == m_srcIp)
  {
    m_progress->setFormat(tr("Failed: receiving timeout (%p%)"));
    m_progress->repaint();
    m_finished = true;
  }

}
//\*****************************************************************************
void FileTransferWgt::slot_sendingTimeout(quint16 id)
{
  if(!m_finished && id == m_id && !m_finished)
  {
    m_progress->setFormat(tr("Possibly failed: sending finished but only %p% confirmed"));
    m_progress->repaint();
    m_finished = true;
  }

}
//\*****************************************************************************
