/***************************************************************************
 *   Copyright (C) 2007 by Anistratov Oleg                                 *
 *   ower@users.sourceforge.net                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation;                         *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 ***************************************************************************/

#include "chatwgt.h"
#include "globals.h"

#include <assert.h>
#include <string.h>
#include <time.h>

#include <QSplitter>
#include <QMessageBox>
#include <QApplication>
#include <QBuffer>
#include <QDir>
#include <QFile>
#include <QFileDialog>
#include <QHBoxLayout>
#include <QInputDialog>
#include <QLocale>
#include <QCursor>
#include <QStatusBar>
#include <QKeySequence>

#include "singlemessagewgt.h"
#include "filetransferwgt.h"
#include "userwgt.h"
#include "userinfo.h"
#include "addchanneldialog.h"
#include "edituserinfodlg.h"
#include "receiverthread.h"
#include "senderthread.h"
#include "largedatagram.h"
#include "largedatagramout.h"
#include "logwgt.h"
#include "preferences.h"
#include "preferencesdlg.h"
#include "chatcore.h"
#include "smileswgt.h"
#include "channelwgt.h"
#include "chattextwgt.h"
#include "inputtextwgt.h"
#include "qchattrayicon.h"
#include "message.h"
#include "singlemsgshistoryview.h"
#include "singlemsgshistory.h"

#include "aboutqchat.h"

const char* BuildDate    = "";

ChatWgt::ChatWgt(ChatCore* chc, QWidget* parent)
 : QMainWindow(parent),
 m_chatCore    (chc),
 m_menuToolbars(NULL),
 mw_channels   (0),
 m_channelsNum (0),
 m_cursorX     (-1),
 m_cursorY     (-1)
{
  m_translator    = new QTranslator;
  m_activityTimer = new QTimer(this);

  createWidgets();

  createActions();

  setupLayout();

  ChatTextWgt::initSmiles(Globals::prefs()->smilesThemePath());
  mw_smiles->init();

  mw_log->hide();
  Globals::m_log = mw_log;

  m_trayIcon->show ();
  m_trayIcon->setContextMenu(m_menuFile);

  //***********************************
  connect(this          , SIGNAL(     singleMessage(const QString &, const QHostAddress &)),
          m_chatCore    , SLOT  (slot_singleMessage(const QString &, const QHostAddress &)));

  connect(m_chatCore    , SIGNAL(profileLoaded(const QString &)),
          this          , SLOT  (slot_reloadProfileData()));

  connect(this          , SIGNAL(wantLoadProfile (const QString &)),
          m_chatCore    , SLOT  (slot_loadProfile(const QString &)));

  connect(this          , SIGNAL(wantRenameProfile (const QString &, const QString &)),
          m_chatCore    , SLOT  (slot_renameProfile(const QString &, const QString &)));

  connect(this          , SIGNAL(wantDeleteProfile (const QString &)),
          m_chatCore    , SLOT  (slot_deleteProfile(const QString &)));

  connect(m_activityTimer, SIGNAL(timeout()),
          this           , SLOT  (activity()));


  //***********************************
  Globals::info()->setStatus(Globals::FREE);

  setWindowTitle(QString("Q_Chat ver. %1"   ).arg(Globals::VersionStr));

  QApplication::setWindowIcon(QIcon(":/images/trayIcon.png"));

  if(!m_translator->load("qchat_" + QLocale().name () + ".qm", Globals::prefs()->settingsDir()))
    if(!m_translator->load("qchat_" + QLocale().name () + ".qm", "/usr/share/qchat/translations/"))
      m_translator->load("qchat_" + QLocale().name () + ".qm", "/usr/local/share/qchat/translations/");

  QApplication::installTranslator(m_translator);

  m_activityTimer->start(1000);

  retranslate();
}
//\*****************************************************************************
ChatWgt::~ChatWgt()
{
  qDebug("~ChatWgt\n");
  m_addChannelDlg->~AddChannelDlg();
}
//\*****************************************************************************
int ChatWgt::findChannel(const QString & name, quint32 type) const
{
  for(int i = 0; i < m_channelsNum; i++)
    if(mw_channels[i]->name() == name && (1 || mw_channels[i]->type() == type))
      return i;

  return -1;
}
//\*****************************************************************************
int ChatWgt::findChannel(const QHostAddress & ip, quint32 type) const
{
  for(int i = 0; i < m_channelsNum; i++)
    if(mw_channels[i]->destIp() == ip && (mw_channels[i]->type() == type))
      return i;

  return -1;
}
//\*****************************************************************************
void ChatWgt::slot_setIP(const QString & ip)
{
  quint64 ip_ = QHostAddress(ip.toLocal8Bit().data()).toIPv4Address();
  if(ip_)
    Globals::prefs()->setIp(ip_);
}
void ChatWgt::slot_setUsername(const QString & name)
{
  Globals::info()->setNickname(name);
}
//\*****************************************************************************
void ChatWgt::slot_singleMessageIn(Message* msg)
{
  m_singleMessagesHistoryView->appendMsg(msg);

  SingleMessageWgt* smw = new SingleMessageWgt(msg, 1);
  connect(smw , SIGNAL(singleMessage   (const QString &, const QHostAddress &)),
          this, SIGNAL(singleMessageOut(const QString &, const QHostAddress &)));
}
//\*****************************************************************************
void ChatWgt::createChannel(const QString & name, const QHostAddress & addr)
{
  int new_idx;
  ChannelWgt* new_ch;

  if(addr.toIPv4Address() == 0xffffffff)
  {
    if(name == "Log" || name == "log")
    {
      new_idx = mw_tabs->addTab(mw_log, QString("Log"));
      mw_tabs->setCurrentIndex(new_idx);
      return;
    }

    if(findChannel(name) >= 0)
      return;
  }
  else
  {
    if((new_idx = findChannel(name, 1)) >= 0)
      return;

    for(qint32 i = 0; i < m_channelsNum; i++)
      if(mw_channels[i]->destIp() == addr)
        return;
  }

  m_channelsNum++;
  mw_channels = (ChannelWgt**)realloc(mw_channels, m_channelsNum * sizeof(ChannelWgt*));
  assert(NULL != mw_channels);

  if(addr.toIPv4Address() == 0xffffffff)
  {
    new_ch = new ChannelWgt(name, this);
    new_idx = mw_tabs->addTab(new_ch, name);
  }
  else
  {
    new_ch = new ChannelWgt(name, this, 1, addr);
    new_idx = mw_tabs->addTab(new_ch, name + "(private)");
  }

  mw_channels[m_channelsNum - 1] = new_ch;

  mw_channels[m_channelsNum - 1]->show();

  connect(new_ch, SIGNAL(wantActivate   ()), this, SLOT(slot_activateWindow ()));

  connect(new_ch    , SIGNAL(     statusAnswer   (QString, QHostAddress, quint32, bool, bool)),
          m_chatCore, SLOT  (slot_statusAnswer   (QString, QHostAddress, quint32, bool, bool)));

  connect(new_ch    , SIGNAL(     infoAnswer     (QString, QHostAddress, quint32, uchar)),
          m_chatCore, SLOT  (slot_infoAnswer     (QString, QHostAddress, quint32, uchar)));

  connect(new_ch    , SIGNAL(sendSomeData(QString, QHostAddress, uint, QString, quint32, QByteArray*)),
          m_chatCore, SLOT  (slot_prepareAndSend(QString, QHostAddress, uint, QString, quint32, QByteArray*)));

  connect(new_ch    , SIGNAL(sendMsgsHistory       (const QString &, const QHostAddress &, const QByteArray &, uint)),
          m_chatCore, SLOT  (slot_msgsHistoryAnswer(const QString &, const QHostAddress &, const QByteArray &, uint)));

  connect(new_ch    , SIGNAL(sendMsgsNum       (const QString &, const QHostAddress &, quint32, uint)),
          m_chatCore, SLOT  (slot_msgsNumAnswer(const QString &, const QHostAddress &, quint32, uint)));

  connect(m_preferencesDlg, SIGNAL(ulRefreshIntervalChanged(uint)),
          new_ch          , SLOT  (slot_changeUlRefreshInterval(uint)));

  connect(m_preferencesDlg, SIGNAL(ulDeepRefreshIntervalChanged(uint)),
          new_ch          , SLOT  (slot_changeUlDeepRefreshInterval(uint)));

  connect(new_ch    , SIGNAL(sendMessage     (const QString &, const QHostAddress &, quint32, QTextDocument*)),
          m_chatCore, SLOT  (slot_sendMessage(const QString &, const QHostAddress &, quint32, QTextDocument*)));

  connect(new_ch    , SIGNAL(wantSaveState  (const QString &, const QByteArray &)),
          m_chatCore, SLOT  (setChannelState(const QString &, const QByteArray &)));

  connect(new_ch    , SIGNAL(avatarAnswer     (const QString &, const QHostAddress&, quint32)),
          m_chatCore, SLOT  (slot_avatarAnswer(const QString &, const QHostAddress&, quint32)));


  mw_channels[m_channelsNum - 1]->setFocus();
  mw_channels[m_channelsNum - 1]->setFocus2InputText();
  mw_channels[m_channelsNum - 1]->setSndOnMsgIn(Globals::prefs()->soundOnMsgIn());

  mw_channels[m_channelsNum - 1]->initChannel();

  mw_tabs->setCurrentIndex(new_idx);

  if(addr.toIPv4Address() != 0xffffffff)
    m_chatCore->slot_privateChatRequest(Globals::info()->nickname(), addr);

  new_ch->restoreState(m_chatCore->channelState(name));
}
//\*****************************************************************************
void ChatWgt::slot_delChannell()
{
  ChannelWgt* wgt = (ChannelWgt*)mw_tabs->currentWidget();

  if(mw_tabs->currentWidget() == mw_log)
  {
    mw_tabs->removeTab(mw_tabs->currentIndex());
    return;
  }

  QString name_id = wgt->name();
  quint32 type    = wgt->type();
  int num = findChannel(name_id, type);

  if(name_id == "Main")
    return;

  QMessageBox* msgbx;
  int ans;
  msgbx = new QMessageBox(tr("Are you shure?"), tr("Are you shure?"),
                          QMessageBox::Question, QMessageBox::Yes, QMessageBox::No, 0, this, 0);

  ans = msgbx->exec();
  delete msgbx;
  if(ans == QMessageBox::No)
    return;

  if(num >= 0)
  {
    mw_channels[num] = mw_channels[m_channelsNum - 1];
    m_channelsNum--;
  }

  delete wgt;
}
//\*****************************************************************************
void ChatWgt::slot_license()
{
   QString str =
     "This program is free software; you can redistribute it and/or modify\n"
     "it under the terms of the GNU General Public License version 2\n"
     "as published by the Free Software Foundation;\n\n"
     "This program is distributed in the hope that it will be useful,\n"
     "but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
     "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
     "GNU General Public License for more details.\n\n"
     "Copyright (C) 2007 by Anistratov Oleg\n"
     "ower@users.sourceforge.net";

   QMessageBox::information (this, "License", str);
}
//\*****************************************************************************
void ChatWgt::slot_aboutQt()
{
  QMessageBox::aboutQt(this);
}
//\*****************************************************************************
void ChatWgt::slot_about()
{
  AboutQChat* dlg = new AboutQChat(this);

  dlg->exec();

  delete dlg;
}
//\*****************************************************************************
void ChatWgt::slot_showSettings()
{
  m_userInfoDlg->setReadOnly(false);
  m_userInfoDlg->slot_loadInfo(Globals::info());
  m_userInfoDlg->slot_notEdited();
  m_userInfoDlg->toggleVisible();
}
//\*****************************************************************************
void ChatWgt::slot_showPreferences()
{
  m_preferencesDlg->init();
  m_preferencesDlg->toggleVisible();
}
//\*****************************************************************************
void ChatWgt::slot_showUserInfo(UserWgt* user)
{
  m_userInfoDlg->slot_loadInfo(user->info());
  m_userInfoDlg->setReadOnly(true);
  m_userInfoDlg->show();
}
//\*****************************************************************************
void ChatWgt::slot_activateWindow()
{
  if(!QApplication::focusWidget())
    m_trayIcon->setAnimatedIcon(":/images/newMsg.gif");

  if(Globals::prefs()->activateOnMsgIn() && isHidden())
    show();

  if(windowState() & Qt::WindowMaximized)
    this->setWindowState(Qt::WindowMaximized|Qt::WindowActive);
  else if(windowState () & Qt::WindowMinimized)
  {
    this->setWindowState(Qt::WindowNoState);
    this->setWindowState(Qt::WindowActive );
  }
  else
    this->setWindowState(Qt::WindowActive);
}
//\*****************************************************************************
void ChatWgt::slot_sendFile(const QHostAddress & addr)
{
  QString filename;

  filename = QFileDialog::getOpenFileName(
                    this,
                    tr("Choose a file to open"),
                    "/"
                    "");

  qDebug("[ChatWgt::slot_sendFile]: filename = %s", filename.toLocal8Bit().data());

  if(filename.isEmpty())
    return;

  FileTransferWgt* ftw = new FileTransferWgt(filename, 0);
  quint32 id           = m_chatCore->initSendingFile(addr, filename, ftw);

  if(id)
  {
    ftw->setID(id);
    ftw->setSrcIP(addr.toIPv4Address());
  }
}
//\*****************************************************************************
void ChatWgt::slot_receiveFile(const QString & filename, quint16 ID, quint64 srcIP)
{
  FileTransferWgt* ftw = new FileTransferWgt(filename, 1);

  ftw->setID   (ID   );
  ftw->setSrcIP(srcIP);

  m_chatCore->initReceivingFile(ftw);
}
//\*****************************************************************************
void ChatWgt::slot_openSocketError(quint16 port)
{
  mw_log->addError(QString("Couldn't open UDP socket on port %1 ").arg(port));
}
//\*****************************************************************************
void ChatWgt::slot_exit()
{
  for(int i = 0; i < m_channelsNum; i++)
    mw_channels[i]->slot_disconnected();

  m_chatCore->stopThreads();
  m_chatCore->slot_saveSettings();

  QApplication::quit();
}
//\*****************************************************************************
void ChatWgt::slot_trayIconClicked(QSystemTrayIcon::ActivationReason reason)
{
  if(reason == QSystemTrayIcon::Trigger)
  {
    m_trayIcon->setStaticIcon(":/images/trayIcon.png");
    if(isHidden() || windowState () & Qt::WindowMinimized)
    {
      show();
      if(windowState() & Qt::WindowMaximized)
        setWindowState(Qt::WindowMaximized|Qt::WindowActive);
      else
        setWindowState(Qt::WindowNoState);
    }
    else
      hide();
  }
}
//\*****************************************************************************
void ChatWgt::slot_showSmiles()
{
  if(!mw_smiles || !mw_smiles->inited())
    return;

  if(mw_smiles->isHidden())
  {
    mw_smiles->setWindowFlags(Qt::FramelessWindowHint | mw_smiles->windowFlags() | Qt::WindowStaysOnTopHint);
    mw_smiles->resize(1, 1);
    mw_smiles->setMaximumSize(1, 1);

    mw_smiles->move(QCursor().pos());

    mw_smiles->show();
  }
  else
    mw_smiles->hide();
}
//\*****************************************************************************
void ChatWgt::slot_insertSmile(const QString & smile)
{
  ChannelWgt* chnnl = qobject_cast<ChannelWgt*>(mw_tabs->currentWidget());

  if(chnnl)
    chnnl->inputText()->append(smile);
}
//\*****************************************************************************
QString ChatWgt::currentChannelName() const
{
  ChannelWgt* chnnl = qobject_cast<ChannelWgt*>(mw_tabs->currentWidget());

  if(chnnl)
    return chnnl->name();
  else
    return "";
}
//\*****************************************************************************
void ChatWgt::fillProfilesCmbx(const QStringList & profiles, const QString & current)
{
  disconnect(m_profilesCmbx, SIGNAL(currentIndexChanged(const QString &)),
             m_chatCore    , SLOT  (slot_loadProfile   (const QString &)));

  QStringList::const_iterator i;
  QStringList::const_iterator end = profiles.constEnd();
  int idx;

  for(i = profiles.constBegin(); i != end; ++i)
    m_profilesCmbx->addItem(*i);

  if((idx = m_profilesCmbx->findText(current)) >= 0)
    m_profilesCmbx->setCurrentIndex(idx);

  connect(m_profilesCmbx, SIGNAL(currentIndexChanged(const QString &)),
          m_chatCore    , SLOT  (slot_loadProfile   (const QString &)));
}
//\*****************************************************************************
void ChatWgt::slot_setCurrentProfileName(const QString & name)
{
  int idx;

  if((idx = m_profilesCmbx->findText(name)) >= 0)
    m_profilesCmbx->setCurrentIndex(idx);
  else
    m_profilesCmbx->addItem(name);

  if((idx = m_profilesCmbx->findText(name)) >= 0)
    m_profilesCmbx->setCurrentIndex(idx);
}

void ChatWgt::slot_reloadProfileData()
{
  if(!m_userInfoDlg->isHidden() && !m_userInfoDlg->readOnly())
    m_userInfoDlg->slot_loadInfo(Globals::info());

  if(!m_preferencesDlg->isHidden())
    m_preferencesDlg->init();

  ChatTextWgt::initSmiles(Globals::prefs()->smilesThemePath());
  mw_smiles->loadTheme(Globals::prefs()->smilesThemePath());

  for(int i = 0; i < m_channelsNum; i++)
    m_chatCore->slot_statusAnswer(mw_channels[i]->name(), Globals::prefs()->broadcast(), 0, 0, 1);
}

void ChatWgt::createActions()
{
  m_showSettingsAct      = new QAction(QIcon(":/images/userInfo.png"     ), "", this);
  m_showPreferencesAct   = new QAction(QIcon(":/images/preferences.png"  ), "", this);
  m_exitAct              = new QAction(QIcon(":/images/quit.png"         ), "", this);
  m_showSmilesAct        = new QAction(QIcon(":/images/smiles.png"       ), "", this);
  m_addChannelAct        = new QAction(QIcon(":/images/addtab.png"       ), "", this);
  m_delChannelAct        = new QAction(QIcon(":/images/deltab.png"       ), "", this);
  m_aboutAct             = new QAction(QIcon(":/images/help.png"         ), "", this);
  m_aboutQtAct           = new QAction(QIcon(":/images/help.png"         ), "", this);
  m_licenseAct           = new QAction(QIcon(":/images/help.png"         ), "", this);
  m_writeSettingsAct     = new QAction(QIcon(":/images/writeSettings.svg"), "", this);

  m_addProfileAct        = new QAction(QIcon(":/images/addProfile.svg")   , "", this);
  m_deleteProfileAct     = new QAction(QIcon(":/images/deleteProfile.svg"), "", this);
  m_renameProfileAct     = new QAction(QIcon(":/images/renameProfile.svg"), "", this);

  (m_translateEnAct      = new QAction(this))->setData("en");
  (m_translatePlAct      = new QAction(this))->setData("pl");
  (m_translateRuAct      = new QAction(this))->setData("ru");
  (m_translateUkAct      = new QAction(this))->setData("uk");

  (m_showMainTBarAct     = new QAction(this))->setCheckable(true);
  (m_showProfilesTBarAct = new QAction(this))->setCheckable(true);

  m_showSingleMessagesAct= new QAction(this);

  m_menuTranslations->addAction(m_translateEnAct);
  m_menuTranslations->addAction(m_translatePlAct);
  m_menuTranslations->addAction(m_translateRuAct);
  m_menuTranslations->addAction(m_translateUkAct);

  m_menuFile    ->addAction(m_addChannelAct);
  m_menuFile    ->addAction(m_delChannelAct);
  m_menuFile    ->addAction(m_exitAct);

  m_menuSettings->addAction(m_showSettingsAct);
  m_menuSettings->addAction(m_showPreferencesAct);
  m_menuSettings->addAction(m_writeSettingsAct);
  m_menuSettings->addMenu  (m_menuTranslations);
  m_menuSettings->addMenu  (m_menuToolbars);

  m_menuView    ->addAction(m_showSmilesAct);
  m_menuView    ->addAction(m_showSingleMessagesAct);

  m_menuHelp    ->addAction(m_aboutAct);
  m_menuHelp    ->addAction(m_aboutQtAct);
  m_menuHelp    ->addAction(m_licenseAct);

  m_menuToolbars->addAction(m_showMainTBarAct);
  m_menuToolbars->addAction(m_showProfilesTBarAct);

  m_menuBar     ->addMenu(m_menuFile);
  m_menuBar     ->addMenu(m_menuView);
  m_menuBar     ->addMenu(m_menuSettings);
  m_menuBar     ->addMenu(m_menuHelp);

  connect(m_showSettingsAct    , SIGNAL(triggered(bool)), this             , SLOT(slot_showSettings()));
  connect(m_showPreferencesAct , SIGNAL(triggered(bool)), this             , SLOT(slot_showPreferences()));
  connect(m_exitAct            , SIGNAL(triggered(bool)), this             , SLOT(slot_exit()));
  connect(m_showSmilesAct      , SIGNAL(triggered(bool)), this             , SLOT(slot_showSmiles()));
  connect(m_addChannelAct      , SIGNAL(triggered(bool)), m_addChannelDlg  , SLOT(getValues()));
  connect(m_delChannelAct      , SIGNAL(triggered(bool)), this             , SLOT(slot_delChannell()));
  connect(m_aboutAct           , SIGNAL(triggered(bool)), this             , SLOT(slot_about()));
  connect(m_aboutQtAct         , SIGNAL(triggered(bool)), this             , SLOT(slot_aboutQt()));
  connect(m_licenseAct         , SIGNAL(triggered(bool)), this             , SLOT(slot_license()));
  connect(m_writeSettingsAct   , SIGNAL(triggered(bool)), m_chatCore       , SLOT(slot_saveSettings()));
  connect(m_renameProfileAct   , SIGNAL(triggered(bool)), this             , SLOT(slot_editProfileName()));
  connect(m_addProfileAct      , SIGNAL(triggered(bool)), this             , SLOT(slot_addProfile()));
  connect(m_deleteProfileAct   , SIGNAL(triggered(bool)), this             , SLOT(slot_delProfile()));
  connect(m_showMainTBarAct    , SIGNAL(triggered(bool)), m_mainToolBar    , SLOT(setVisible(bool)));
  connect(m_showProfilesTBarAct, SIGNAL(triggered(bool)), m_profilesToolBar, SLOT(setVisible(bool)));
  connect(m_showSingleMessagesAct, SIGNAL(triggered(bool)), this           , SLOT(slot_showSingleMessagesHistory()));

  connect(m_translatePlAct     , SIGNAL(triggered(bool)), this             , SLOT(setLanguage()));
  connect(m_translateUkAct     , SIGNAL(triggered(bool)), this             , SLOT(setLanguage()));
  connect(m_translateRuAct     , SIGNAL(triggered(bool)), this             , SLOT(setLanguage()));
  connect(m_translateEnAct     , SIGNAL(triggered(bool)), this             , SLOT(setLanguage()));
}

void ChatWgt::createWidgets()
{
  m_profilesLab          = new QLabel(this);

  mw_tabs                = new QTabWidget (this);
  m_addChannelBtn        = new QToolButton(mw_tabs);
  m_delChannelBtn        = new QToolButton(mw_tabs);

  m_addChannelDlg        = new AddChannelDlg(this);
  m_userInfoDlg          = new EditUserInfoDlg (0);
  m_preferencesDlg       = new PreferencesDlg  (0);

  mw_log                 = new LogWgt(this);
  mw_smiles              = new SmilesWgt;
  m_trayIcon             = new QChatTrayIcon(QIcon(":/images/trayIcon.png"), this);
  m_profilesCmbx         = new QComboBox(this);

  m_menuBar              = new QMenuBar   (this);
  (m_mainToolBar         = new QToolBar   (this))->setObjectName("Main Toolbar");
  (m_profilesToolBar     = new QToolBar   (this))->setObjectName("Profiles Toolbar");
  m_menuFile             = new QMenu      (this);
  m_menuView             = new QMenu      (this);
  m_menuSettings         = new QMenu      (this);
  m_menuHelp             = new QMenu      (this);
  m_menuTranslations     = new QMenu      (this);
  m_menuToolbars         = new QMenu      (this);

  m_singleMessagesHistoryView = new SingleMsgsHistoryView;

  setStatusBar(new QStatusBar(this));
  statusBar()->setSizeGripEnabled(false);

  m_profilesCmbx->setMinimumWidth(150);

  connect(m_preferencesDlg, SIGNAL(portChanged  (int))            , m_chatCore, SLOT(slot_bindInputPort(int)));
  connect(m_preferencesDlg, SIGNAL(wantChangeSmileTheme (const QString &)),
          this            , SLOT  (slot_changeSmileTheme(const QString &)));

  connect(mw_smiles      , SIGNAL(smileClicked(const QString &)), this, SLOT(slot_insertSmile(const QString &)));
  connect(m_addChannelDlg, SIGNAL(dataAccepted(const QString &)), this, SLOT(slot_addChannell(const QString &)));
  connect(m_trayIcon     , SIGNAL(activated           (QSystemTrayIcon::ActivationReason)),
          this           , SLOT  (slot_trayIconClicked(QSystemTrayIcon::ActivationReason)));
  connect(m_preferencesDlg, SIGNAL(styleSheetChanged(QString)), this, SIGNAL(wantChangeStyleSheet(QString)));

  connect(m_singleMessagesHistoryView, SIGNAL(wantViewMessage(int)), this, SLOT(showSingleMessage(int)));
}

void ChatWgt::setupLayout()
{
  mw_tabs->setCornerWidget (m_addChannelBtn, Qt::TopLeftCorner);
  mw_tabs->setCornerWidget (m_delChannelBtn, Qt::TopRightCorner);

  m_addChannelBtn->setDefaultAction(m_addChannelAct);
  m_delChannelBtn->setDefaultAction(m_delChannelAct);

  m_mainToolBar->addAction(m_showSmilesAct);
  m_mainToolBar->addAction(m_showSettingsAct);
  m_mainToolBar->addAction(m_showPreferencesAct);
  m_mainToolBar->addAction(m_writeSettingsAct);

  m_profilesToolBar->addWidget(m_profilesLab);
  m_profilesToolBar->addAction(m_addProfileAct);
  m_profilesToolBar->addWidget(m_profilesCmbx);
  m_profilesToolBar->addAction(m_renameProfileAct);
  m_profilesToolBar->addAction(m_deleteProfileAct);

  setMenuBar(m_menuBar);
  addToolBar(m_mainToolBar);
  addToolBar(m_profilesToolBar);
  setCentralWidget(mw_tabs);
}

void ChatWgt::slot_editProfileName()
{
  bool ok;
  QString str = QInputDialog::getText(this,
                                      QString(tr("Input new name for '%1'")).arg(m_profilesCmbx->currentText()),
                                      tr("New name:"),
                                      QLineEdit::Normal,
                                      m_profilesCmbx->currentText(),
                                      &ok);

  if(ok && !str.isEmpty() && (m_profilesCmbx->findText(str) < 0) )
  {
    emit wantRenameProfile(m_profilesCmbx->currentText(), str);
    m_profilesCmbx->setItemText(m_profilesCmbx->currentIndex(), str);
  }
}

void ChatWgt::slot_addProfile()
{
  bool ok;
  QString str = QInputDialog::getText(this,
                                      tr("Input name for new profile"),
                                      tr("New profile name:"),
                                      QLineEdit::Normal,
                                      "",
                                      &ok);

  if(ok && !str.isEmpty() && (m_profilesCmbx->findText(str) < 0) )
  {
    emit wantLoadProfile(str);
    m_profilesCmbx->addItem(str);
    m_profilesCmbx->setCurrentIndex(m_profilesCmbx->findText(str));
  }
}

void ChatWgt::slot_delProfile()
{
  if(m_profilesCmbx->count() > 1)
  {
     QMessageBox* msgbx;
     int ans;
     msgbx = new QMessageBox(tr("Are you shure?"),
                             tr("Are you shure you want delete profile %1?").arg(m_profilesCmbx->currentText()),
                             QMessageBox::Question, QMessageBox::Yes, QMessageBox::No, 0, this, 0);
     ans = msgbx->exec();
     delete msgbx;

     if(ans == QMessageBox::No)
       return;

     emit wantDeleteProfile(m_profilesCmbx->currentText());
     m_profilesCmbx->removeItem(m_profilesCmbx->currentIndex());
  }
}

void ChatWgt::retranslate()
{
  m_showSettingsAct    ->setText(tr("&Edit User Details..."));
  m_showPreferencesAct ->setText(tr("&Preferences..."      ));
  m_exitAct            ->setText(tr("&Exit"                ));
  m_showSmilesAct      ->setText(tr("&Show Smiles.."       ));
  m_addChannelAct      ->setText(tr("&Add Channel.."       ));
  m_delChannelAct      ->setText(tr("&Delete Channel.."    ));
  m_aboutAct           ->setText(tr("&About"               ));
  m_aboutQtAct         ->setText(tr("About &Qt"            ));
  m_licenseAct         ->setText(tr("&License"             ));
  m_writeSettingsAct   ->setText(tr("&Write Settings"      ));
  m_showMainTBarAct    ->setText(tr("Main Toolbar"         ));
  m_showProfilesTBarAct->setText(tr("Profiles Toolbar"     ));
  m_addProfileAct      ->setText(tr("Add pro&file..."      ));
  m_deleteProfileAct   ->setText(tr("Delete p&rofile..."   ));
  m_renameProfileAct   ->setText(tr("&Rename profile..."   ));
  m_showSingleMessagesAct->setText(tr("Show Single Messages History..."));

  m_mainToolBar        ->setWindowTitle(tr("Main Toolbar"    ));
  m_profilesToolBar    ->setWindowTitle(tr("Profiles Toolbar"));
  m_singleMessagesHistoryView->setWindowTitle(tr("Single Messages History"));

  m_menuFile           ->setTitle(tr("&Chat"    ));
  m_menuView           ->setTitle(tr("&View"    ));
  m_menuSettings       ->setTitle(tr("&Settings"));
  m_menuHelp           ->setTitle(tr("&Help"    ));
  m_menuTranslations   ->setTitle(tr("&Language"));
  m_menuToolbars       ->setTitle(tr("Toolbars" ));

  m_translatePlAct     ->setText(tr("Polish"));
  m_translateUkAct     ->setText(tr("Ukrainian"));
  m_translateRuAct     ->setText(tr("Russian"));
  m_translateEnAct     ->setText(tr("English"));

  m_profilesLab        ->setText(tr("Current profile:"));
  m_showSingleMessagesAct->setShortcut(tr("Ctrl+M"    , "Single Messages History"));
  m_showSettingsAct    ->setShortcut(tr("Ctrl+D"      , "Show User Details"));
  m_showPreferencesAct ->setShortcut(tr("Ctrl+P"      , "Show Preferences"));
  m_exitAct            ->setShortcut(tr("Ctrl+Q"      , "Quit"));
  m_showSmilesAct      ->setShortcut(tr("Ctrl+S"      , "Show Smiles"));
  m_addChannelAct      ->setShortcut(tr("Ctrl+N"      , "Add Channel"));
  m_delChannelAct      ->setShortcut(tr("Ctrl+W"      , "Del Channel"));
//   m_aboutAct           ->setShortcut(tr("Ctrl+A"      , "About"));
  m_aboutQtAct         ->setShortcut(tr("Ctrl+Shift+A", "About Qt"));
  m_licenseAct         ->setShortcut(tr("Ctrl+L"      , "Show License"));
  m_writeSettingsAct   ->setShortcut(tr("Ctrl+Shift+S", "Write Settings"));

  m_addProfileAct      ->setShortcut(tr("Ctrl+Shift+N", "Add Profile"));
  m_deleteProfileAct   ->setShortcut(tr("Ctrl+Shift+D", "Del Profile"));
  m_renameProfileAct   ->setShortcut(tr("Ctrl+Shift+R", "Rename Profile"));

  m_translateUkAct     ->setShortcut(tr("Ctrl+Alt+U"  , "Set Ukrainian"));
  m_translatePlAct     ->setShortcut(tr("Ctrl+Alt+P"  , "Set Polish"));
  m_translateRuAct     ->setShortcut(tr("Ctrl+Alt+R"  , "Set Russian"));
  m_translateEnAct     ->setShortcut(tr("Ctrl+Alt+E"  , "Set English"));
  m_showMainTBarAct    ->setShortcut(tr("Ctrl+Shift+M", "Show Main Toolbar"));
  m_showProfilesTBarAct->setShortcut(tr("Ctrl+Shift+P", "Show Profiles Toolbar"));
}

void ChatWgt::setLanguage()
{
  QString lang = ((QAction*)sender())->data().toString();

  m_chatCore->setLang(lang);

#if defined(Q_OS_LINUX)
  QDir dir(QCoreApplication::applicationDirPath() + "/../share/qchat/translations");
#else
  QDir dir(QCoreApplication::applicationDirPath() + "/translations");
#endif

  if(!m_translator->load("qchat_" + m_chatCore->lang() + ".qm", Globals::prefs()->settingsDir()))
    if(!m_translator->load("qchat_" + m_chatCore->lang() + ".qm", dir.path()))
      if(!m_translator->load("qchat_" + m_chatCore->lang() + ".qm", "/usr/share/qchat/translations/"))
        m_translator->load("qchat_" + m_chatCore->lang() + ".qm", "/usr/local/share/qchat/translations/");

  QApplication::installTranslator(m_translator);

  retranslate();
}

void ChatWgt::slot_changeSmileTheme(const QString & path)
{
  qDebug("[ChatWgt::slot_changeSmileTheme]: path = %s", path.toLocal8Bit().data());

  if(path != Globals::prefs()->smilesThemePath())
  {
    Globals::prefs()->setSmilesThemePath(path);
    ChatTextWgt::initSmiles(path);
    mw_smiles->loadTheme(path);
  }
}

QMenu* ChatWgt::createPopupMenu()
{
  QMenu* menu = new QMenu(this);

  menu->addAction(m_showMainTBarAct);
  menu->addAction(m_showProfilesTBarAct);

  return menu;
}

void ChatWgt::restoreAndShow()
{
  QMessageBox::StandardButton ans;
  QString msg;

  restoreState(m_chatCore->state());

  if(!m_chatCore->geometry().isEmpty())
    restoreGeometry(m_chatCore->geometry());
  else
    resize(800, 600);

  show();

  m_showMainTBarAct    ->setChecked(m_mainToolBar    ->isVisible());
  m_showProfilesTBarAct->setChecked(m_profilesToolBar->isVisible());

  if(!m_chatCore->lang().isEmpty())
  {
#if defined(Q_OS_LINUX)
  QDir dir(QCoreApplication::applicationDirPath() + "/../share/qchat/translations");
#else
  QDir dir(QCoreApplication::applicationDirPath() + "/translations");
#endif

    if(!m_translator->load("qchat_" + m_chatCore->lang() + ".qm", Globals::prefs()->settingsDir()))
      if(!m_translator->load("qchat_" + m_chatCore->lang() + ".qm", dir.path()))
        if(!m_translator->load("qchat_" + m_chatCore->lang() + ".qm", "/usr/share/qchat/translations/"))
          m_translator->load("qchat_" + m_chatCore->lang() + ".qm", "/usr/local/share/qchat/translations/");

    QApplication::installTranslator(m_translator);
  }

  retranslate();

  if(m_chatCore->needCheckIp() == 1)
  {
    msg = tr("Your network settings don't corresponds any of existing network interfaces.\n"
             "The program may not work.\n\n"
             "Do you want to configure it now?");

    ans = QMessageBox::warning(this,
                               tr("Incorrect network settings"),
                               msg,
                               QMessageBox::Yes | QMessageBox::No);

    if(ans == QMessageBox::Yes)
      slot_showPreferences();
  }

  else if(m_chatCore->needCheckIp() == 2)
  {
    msg = tr("Couldn't find any network interface that can broadcast.\n"
             "The program may not work.\n\n"
             "Do you want to see your network settings?");

    ans = QMessageBox::warning(this,
                               tr("No valid network interface!"),
                               msg,
                               QMessageBox::Yes | QMessageBox::No);

    if(ans == QMessageBox::Yes)
      slot_showPreferences();
  }
}

void ChatWgt::slot_focusChanged(QWidget* /*old*/, QWidget* new_)
{
  m_trayIcon->setStaticIcon(":/images/trayIcon.png");

  if(!mw_smiles || !mw_smiles->inited())
    return;

  if(new_ && new_ != mw_smiles)
    mw_smiles->hide();
}

void ChatWgt::slot_processData(QC_DatagramHeader* Hdr)
{
  int chnnl;

  if(Hdr->chnnl_id == 1)
  {
    if(Hdr->src_ip == Globals::prefs()->ip() || Hdr->src_ip == 0x7f000001)
      chnnl = findChannel(QString().fromUtf8(ChatCore::getParametr("Channel", Hdr->parametrs)));
    else
      chnnl = findChannel(QHostAddress(Hdr->src_ip), 1);
  }
  else
    chnnl = findChannel(QString().fromUtf8(ChatCore::getParametr("Channel", Hdr->parametrs)));

  qDebug("[ChatWgt::slot_processData]: chnnl_num = %d\n", chnnl);

  if(chnnl >= 0)
    mw_channels[chnnl]->processData(Hdr);

  if(Hdr->type == Globals::INFO_ANSWER)
  {
    if(m_userInfoDlg->readOnly() && m_userInfoDlg->userIp() == Hdr->src_ip && !m_userInfoDlg->isHidden())
    {
       UserWgt* user = mw_channels[chnnl]->findUser(Hdr->src_ip);
       if(user)
         m_userInfoDlg->slot_loadInfo(user->info());
    }
  }

  delete Hdr;
}

void ChatWgt::activity()
{

}

void ChatWgt::closeEvent(QCloseEvent * ev)
{
  int res;
  MessageWithCheckBox* mes;
  QMessageBox* mbox;

  if(Globals::prefs()->warningAboutTray())
  {
    mbox = new QMessageBox;
    mbox->setIcon(QMessageBox::Information);
    mes = new MessageWithCheckBox(tr("Hiding in tray"),
                                  tr("Now QChat will be available from system tray.\n"
                                     "If you want to quit from QChat - use Exit action from menu Chat."),
                                  tr("Don't show this message again"), &res, this);
    mes->setIcon(mbox->iconPixmap());
    mes->exec();
    Globals::prefs()->setWarningAboutTray(res);

    delete mes;
    delete mbox;
  }

  ev->accept();
}

void ChatWgt::slot_showSingleMessagesHistory()
{
  m_singleMessagesHistoryView->setHidden(!m_singleMessagesHistoryView->isHidden());
}

void ChatWgt::showSingleMessage(int n)
{
  Message* msg = m_chatCore->m_singleMsgsHistory->message(n);

  if(msg)
  {
    SingleMessageWgt* smw = new SingleMessageWgt(msg, 1);
    connect(smw , SIGNAL(singleMessage   (const QString &, const QHostAddress &)),
            this, SIGNAL(singleMessageOut(const QString &, const QHostAddress &)));
  }
}
