# -*- coding: utf-8 -*-

# Copyright (C) 2010-2011 by Mike Gabriel <mike.gabriel@das-netzwerkteam.de>
#
# Python X2go is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Python X2go is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the
# Free Software Foundation, Inc.,
# 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.

"""\
X2goServerSessionList and X2goServerSessionInfo classes - data handling for 
X2go server sessions.

This backend handles X2go server implementations that respond with session infos 
via server-side STDOUT.

"""
__NAME__ = 'x2goserversessioninfo-pylib'

class X2goServerSessionInfoSTDOUT(object):
    """\
    L{X2goServerSessionInfo} is used to store all information
    that is retrieved from the connected X2go server on 
    C{X2goTerminalSessionBACKEND.start()} resp. C{X2goTerminalSessionBACKEND.resume()}.

    """
    def __str__(self):
        return self.name
    def __repr__(self):
        return "<%s instance: %s>" % (self.__class__, self.name)

    def _parse_x2golistsessions_line(self, x2go_output):
        """\
        Parse a single line of X2go's listsessions output.

        """
        try:
            l = x2go_output.split("|")
            self.name = l[1]
            self.cookie = l[6]
            self.agent_pid = int(l[0])
            self.display = int(l[2])
            self.status = l[4]
            self.graphics_port = int(l[8])
            self.snd_port = int(l[9])
            self.sshfs_port = int(l[13])
            self.username = l[11]
            self.hostname = l[3]
            # TODO: turn into datetime object
            self.date_created = l[5]
            # TODO: turn into datetime object
            self.date_suspended = l[10]
            self.local_container = ''
        except IndexError, e:
            # DEBUGGING CODE
            print 'Encountered IndexError: %s' % str(e)
            print 'THIS SHOULD NOT HAPPEN... HERE IS THE x2golistsessions OUTPUT THAT CAUSED THE ERROR...'
            print x2go_output
            raise e
        except ValueError, e:
            # DEBUGGING CODE
            print 'Encountered IndexError: %s' % str(e)
            print 'THIS SHOULD NOT HAPPEN... HERE IS THE x2golistsessions OUTPUT THAT CAUSED THE ERROR...'
            print x2go_output
            raise e

    def is_running(self):

        return self.status == 'R'

    def is_suspended(self):

        return self.status == 'S'

    def _parse_x2gostartagent_output(self, x2go_output):
        """\
        Parse x2gostartagent output.

        """
        l = x2go_output.split("\n")
        self.name = l[3]
        self.cookie = l[1]
        self.agent_pid = int(l[2])
        self.display = int(l[0])
        self.graphics_port = int(l[4])
        self.snd_port = int(l[5])
        self.sshfs_port = int(l[6])
        self.username = ''
        self.hostname = ''
        # TODO: we have to see how we fill these fields here...
        self.date_created = ''
        self.date_suspended = ''
        # TODO: presume session is running after x2gostartagent, this could be better
        self.status = 'R'
        self.local_container = ''
        self.remote_container = ''

    def initialize(self, x2go_output, username='', hostname='', local_container='', remote_container=''):
        """\
        Parse X2go server's C{x2gostartagent} stdout values.

        @param x2go_output: X2go server's C{x2gostartagent} command output, each value 
            separated by a newline character.
        @type x2go_output: str
        @param username: session user name
        @type username: str
        @param hostname: hostname of X2go server
        @type hostname: str
        @param local_container: X2go client session directory for config files, cache and session logs
        @type local_container: str
        @param remote_container: X2go server session directory for config files, cache and session logs
        @type remote_container: str

        """
        self._parse_x2gostartagent_output(x2go_output)
        self.username = username
        self.hostname = hostname
        self.local_container = local_container
        self.remote_container = remote_container

    def clear(self):
        """\
        Clear all properties of a L{X2goServerSessionInfo} object.

        """
        self.name = ''
        self.cookie = ''
        self.agent_pid = ''
        self.display = ''
        self.graphics_port = ''
        self.snd_port = ''
        self.sshfs_port = ''
        self.username = ''
        self.hostname = ''
        self.date_created = ''
        self.date_suspended = ''
        self.status = ''
        self.local_container = ''
        self.remote_container = ''

    __init__ = clear


class X2goServerSessionListSTDOUT(object):
    """\
    L{X2goServerSessionListSTDOUT} is used to store all information
    that is retrieved from a connected X2go server on a
    C{X2goControlSessionBACKEND.list_sessions()} call.

    """
    def __init__(self, x2go_output, info_backend=X2goServerSessionInfoSTDOUT):
        """\
        @param x2go_output: X2go server's C{x2golistsessions} command output, each 
            session separated by a newline character. Session values are separated 
            by Unix Pipe Symbols ('|')
        @type x2go_output: str

        """
        self.sessions = {}
        lines = x2go_output.split("\n")
        for line in lines:
            if not line:
                continue
            s_info = info_backend()
            s_info._parse_x2golistsessions_line(line)
            self.sessions[s_info.name] = s_info

    def __call__(self):
        return self.sessions

    def get_session_info(self, session_name):
        """\
        STILL UNDOCUMENTED

        """
        try:
            return self.sessions[session_name]
        except KeyError:
            return None
