/*
 * Copyright (c) 2006 Bea Lam. All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

//
//  BBOBEXServerSession.m
//  BTUtil
//  
//  This does not handle the configuration of Target and Connection ID headers. 
//  But, it would be simple to add this support if necessary -- all you need to 
//  do is read the Connection ID header for each request. I have excluded this 
//  because I'm not sure about the benefits of this balanced against the need to
//  add an extra ConnectionID argument to every request-type delegate method
//  (to inform the delegate of the ConnectionID used for a request). Feedback
//  and comments about this would be very welcome.
//	
//	Anyway, this does not mean a BBOBEXServerSession cannot receive Target 
//  headers. It just means that the delegate can't send a response back to the 
//  client which contains appropriate Who and ConnectionID headers, and 
//  according to the IrOBEX specs, this means that the client should realise
//  it's connected to the generic 'Inbox' service rather that to a specific
//  target service. So a BBOBEXServerSession should be perfectly able to handle
//  services such as the File Transfer Service which require clients to send
//  an appropriate Target header.
//
//  The obexServerError:error:summary: callback is not called if an error 
//  occurs in a method that returns an error code (i.e. the
//  runWithIncomingRFCOMMChannel and close methods). It would be confusing if 
//  the error callback was called as this would be before the error code was 
//  returned and checked by the programmer.
//
//  (Should actually check whether a set of headers all fit into one packet)
//
//  (This would be much better if it was object-oriented.)
//

#import "BBOBEXUtil.h"
#import "BBOBEXServer.h"
#import "BBOBEXServerSession.h"

#define DEBUG_NAME @"[BBOBEXServerSession] "


static BOOL debug;

static SEL callbackCommandReceived;
static SEL callbackServerConnectRequest, callbackServerConnectComplete,
	callbackServerDisconnectComplete, 
	callbackServerPutRequest, callbackServerPutProgress, callbackServerPutComplete,
	callbackServerGetRequest, callbackServerGetProgress, callbackServerGetComplete,
	callbackServerSetPathRequest, callbackServerSetPathComplete,
	callbackServerAbortRequest, callbackServerAbortComplete, 
	callbackServerError;	


@implementation BBOBEXServerSession

+ (void)initialize 
{
	debug = NO;	
	
	callbackCommandReceived = @selector(handleSessionEvent:);	
	
	callbackServerConnectRequest = @selector(obexServerSessionConnectRequest:);
	callbackServerConnectComplete = @selector(obexServerSessionConnectComplete:);
	
	callbackServerDisconnectComplete = @selector(obexServerSessionDisconnectComplete:);
	
	callbackServerPutRequest = @selector(obexServerSessionPutRequest:withName:type:length:writeToFile:);
	callbackServerPutProgress = @selector(obexServerSessionPutProgress:forFile:byAmount:);
	callbackServerPutComplete = @selector(obexServerSessionPutComplete:);
	
	callbackServerGetRequest = @selector(obexServerSessionGetRequest:withName:type:readFromFile:);
	callbackServerGetProgress = @selector(obexServerSessionGetProgress:forFile:byAmount:);
	callbackServerGetComplete = @selector(obexServerSessionGetComplete:);	
	
	callbackServerSetPathRequest = @selector(obexServerSessionSetPathRequest:forPath:withFlags:);	
	callbackServerSetPathComplete = @selector(obexServerSessionSetPathComplete:);
	
	callbackServerAbortRequest =  @selector(obexServerSessionAbortRequest:forOperation:);
	callbackServerAbortComplete = @selector(obexServerSessionAbortComplete:);
	
	callbackServerError = @selector(obexServerSessionError:error:summary:);			
}

- (id)initWithServer:(BBOBEXServer *)server
{
	self = [super init];
	
	mRunning = NO;
	
	// don't retain, cos BBOBEXServer retains BBOBEXServerSessions, and it 
	// would be confusing if they retain each other
	mOBEXServer = server;	
	
	mOBEXSession = nil;
	mMaxPacketLength = 0x2000;	// Apple OBEXSample's initial value for max packet length
	mCurrentOp = 0;		
	
	return self;
}

- (id)init
{
	return [self initWithServer:nil];
}

- (id)delegate
{
	return [mOBEXServer delegate];
}


#pragma mark -
#pragma mark errors/debug



- (void)reportError:(OBEXError)error summary:(NSString *)summary
{
	if (summary == nil) summary = @"";
	
	if (debug) {
		NSString *errDesc = 
			[NSString stringWithFormat:@"%@ (errno=%d)", summary, error];		
		NSLog([DEBUG_NAME stringByAppendingString:errDesc]);
	}
	
	if ([[self delegate] respondsToSelector:callbackServerError]) {
		[[self delegate] obexServerSessionError:self
							 error:error
						   summary:summary];
	}
}


#pragma mark -
#pragma mark Tell delegate about stuff

- (int)notifyConnectRequest
{
	if ([[self delegate] respondsToSelector:callbackServerConnectRequest]) {
		return [[self delegate] obexServerSessionConnectRequest:self];
	}
	// if method not implemented, accept the connection
	return kOBEXSuccess;
}

- (void)notifyConnectSuccess
{
	if ([[self delegate] respondsToSelector:callbackServerConnectComplete]) {
		[[self delegate] obexServerSessionConnectComplete:self];
	}
}

- (void)notifyDisconnectSuccess
{
	if ([[self delegate] respondsToSelector:callbackServerDisconnectComplete]) {
		[[self delegate] obexServerSessionDisconnectComplete:self];
	}
}

- (int)notifyPutRequestWithName:(NSString *)name
						   type:(NSString *)type
						 length:(unsigned)fileLength
{
	if ([[self delegate] respondsToSelector:callbackServerPutRequest]) {
		NSFileHandle *handle = nil;
		int response = [[self delegate] obexServerSessionPutRequest:self
														   withName:name
															   type:type
															 length:fileLength 
														writeToFile:&handle];
		// keep the new file handle
		[handle retain];
		[mPutFileHandle release];
		mPutFileHandle = handle;
		
		return response;
	}
	return kOBEXResponseCodeNotImplementedWithFinalBit;
}


- (int)notifyPutProgressByAmount:(unsigned)amount
{
	if ([[self delegate] respondsToSelector:callbackServerPutProgress]) {
		return [[self delegate] obexServerSessionPutProgress:self
													 forFile:mPutFileHandle
													byAmount:amount];
	}
	// if delegate doesn't implement method, it doesn't matter, keep the 
	// PUT going
	return kOBEXSuccess;
}

- (void)notifyPutSuccess
{
	if ([[self delegate] respondsToSelector:callbackServerPutComplete]) {
		[[self delegate] obexServerSessionPutComplete:self
											  forFile:mPutFileHandle];
	}
}


- (int)notifyGetRequestWithName:(NSString *)name
						   type:(NSString *)type
{
	if ([[self delegate] respondsToSelector:callbackServerGetRequest]) {
		NSFileHandle *handle = nil;
		int response = [[self delegate] obexServerSessionGetRequest:self
														   withName:name
															   type:type
													   readFromFile:&handle];
		// keep the new file handle
		[handle retain];
		[mGetFileHandle release];
		mGetFileHandle = handle;
		
		return response;	
	}
	// if delegate doesn't implement method, disallow the Get
	return kOBEXResponseCodeNotImplementedWithFinalBit;
}

- (int)notifyGetProgressByAmount:(unsigned)amount
{
	if ([[self delegate] respondsToSelector:callbackServerGetProgress]) {
		return [[self delegate] obexServerSessionGetProgress:self
													 forFile:mGetFileHandle
													byAmount:amount];
	}
	// if delegate doesn't implement method, it doesn't matter, keep the 
	// Get going
	return kOBEXSuccess;
}

- (void)notifyGetSuccess
{
	if ([[self delegate] respondsToSelector:callbackServerGetComplete]) {
		[[self delegate] obexServerSessionGetComplete:self
											  forFile:mGetFileHandle];
	}
}

- (int)notifySetPathRequestWithName:(NSString *)name
							  flags:(OBEXFlags)flags
{
	if ([[self delegate] respondsToSelector:callbackServerSetPathRequest]) {
		return [[self delegate] obexServerSessionSetPathRequest:self
														forPath:name 
													  withFlags:flags];
	}
	// if delegate doesn't implement method, disallow the SetPath
	return kOBEXResponseCodeNotImplementedWithFinalBit;
}

- (void)notifySetPathSuccess
{
	if ([[self delegate] respondsToSelector:callbackServerSetPathComplete]) {
		[[self delegate] obexServerSessionSetPathComplete:self];
	}
}

- (int)notifyAbortRequestForOp:(OBEXOpCode)opCode
{
	if ([[self delegate] respondsToSelector:callbackServerAbortRequest]) {
		return [[self delegate] obexServerSessionAbortRequest:self
												 forOperation:opCode];
	}
	// if delegate doesn't implement method, disallow the Abort
	return kOBEXResponseCodeNotImplementedWithFinalBit;
}

- (void)notifyAbortSuccess
{
	if ([[self delegate] respondsToSelector:callbackServerAbortComplete]) {
		[[self delegate] obexServerSessionAbortComplete:self];
	}
}

// unlike client side, this is only used for multi-packet ops (i.e. PUT, GET)
- (void)startedOp:(OBEXOpCode)opCode
{
	mCurrentOp = opCode;
}


- (void)completedCurrentOp
{
	mCurrentOp = 0;
}


#pragma mark -

- (OBEXError)runWithIncomingRFCOMMChannel:(IOBluetoothRFCOMMChannel *)channel
{
	if (debug) NSLog(DEBUG_NAME @"[runWithIncomingRFCOMMChannel] entry.");
	
	// do nothing if not attached to a server
	if (!mOBEXServer) {
		if (debug) NSLog(@"Error, internal BBOBEXServer not set, should have been given on init");
		return kOBEXInternalError;
	}
	
	// do nothing if already started.
	if (mOBEXSession) {
		if (debug) NSLog(@"Error, session already running");
		return kOBEXSessionAlreadyConnectedError;
	}
	
	// make obex session
	mOBEXSession = [[IOBluetoothOBEXSession alloc] initWithIncomingRFCOMMChannel:channel
																   eventSelector:callbackCommandReceived
																  selectorTarget:self
																		  refCon:NULL];	
	if (!mOBEXSession) {
		if (debug) NSLog(@"Error creating OBEXSession");
		return kOBEXNoResourcesError;
	}
	
	// set the obex session to handle events on channel
	OBEXError status = [channel setDelegate:mOBEXSession];
	if (status != kIOReturnSuccess) {
		if (debug) NSLog(@"Error setting OBEXSession as delegate of RFCOMMChannel");
		[mOBEXSession release];
		mOBEXSession = nil;		
		return status;
	}
	
	mRunning = YES;
	if (debug) NSLog(DEBUG_NAME @"[runWithIncomingRFCOMMChannel] Starting session...");
	
	return kOBEXSuccess;
}

- (BOOL)isRunning
{
	return mRunning;
}

- (OBEXError)close
{
	mRunning = NO;
	
	if (mOBEXSession) {
		OBEXError status = kOBEXSuccess;
		
		// This always seem to get an kOBEXSessionNoTransportError error when 
		// closing the transport connection, which is odd since 
		// hasOpenTransportConnection returns YES.
		if ([mOBEXSession hasOpenTransportConnection]) {
			status = [mOBEXSession closeTransportConnection];
		}		
		
		// must set the event selector and target to NULL, otherwise the
		// mOBEXSession might continue to try to send us events (e.g. if there's
		// a link error)
		[mOBEXSession setEventSelector:NULL
								target:nil
								refCon:NULL];
		
		[mOBEXSession release];
		mOBEXSession = nil;		
		
		return status;
	}
	
	return kOBEXSessionNotConnectedError;
}

- (void)setMaxPacketLength:(OBEXMaxPacketLength)length
{
	mMaxPacketLength = length;
}

- (BOOL)hasOpenOBEXConnection
{
	if (mOBEXSession) {
		return [mOBEXSession hasOpenOBEXConnection];
	}
	return NO;
}

- (BOOL)hasOpenTransportConnection
{
	if (mOBEXSession) {
		return [mOBEXSession hasOpenTransportConnection];
	}
	return NO;
}

- (IOBluetoothRFCOMMChannel *)getRFCOMMChannel
{
	if (mOBEXSession) {
		return [mOBEXSession getRFCOMMChannel];
	} 
	
	return nil;
}

- (BBOBEXServer *)getOBEXServer
{
	return mOBEXServer;
}

+ (void)setDebug:(BOOL)doDebug
{
	debug = doDebug;
}


#pragma mark -
#pragma mark Connect

- (BOOL)sendConnectResponse:(OBEXOpCode)responseCode
				withHeaders:(CFMutableDataRef)headersDataRef
{
	void *headers = (headersDataRef)? (void*)CFDataGetBytePtr(headersDataRef) : NULL;
	size_t headersLength = (headersDataRef)? CFDataGetLength(headersDataRef): 0;	
	
	OBEXError status;
	status = [mOBEXSession OBEXConnectResponse:responseCode
										 flags:(OBEXFlags)kOBEXConnectFlagNone
							   maxPacketLength:mMaxPacketLength
							   optionalHeaders:headers
						 optionalHeadersLength:headersLength
								 eventSelector:callbackCommandReceived
								selectorTarget:self
										refCon:NULL];
	
	if (status != kOBEXSuccess) {
		[self reportError:status summary:@"OBEXConnectResponse failed"];
		return NO;
	}
	return YES;
}


- (void)handleConnectCommand:(const OBEXConnectCommandData*)eventData
{
	if (debug) NSLog(DEBUG_NAME @"[handleConnectCommand] entry.");
	
	// set max packet length to the client's max packet length
	mMaxPacketLength = eventData->maxPacketSize;
	if (debug) NSLog(DEBUG_NAME @"Negotiated max packet length: %d", mMaxPacketLength);
	
	// notify delegate
	int delegateReply = [self notifyConnectRequest];
	
	// prepare and send the response
	
	OBEXOpCode responseCode = (delegateReply == kOBEXSuccess)? 
		kOBEXResponseCodeSuccessWithFinalBit : delegateReply;
	
	if (debug) NSLog(DEBUG_NAME "Finish CONNECT, respond '%@' (0x%02x)", 
					 [BBOBEXUtil describeServerResponse:responseCode], responseCode);				
	
	// send the response (don't send any response headers)
	BOOL responseSent = [self sendConnectResponse:responseCode withHeaders:NULL];
	
	if (responseSent && delegateReply == kOBEXSuccess) {
		[self notifyConnectSuccess];
	} else {
		[mOBEXSession release];
		mOBEXSession = nil;				
	}
}


#pragma mark -
#pragma mark Disconnect


- (void)handleDisconnectCommand:(const OBEXDisconnectCommandData*)eventData
{
	if (debug) NSLog(DEBUG_NAME @"[handleDisconnectCommand] entry.");
	
	// send OK response to the client (disconnect should always be allowed)
	OBEXError status;
	status = [mOBEXSession OBEXDisconnectResponse:kOBEXResponseCodeSuccessWithFinalBit
								  optionalHeaders:NULL
							optionalHeadersLength:(size_t)0
									eventSelector:callbackCommandReceived
								   selectorTarget:self
										   refCon:NULL];
	
	if (status == kOBEXSuccess) {
		if (debug) NSLog(DEBUG_NAME @"[handleDisconnectCommand] sent response OK.");
				
		// notify the delegate
		[self notifyDisconnectSuccess];
		
	} else {
		[self reportError:status summary:@"OBEXDisconnectResponse failed"];
	}
	
	// close this session
	[self close];
}



#pragma mark -
#pragma mark Put

- (void)cleanupPut
{
	[mPutFileHandle release];
	mPutFileHandle = nil;
}

- (void)completedPut:(BOOL)successful
{
	[self completedCurrentOp];	
	
	if (successful) {
		[self notifyPutSuccess];
	}
	
	[self cleanupPut];	
}


- (void)clientAbortedPut
{
	// should I truncate the file handle? probably not, leave to delegate.
	
	[self completedPut:NO];
}


- (BOOL)sendPutResponse:(OBEXOpCode)responseCode
{	
	OBEXError status;
	status = [mOBEXSession OBEXPutResponse:responseCode
						   optionalHeaders:NULL
					 optionalHeadersLength:(size_t)0
							 eventSelector:callbackCommandReceived 
							selectorTarget:self
									refCon:NULL];	
	
	if (status != kOBEXSuccess) {
		[self reportError:status summary:@"OBEXPutResponse failed"];
		return NO;
	}
	return YES;
}

- (int)receivedPutData:(CFDataRef)dataRef;
{
	if (debug) NSLog(DEBUG_NAME "[receivedPutData] Entry\n");

	if (dataRef) {
		NSData *data = [NSData dataWithData:(NSData *)dataRef];
			
		// write data to the file
		[mPutFileHandle writeData:data];
			
		// notify the delegate
		return [self notifyPutProgressByAmount:[data length]];
	}
	return kOBEXSuccess;
}


// This will try to read the significant headers and ask delegate 
// whether to continue.
// Returns delegate reply.
- (int)queryDelegateForPutRequest:(CFDictionaryRef)headersDictRef
{
	if (debug) NSLog(DEBUG_NAME "[queryDelegateForPutRequest] Entry\n");	
	
	CFStringRef nameStringRef = NULL;	
	CFStringRef typeStringRef = NULL;	
	size_t bodyLength = -1;	
	long connID = -1;
		
	if (headersDictRef) {
		
		/*
		if (debug) {
			NSLog(@"PUT request: <<<<<<");
			CFShow(headersDictRef);
		}*/
		
		// read name header
		if ((CFDictionaryGetCountOfKey(headersDictRef, kOBEXHeaderIDKeyName) > 0)) {
			nameStringRef = (CFStringRef)CFDictionaryGetValue(headersDictRef, kOBEXHeaderIDKeyName);
		}
		
		// read type header
		if ((CFDictionaryGetCountOfKey(headersDictRef, kOBEXHeaderIDKeyType) > 0)) {
			typeStringRef = (CFStringRef)CFDictionaryGetValue(headersDictRef, kOBEXHeaderIDKeyType);
		}	
		
		// read length header
		if (CFDictionaryGetCountOfKey(headersDictRef, kOBEXHeaderIDKeyLength) > 0) {
			// don't need to release this stringref, will be released when 
			// dictionary is destroyed
			/*
			CFStringRef theStringRef = 
				(CFStringRef) CFDictionaryGetValue(headersDictRef, kOBEXHeaderIDKeyLength);
			if (theStringRef) {				
				*/
				CFDataRef dataRef =
					(CFDataRef)CFDictionaryGetValue(headersDictRef, kOBEXHeaderIDKeyLength);
				if (dataRef) {
					CFDataGetBytes(dataRef, CFRangeMake(0, 4), (uint8_t*)&bodyLength);
				}
			//}
		}
	}
	
	NSString *name = (nameStringRef) ? (NSString *)nameStringRef : nil;
	NSString *type = (typeStringRef) ? (NSString *)typeStringRef : nil;
	
	if (debug) {
		NSLog(DEBUG_NAME "Incoming file name is: %@\n", name);
		NSLog(DEBUG_NAME "Incoming file type is: %@\n", type);		
		NSLog(DEBUG_NAME "Incoming file length is: %d\n", (int)bodyLength);	
	}
	
	return [self notifyPutRequestWithName:name
									 type:type
								   length:bodyLength];
}
	
		
- (int)processPutBodyData:(CFDictionaryRef)headersDictRef
{
	if (headersDictRef) {
		
		// read body data
		CFDataRef bodyDataRef;
		if (CFDictionaryGetCountOfKey(headersDictRef, kOBEXHeaderIDKeyBody) > 0) {	
			bodyDataRef = 
				(CFDataRef)CFDictionaryGetValue(headersDictRef, kOBEXHeaderIDKeyBody);
			if (bodyDataRef) {
				int delegateReply = [self receivedPutData:bodyDataRef];
				if (delegateReply != kOBEXSuccess)
					return delegateReply;
			}
		}	
		
		// read end-of-body data
		CFDataRef endOfBodyDataRef;
		if (CFDictionaryGetCountOfKey(headersDictRef, kOBEXHeaderIDKeyEndOfBody) > 0) {	
			endOfBodyDataRef = 
				(CFDataRef)CFDictionaryGetValue(headersDictRef, kOBEXHeaderIDKeyEndOfBody);
			if (endOfBodyDataRef) {
				int delegateReply = [self receivedPutData:endOfBodyDataRef];
				if (delegateReply != kOBEXSuccess)
					return delegateReply;
			}
		}	
		
	}
	return kOBEXSuccess;
}


- (void)handlePutCommand:(const OBEXPutCommandData*)eventData
{
	if (debug) NSLog(DEBUG_NAME "[handlePutCommand] Entry\n");	
	
	BOOL isFirstPacket = NO;
	
	// set current op
	if (mCurrentOp != kOBEXOpCodePut) {
		isFirstPacket = YES;
		if (debug) NSLog(DEBUG_NAME @"This is the first PUT packet");
	}
	[self startedOp:kOBEXOpCodePut];
	
	// get dictionary of header data
	CFDictionaryRef headersDictRef = 
		OBEXGetHeaders(eventData->headerDataPtr, eventData->headerDataLength);	
	
	if (isFirstPacket) {
		int delegateReply = [self queryDelegateForPutRequest:headersDictRef];
		
		if (delegateReply != kOBEXSuccess || mPutFileHandle == nil) {

			// If delegate gave a non-success response code, use it.
			// Otherwise if delegate did not provide a file handle, respond 
			// 'Internal error'
			OBEXOpCode responseCode;
			if (delegateReply != kOBEXSuccess) {
				responseCode = delegateReply;
			} else {
				responseCode = kOBEXResponseCodeInternalServerErrorWithFinalBit;
				[self reportError:kOBEXBadArgumentError 
						  summary:@"Error: delegate returned kOBEXSuccess but did not provide file handle for received data"];
			}			
			
			if (debug) NSLog(DEBUG_NAME "Finish PUT, refusing PUT with response '%@'", [BBOBEXUtil describeServerResponse:responseCode]);
			[self sendPutResponse:responseCode];
			[self completedPut:NO];
			
			if (headersDictRef) 
				CFRelease(headersDictRef);			
			return;
		}	
	}
	
	// read the body data (delegate will be asked whether to continue the PUT)
	int delegateReply = [self processPutBodyData:headersDictRef];
	
	if (delegateReply != kOBEXSuccess) {
		// delegate wants to stop the PUT
		
		if (debug) NSLog(@"Delegate wants to stop PUT, respond %@", [BBOBEXUtil describeServerResponse:delegateReply]);
		[self sendPutResponse:delegateReply];
		[self completedPut:NO];		
		
	} else {
		// continue PUT
		
		if (!headersDictRef || CFDictionaryGetCountOfKey(headersDictRef, kOBEXHeaderIDKeyEndOfBody) > 0) {
			// this is the last packet, finish the PUT operation		
			if (debug) NSLog(DEBUG_NAME "Finish PUT, respond kOBEXResponseCodeSuccessWithFinalBit");
			if ([self sendPutResponse:kOBEXResponseCodeSuccessWithFinalBit]) {
				[self completedPut:YES];
			}
			
		} else {
			// continue with the PUT
			if (debug) NSLog(DEBUG_NAME "Continue PUT, respond kOBEXResponseCodeContinueWithFinalBit");
			[self sendPutResponse:kOBEXResponseCodeContinueWithFinalBit];
		}
	}
	
	if (headersDictRef)
		CFRelease(headersDictRef);
}


#pragma mark -
#pragma mark Get

- (void)cleanupGet
{
	if (mGetHeadersDataRef) {
		CFRelease(mGetHeadersDataRef);
		mGetHeadersDataRef = NULL;
	}
	
	[mCurrentGetBodyHandler release];
	mCurrentGetBodyHandler = nil;	
	
	[mGetFileHandle release];
	mGetFileHandle = nil;
}

- (void)completedGet:(BOOL)successful
{
	[self completedCurrentOp];
	
	if (successful) {
		[self notifyGetSuccess];
	}
	
	[self cleanupGet];	
}

- (void)clientAbortedGet
{
	// should I truncate the file handle? no, probably leave abort handling
	// to delegate.
	
	[self completedGet:NO];
}


- (BOOL)sendGetResponse:(OBEXOpCode)responseCode
			withHeaders:(CFMutableDataRef)headersDataRef
{
	void *headers = (headersDataRef)? (void*)CFDataGetBytePtr(headersDataRef) : NULL;
	size_t headersLength = (headersDataRef)? CFDataGetLength(headersDataRef): 0;		
	
	NSLog(@"sendGetResponse headersLength %d", headersLength);
	
	OBEXError status;
	status = [mOBEXSession OBEXGetResponse:responseCode
						   optionalHeaders:headers
					 optionalHeadersLength:headersLength
							 eventSelector:callbackCommandReceived
							selectorTarget:self
									refCon:NULL];
	
	if (status != kOBEXSuccess) {
		[self reportError:status summary:@"OBEXGetResponse failed"];
		return NO;
	}
	return YES;
}


// Returned data ref must be released (but better not until response is complete).
// Returns NULL if error packing headers.
- (CFMutableDataRef)buildGetResponseHeaders:(BOOL*)outIsLastPacket
								 bodyLength:(unsigned*)outBodyLength
{
	OBEXError status;
	
	// pack the response headers. 
	CFMutableDictionaryRef respHeadersDictRef = CFDictionaryCreateMutable(NULL, 1, &kCFTypeDictionaryKeyCallBacks, &kCFTypeDictionaryValueCallBacks);
	if (!respHeadersDictRef) return NULL;
	
	// the GET headers seem to need 3 bytes padding, maybe for the response code
	// plus packet length in the headers. 
	NSData *bodyData = [mCurrentGetBodyHandler readNextChunkForHeaderLength:3
																	   forOp:kOBEXOpCodeGet
																 isLastChunk:outIsLastPacket];	
	if (!bodyData) {
		return NULL;
	}
	*outBodyLength = [bodyData length];	
	
	// add body header
	status = OBEXAddBodyHeader([bodyData bytes], [bodyData length], *outIsLastPacket,
							   respHeadersDictRef);
	if (status < 0) return NULL;
	
	//NSLog(@"GET response: >>>>>>");
	//CFShow(respHeadersDictRef);

	// get headers as bytes
	CFMutableDataRef headersDataRef = OBEXHeadersToBytes(respHeadersDictRef);
	CFRelease(respHeadersDictRef);	// done with headers dict, release
	
	return headersDataRef;
}


// This will try to read the significant headers and ask delegate 
// whether to continue.
// Returns delegate reply.
- (int)queryDelegateForGetRequest:(CFDictionaryRef)reqHeadersDictRef
{
	if (debug) NSLog(DEBUG_NAME "[queryDelegateForGetRequest] Entry\n");	
	
	CFStringRef nameStringRef = NULL;	
	CFStringRef typeStringRef = NULL;	
	
	if (reqHeadersDictRef) {
		// read name header
		if ((CFDictionaryGetCountOfKey(reqHeadersDictRef, kOBEXHeaderIDKeyName) > 0)) {
			nameStringRef = (CFStringRef)CFDictionaryGetValue(reqHeadersDictRef, kOBEXHeaderIDKeyName);
		}
		
		// read type header
		if ((CFDictionaryGetCountOfKey(reqHeadersDictRef, kOBEXHeaderIDKeyType) > 0)) {
			typeStringRef = (CFStringRef)CFDictionaryGetValue(reqHeadersDictRef, kOBEXHeaderIDKeyType);
		}
	}
	
	NSString *name = (nameStringRef) ? (NSString *)nameStringRef : nil;
	NSString *type = (typeStringRef) ? (NSString *)typeStringRef : nil;
	
	if (debug) {
		NSLog(DEBUG_NAME "Requested file name is: %@\n", name);
		NSLog(DEBUG_NAME "Requested file type is: %@\n", type);
	}
	
	if (name == nil && type == nil) {
		if (debug) NSLog(DEBUG_NAME @"Client did not provide name nor type, refusing GET");
		return kOBEXResponseCodeBadRequestWithFinalBit;
	}

	return [self notifyGetRequestWithName:name
									 type:type];
}


- (void)handleGetCommand:(const OBEXGetCommandData*)eventData
{
	if (debug) NSLog(DEBUG_NAME "[handleGetCommand] Entry\n");	
	
	BOOL isFirstPacket = NO;
	
	// set current op
	if (mCurrentOp != kOBEXOpCodeGet) {
		isFirstPacket = YES;
		NSLog(DEBUG_NAME @"This is the first GET packet");
	}
	[self startedOp:kOBEXOpCodeGet];
	
	// get dictionary of request header data
	CFDictionaryRef reqHeadersDictRef = 
		OBEXGetHeaders(eventData->headerDataPtr, eventData->headerDataLength);	
	
	/*
	if (reqHeadersDictRef) {
		if (debug) {
			NSLog(@"GET request: <<<<<<");
			CFShow(reqHeadersDictRef);	
		}
	}
	 */
	
	if (isFirstPacket) {
		int delegateReply = [self queryDelegateForGetRequest:reqHeadersDictRef];
				
		if (delegateReply != kOBEXSuccess || mGetFileHandle == nil) {
			
			// If delegate gave a non-success response code, use it.
			// Otherwise if delegate did not provide a file handle, respond 
			// 'Internal error'
			OBEXOpCode responseCode;
			if (delegateReply != kOBEXSuccess) {
				responseCode = delegateReply;
			} else {
				responseCode = kOBEXResponseCodeInternalServerErrorWithFinalBit;
				[self reportError:kOBEXBadArgumentError 
						  summary:@"Error: delegate responded kOBEXSuccess but did not provide file handle for reading source data"];
			}
			
			if (debug) NSLog(DEBUG_NAME "Finish GET, refusing GET with response '%@'", 
							 [BBOBEXUtil describeServerResponse:responseCode]);
			[self sendGetResponse:responseCode withHeaders:NULL];
			[self completedGet:NO];
			
			if (reqHeadersDictRef) 
				CFRelease(reqHeadersDictRef);
			return;
		}	
		
		// proceeding with the GET request, make handler to read the GET body
		// data we will send to the client
		// (mGetFileHandle should have been set in notifyGetRequestWithName:type:)
		[mCurrentGetBodyHandler release];
		mCurrentGetBodyHandler = nil;
		mCurrentGetBodyHandler = [[BBOBEXFileReader alloc] initWithFile:mGetFileHandle
															  forSession:mOBEXSession
																isClient:NO];		
		// allow to send the first GET packet
		mNextGetDelegateReply = kOBEXSuccess;
	}	
	
	// finished reading headers
	if (reqHeadersDictRef) 
		CFRelease(reqHeadersDictRef);
	
	// now assemble a response for the GET
	
	// release previous GET response data
	if (mGetHeadersDataRef) {
		CFRelease(mGetHeadersDataRef);
		mGetHeadersDataRef = NULL;
	}
	
	// see if GET headers bytes really needs to be instance variable!!!!!!!!!!!!
	
	// get headers to be sent
	BOOL isLastPacket;
	unsigned bodyDataLength;
	mGetHeadersDataRef = [self buildGetResponseHeaders:&isLastPacket 
											bodyLength:&bodyDataLength];
	if (!mGetHeadersDataRef) {
		[self sendGetResponse:kOBEXResponseCodeInternalServerErrorWithFinalBit withHeaders:NULL];		
		[self reportError:kOBEXInternalError
				  summary:@"Error packing GET response headers, responded 'Internal error'"];
		[self completedGet:NO];		
		return;
	}

	// Delegate wants to continue the GET?
	if (mNextGetDelegateReply != kOBEXSuccess) {
		
		if (debug) NSLog(@"Delegate wants to stop GET, respond %@", [BBOBEXUtil describeServerResponse:mNextGetDelegateReply]);
		[self sendGetResponse:mNextGetDelegateReply withHeaders:NULL];
		[self completedGet:NO];
		
	} else {
		
		// send the GET response
		OBEXOpCode responseCode;
		if (isLastPacket) {
			responseCode = kOBEXResponseCodeSuccessWithFinalBit;
			if (debug) NSLog(DEBUG_NAME "Finish GET, respond kOBEXResponseCodeSuccessWithFinalBit");
		} else {
			responseCode = kOBEXResponseCodeContinueWithFinalBit;
			if (debug) NSLog(DEBUG_NAME "Continue GET, respond kOBEXResponseCodeContinueWithFinalBit");
		}
		
		BOOL responseSent = [self sendGetResponse:responseCode 
									  withHeaders:mGetHeadersDataRef];
		
		if (responseSent) {
			if (isLastPacket) {
				// finish the GET operation	
				[self completedGet:YES];
				
			} else {
			
				// tell delegate that more data was sent, delegate should inform us whether
				// the GET should continue
				mNextGetDelegateReply = [self notifyGetProgressByAmount:bodyDataLength];				
			}
		}
	}
}


#pragma mark -
#pragma mark SetPath

- (BOOL)sendSetPathResponse:(OBEXOpCode)responseCode
				withHeaders:(CFMutableDataRef)headersDataRef
{
	void *headers = (headersDataRef)? (void*)CFDataGetBytePtr(headersDataRef) : NULL;
	size_t headersLength = (headersDataRef)? CFDataGetLength(headersDataRef): 0;		
	
	OBEXError status;
	status = [mOBEXSession OBEXSetPathResponse:responseCode
							   optionalHeaders:headers
						 optionalHeadersLength:headersLength
								 eventSelector:callbackCommandReceived
								selectorTarget:self
										refCon:NULL];
	
	if (status != kOBEXSuccess) {
		[self reportError:status summary:@"OBEXSetPathResponse failed"];
		return NO;
	}
	return YES;
}

// This will try to read the significant headers and ask delegate 
// whether to continue.
// Returns delegate reply.
- (int)queryDelegateForSetPathRequest:(CFDictionaryRef)reqHeadersDictRef
					   requestFlags:(OBEXFlags)flags
{
	if (debug) NSLog(DEBUG_NAME "[queryDelegateForSetPathRequest] Entry\n");	
	
	CFStringRef nameStringRef = NULL;	
	
	if (reqHeadersDictRef) {
		// read name header
		if ((CFDictionaryGetCountOfKey(reqHeadersDictRef, kOBEXHeaderIDKeyName) > 0)) {
			nameStringRef = (CFStringRef)CFDictionaryGetValue(reqHeadersDictRef, kOBEXHeaderIDKeyName);
		}
	}
	
	NSString *name = (nameStringRef) ? (NSString *)nameStringRef : nil;
	if (debug) NSLog(DEBUG_NAME "Requested path name is: %@\n", name);
	
	return [self notifySetPathRequestWithName:name
										flags:flags];
}

- (void)handleSetPathCommand:(const OBEXSetPathCommandData*)eventData
{
	if (debug) NSLog(DEBUG_NAME @"[handleSetPathCommand] entry.");
	
	// get dictionary of header data
	CFDictionaryRef headersDictRef = 
		OBEXGetHeaders(eventData->headerDataPtr, eventData->headerDataLength);	
	
	// does delegate allow this SETPATH?
	int delegateReply = [self queryDelegateForSetPathRequest:headersDictRef
											  requestFlags:eventData->flags];
	
	OBEXOpCode responseCode = (delegateReply == kOBEXSuccess)? 
		kOBEXResponseCodeSuccessWithFinalBit : delegateReply;
	
	// send the response	
	if (debug) NSLog(DEBUG_NAME "Finish SETPATH, respond '%@' (0x%02x)", 
					 [BBOBEXUtil describeServerResponse:responseCode], responseCode);	
	BOOL responseSent = [self sendSetPathResponse:responseCode withHeaders:NULL];	
	
	if (responseSent && delegateReply == kOBEXSuccess) {
		[self notifySetPathSuccess];
	}	
}



#pragma mark -
#pragma mark Abort

- (BOOL)sendAbortResponse:(OBEXOpCode)responseCode
			  withHeaders:(CFMutableDataRef)headersDataRef
{	
	void *headers = (headersDataRef)? (void*)CFDataGetBytePtr(headersDataRef) : NULL;
	size_t headersLength = (headersDataRef)? CFDataGetLength(headersDataRef): 0;		
	
	OBEXError status;
	status = [mOBEXSession OBEXAbortResponse:responseCode
							 optionalHeaders:headers
					   optionalHeadersLength:headersLength
							   eventSelector:callbackCommandReceived
							  selectorTarget:self
									  refCon:NULL];
	
	if (status != kOBEXSuccess) {
		[self reportError:status summary:@"OBEXAbortResponse failed"];
		return NO;
	}
	return YES;
}

- (void)handleAbortCommand:(const OBEXAbortCommandData*)eventData
{
	if (debug) NSLog(DEBUG_NAME @"[handleAbortCommand] entry.");
	
	// Only need to abort for operations which require more than 1 packet.
	// E.g. you can't abort connection request cos you receive the connect
	// request and then send a response immediately and then the operation's
	// all over.	
	if (mCurrentOp != kOBEXOpCodePut && mCurrentOp != kOBEXOpCodeGet) {
		// refuse the request as there is nothing to abort
		[self reportError:kOBEXSessionBadRequestError 
				  summary:@"Got ABORT request but nothing to abort, responding kOBEXResponseCodeBadRequestWithFinalBit"];
		[self sendAbortResponse:kOBEXResponseCodeBadRequestWithFinalBit
					withHeaders:NULL];		
		return;
	}
	
	// does delegate allow this ABORT?
	int delegateReply = [self notifyAbortRequestForOp:mCurrentOp];
	
	OBEXOpCode responseCode;
	if (delegateReply == kOBEXSuccess) {
		responseCode = kOBEXResponseCodeSuccessWithFinalBit;

		switch (mCurrentOp) {
			case kOBEXOpCodePut:
				[self clientAbortedPut];
				break;
			case kOBEXOpCodeGet:
				[self clientAbortedGet];
				break;
		}
		
	} else {
		responseCode = delegateReply;
	}
	
	// send the response	
	if (debug) NSLog(DEBUG_NAME "Finish ABORT, respond '%@' (0x%02x)", 
					 [BBOBEXUtil describeServerResponse:responseCode], responseCode);
	BOOL responseSent = [self sendAbortResponse:responseCode withHeaders:NULL];	
	
	if (responseSent && delegateReply == kOBEXSuccess) {
		[self notifyAbortSuccess];
	}	
}


#pragma mark -
#pragma mark Process received client commands

/*
 Process received commands from an OBEX client.
 See OBEX.h for struct details.
 */
- (void)handleSessionEvent:(const OBEXSessionEvent *)event
{
	if (!mRunning) {
		if (debug) NSLog(DEBUG_NAME @"[handleSessionEvent] Ignoring event after session was closed: event->type = %x", event->type);
		return;
	}
	
	switch (event->type)
	{
		case (kOBEXSessionEventTypeConnectCommandReceived):
			[self handleConnectCommand:&event->u.connectCommandData];
			break;
			
		case (kOBEXSessionEventTypeDisconnectCommandReceived):
			[self handleDisconnectCommand:&event->u.disconnectCommandData];
			break;
			
		case (kOBEXSessionEventTypePutCommandReceived):
			[self handlePutCommand:&event->u.putCommandData];
			break;
			
		case (kOBEXSessionEventTypeAbortCommandReceived):
			[self handleAbortCommand:&event->u.abortCommandData];
			break;
			
		case (kOBEXSessionEventTypeGetCommandReceived):
			[self handleGetCommand:&event->u.getCommandData];
			break;
		 
		case (kOBEXSessionEventTypeSetPathCommandReceived):
			[self handleSetPathCommand:&event->u.setPathCommandData];
			break;
		 
		case (kOBEXSessionEventTypeError):
		{
			OBEXErrorData errData = event->u.errorData;
			if (debug) NSLog(DEBUG_NAME @"[handleSessionEvent:] kOBEXSessionEventTypeError (error=%d)", errData.error);
			
			NSString *errMsg;
			if (errData.error == kOBEXSessionTransportDiedError) {
				errMsg = @"bluetooth transport connection broken";
			} else {
				errMsg = [NSString stringWithFormat:@"OBEXSession error (error=%d)", errData.error];
			}
			
			// don't know how to deal with it, so just finish the current operation
			[self reportError:errData.error summary:errMsg];
			[self completedCurrentOp];
			break;
		}	
		default:
			if (debug) NSLog(DEBUG_NAME @"[handleSessionEvent] unknown command type (%x)", event->type);
			break;
	}
}


#pragma mark -

- (void)dealloc
{
	if (mOBEXSession && [mOBEXSession hasOpenTransportConnection]) {
		[mOBEXSession closeTransportConnection];
	}	
	
	[mPutFileHandle release];
	
	[mCurrentGetBodyHandler release];
	[mGetFileHandle release];
	if (mGetHeadersDataRef) 
		CFRelease(mGetHeadersDataRef);
	
	[mOBEXSession release];

	[super dealloc];
}


@end
