

from pypy.conftest import gettestobjspace
from pypy.translator.tool.cbuild import compile_c_module, \
     ExternalCompilationInfo
from pypy.module._ffi.interp_ffi import TYPEMAP

import os, sys, py

def setup_module(mod):
    if sys.platform != 'linux2':
        py.test.skip("Linux only tests by now")

class AppTestFfi:
    def prepare_c_example():
        from pypy.tool.udir import udir
        c_file = udir.join("xlib.c")
        c_file.write(py.code.Source('''
        #include <stdlib.h>
        #include <stdio.h>

        struct x
        {
           int x1;
           short x2;
           char x3;
           struct x* next;
        };

        void nothing()
        {
        }

        char inner_struct_elem(struct x *x1)
        {
           return x1->next->x3;
        }

        struct x* create_double_struct()
        {
           struct x* x1, *x2;

           x1 = (struct x*)malloc(sizeof(struct x));
           x2 = (struct x*)malloc(sizeof(struct x));
           x1->next = x2;
           x2->x2 = 3;
           return x1;
        }

        void free_double_struct(struct x* x1)
        {
            free(x1->next);
            free(x1);
        }
        
        const char *static_str = "xxxxxx";
        
        unsigned short add_shorts(short one, short two)
        {
           return one + two;
        }

        char get_char(char* s, unsigned short num)
        {
           return s[num];
        }

        char *char_check(char x, char y)
        {
           if (y == static_str[0])
              return static_str;
           return NULL;
        }

        int get_array_elem(int* stuff, int num)
        {
           return stuff[num];
        }

        struct x* get_array_elem_s(struct x** array, int num)
        {
           return array[num];
        }

        long long some_huge_value()
        {
           return 1LL<<42;
        }

        unsigned long long some_huge_uvalue()
        {
           return 1LL<<42;
        }

        long long pass_ll(long long x)
        {
           return x;
        }

        int* allocate_array()
        {
            int *res = (int*)malloc(sizeof(int));
            res[0] = 3;
            return res;
        }

        '''))
        return compile_c_module([c_file], 'x', ExternalCompilationInfo())
    prepare_c_example = staticmethod(prepare_c_example)
    
    def setup_class(cls):
        space = gettestobjspace(usemodules=('_ffi','struct'))
        cls.space = space
        cls.w_lib_name = space.wrap(cls.prepare_c_example())
        cls.w_sizes_and_alignments = space.wrap(dict(
            [(k, (v.c_size, v.c_alignment)) for k,v in TYPEMAP.iteritems()]))

    def test_libload(self):
        import _ffi
        _ffi.CDLL('libc.so.6')

    def test_getattr(self):
        import _ffi
        libc = _ffi.CDLL('libc.so.6')
        func = libc.ptr('rand', [], 'i')
        assert libc.ptr('rand', [], 'i') is func # caching
        assert libc.ptr('rand', [], 'l') is not func
        assert isinstance(func, _ffi.FuncPtr)
        raises(AttributeError, "libc.xxxxxxxxxxxxxx")

    def test_getchar(self):
        import _ffi
        lib = _ffi.CDLL(self.lib_name)
        get_char = lib.ptr('get_char', ['s', 'H'], 'c')
        assert get_char('dupa', 2) == 'p'
        assert get_char('dupa', 1) == 'u'
        raises(ValueError, "get_char('xxx', 2 ** 17)")
        raises(ValueError, "get_char('xxx', -1)")
        get_char = lib.ptr('get_char', ['z', 'H'], 'c')
        assert get_char('dupa', 2) == 'p'

    def test_returning_str(self):
        import _ffi
        lib = _ffi.CDLL(self.lib_name)
        char_check = lib.ptr('char_check', ['c', 'c'], 's')
        assert char_check('y', 'x') == 'xxxxxx'
        assert char_check('x', 'y') is None

    def test_short_addition(self):
        import _ffi
        lib = _ffi.CDLL(self.lib_name)
        short_add = lib.ptr('add_shorts', ['h', 'h'], 'H')
        assert short_add(1, 2) == 3

    def test_rand(self):
        import _ffi
        libc = _ffi.CDLL('libc.so.6')
        func = libc.ptr('rand', [], 'i')
        first = func()
        count = 0
        for i in range(100):
            res = func()
            if res == first:
                count += 1
        assert count != 100

    def test_pow(self):
        import _ffi
        libm = _ffi.CDLL('libm.so')
        pow = libm.ptr('pow', ['d', 'd'], 'd')
        assert pow(2.0, 2.0) == 4.0
        assert pow(3.0, 3.0) == 27.0
        assert pow(2, 2) == 4.0
        raises(TypeError, "pow('x', 2.0)")

    def test_strlen(self):
        import _ffi
        libc = _ffi.CDLL('libc.so.6')
        strlen = libc.ptr('strlen', ['s'], 'i')
        assert strlen("dupa") == 4
        assert strlen("zupa") == 4
        strlen = libc.ptr('strlen', ['P'], 'i')
        assert strlen("ddd\x00") == 3
        strdup = libc.ptr('strdup', ['s'], 's')
        assert strdup("xxx") == "xxx"

    def test_time(self):
        import _ffi
        libc = _ffi.CDLL('libc.so.6')
        time = libc.ptr('time', ['P'], 'l')
        assert time(None) != 0

    def test_gettimeofday(self):
        import _ffi
        struct_type = _ffi.Structure([('tv_sec', 'l'), ('tv_usec', 'l')])
        structure = struct_type()
        libc = _ffi.CDLL('libc.so.6')
        gettimeofday = libc.ptr('gettimeofday', ['P', 'P'], 'i')
        assert gettimeofday(structure, None) == 0
        struct2 = struct_type()
        assert gettimeofday(struct2, None) == 0
        assert structure.tv_usec != struct2.tv_usec
        assert (structure.tv_sec == struct2.tv_sec) or (structure.tv_sec == struct2.tv_sec - 1)
        raises(AttributeError, "structure.xxx")
        structure.free()

    def test_structreturn(self):
        import _ffi
        X = _ffi.Structure([('x', 'l')])
        x = X()
        x.x = 121
        Tm = _ffi.Structure([('tm_sec', 'i'),
                             ('tm_min', 'i'),
                             ('tm_hour', 'i'),
                             ("tm_mday", 'i'),
                             ("tm_mon", 'i'),
                             ("tm_year", 'i'),
                             ("tm_wday", 'i'),
                             ("tm_yday", 'i'),
                             ("tm_isdst", 'i')])
        libc = _ffi.CDLL('libc.so.6')
        gmtime = libc.ptr('gmtime', ['P'], 'P')
        t = Tm.fromaddress(gmtime(x))
        assert t.tm_year == 70
        assert t.tm_sec == 1
        assert t.tm_min == 2      
        x.free()

    def test_nested_structures(self):
        import _ffi
        lib = _ffi.CDLL(self.lib_name)
        inner = lib.ptr("inner_struct_elem", ['P'], 'c')
        X = _ffi.Structure([('x1', 'i'), ('x2', 'h'), ('x3', 'c'), ('next', 'P')])
        next = X(next=None, x3='x')
        x = X(next=next, x1=1, x2=2, x3='x')
        assert X.fromaddress(x.next).x3 == 'x'
        x.free()
        next.free()
        create_double_struct = lib.ptr("create_double_struct", [], 'P')
        x = create_double_struct()
        x = X.fromaddress(x)
        assert X.fromaddress(x.next).x2 == 3
        free_double_struct = lib.ptr("free_double_struct", ['P'], None)
        free_double_struct(x)
        

    def test_array(self):
        import _ffi
        lib = _ffi.CDLL(self.lib_name)
        A = _ffi.Array('i')
        get_array_elem = lib.ptr('get_array_elem', ['P', 'i'], 'i')
        a = A(10)
        a[8] = 3
        a[7] = 1
        a[6] = 2
        assert get_array_elem(a, 9) == 0
        assert get_array_elem(a, 8) == 3
        assert get_array_elem(a, 7) == 1
        assert get_array_elem(a, 6) == 2
        assert a[3] == 0
        a.free()
        a = A([1, 2, 3, 4])
        assert get_array_elem(a, 0) == 1
        assert a[3] == 4
        a.free()

    def test_array_of_structure(self):
        import _ffi
        lib = _ffi.CDLL(self.lib_name)
        A = _ffi.Array('P')
        X = _ffi.Structure([('x1', 'i'), ('x2', 'h'), ('x3', 'c'), ('next', 'P')])
        x = X(x2=3)
        a = A(3)
        a[1] = x
        get_array_elem_s = lib.ptr('get_array_elem_s', ['P', 'i'], 'P')
        ptr1 = get_array_elem_s(a, 0)
        assert ptr1 is None
        assert X.fromaddress(get_array_elem_s(a, 1)).x2 == 3
        assert get_array_elem_s(a, 1) == x.buffer
        x.free()
        a.free()

    def test_bad_parameters(self):
        import _ffi
        lib = _ffi.CDLL(self.lib_name)
        nothing = lib.ptr('nothing', [], None)
        assert nothing() is None
        raises(AttributeError, "lib.ptr('get_charx', [], None)")
        raises(ValueError, "lib.ptr('get_char', ['xx'], None)")
        raises(ValueError, "lib.ptr('get_char', ['x'], None)")
        raises(ValueError, "lib.ptr('get_char', [], 'x')")
        raises(ValueError, "_ffi.Structure(['x1', 'xx'])")
        raises(ValueError, _ffi.Structure, [('x1', 'xx')])
        raises(ValueError, "_ffi.Array('xx')")

    def test_implicit_structure(self):
        skip("Does not work yet")
        import _ffi
        lib = _ffi.CDLL(self.lib_name)
        X = _ffi.Structure([('x1', 'i'), ('x2', 'h'), ('x3', 'c'), ('next', 'self')])
        inner = lib.ptr("inner_struct_elem", [X], 'c')
        x = X(next=X(next=None, x3='x'), x1=1, x2=2, x3='x')
        assert x.next.x3 == 'x'
        assert inner(x) == 'x'
        create_double_struct = lib.ptr("create_double_struct", [], X)
        x = create_double_struct()
        assert x.next.x2 == 3
        

    def test_longs_ulongs(self):
        import _ffi
        lib = _ffi.CDLL(self.lib_name)
        some_huge_value = lib.ptr('some_huge_value', [], 'q')
        assert some_huge_value() == 1<<42
        some_huge_uvalue = lib.ptr('some_huge_uvalue', [], 'Q')
        assert some_huge_uvalue() == 1<<42
        x = lib.ptr('some_huge_value', ['Q'], None)
        raises(ValueError, "x(-1)")
        pass_ll = lib.ptr('pass_ll', ['q'], 'q')
        assert pass_ll(1<<42) == 1<<42
    
    def test_callback(self):
        skip("Not working")
        import _ffi
        libc = _ffi.CDLL('libc.so.6')
        to_sort = "kljhgfa"
        ll_to_sort = _ffi.Array('c')(to_sort)
        qsort = libc.ptr('qsort', ['P', 'i', 'i', 'P'], None)
        def compare(a, b):
            return a < b
        qsort(ll_to_sort, len(to_sort), 1,
              _ffi.CallbackPtr(compare, ['i', 'i'], 'i'))
        res = [ll_to_sort[i] for i in range(len(to_sort))]
        assert res == sorted(to_sort)

    def test_setattr_struct(self):
        import _ffi
        X = _ffi.Structure([('value1', 'i'), ('value2', 'i')])
        x = X(value1=1, value2=2)
        assert x.value1 == 1
        assert x.value2 == 2
        x.value1 = 3
        assert x.value1 == 3
        raises(AttributeError, "x.foo")
        raises(AttributeError, "x.foo = 1")
        x.free()

    def test_sizes_and_alignments(self):
        import _ffi
        for k, (s, a) in self.sizes_and_alignments.iteritems():
            assert _ffi.sizeof(k) == s
            assert _ffi.alignment(k) == a

    def test_array_addressof(self):
        import _ffi
        lib = _ffi.CDLL(self.lib_name)
        alloc = lib.ptr('allocate_array', [], 'P')
        A = _ffi.Array('i')
        a = A.fromaddress(alloc(), 1)
        assert a[0] == 3
        assert A.fromaddress(a.buffer, 1)[0] == 3
        # a.free() - don't free as ll2ctypes is complaining massively

    def test_shape(self):
        import _ffi
        A = _ffi.Array('i')
        a = A(1)
        assert a.shape is A
        a.free()
        S = _ffi.Structure([('v1', 'i')])
        s = S(v1=3)
        assert s.shape is S
        s.free()

    def test_negative_pointers(self):
        import _ffi
        A = _ffi.Array('P')
        a = A(1)
        a[0] = -1234
        a.free()
        
    def test_passing_raw_pointers(self):
        import _ffi
        lib = _ffi.CDLL(self.lib_name)
        A = _ffi.Array('i')
        get_array_elem = lib.ptr('get_array_elem', ['P', 'i'], 'i')
        a = A(1)
        a[0] = 3
        res = get_array_elem(a.buffer, 0)
        assert res == 3
        a.free()
