#-*- coding:utf-8 -*-

#  Pybik -- A 3 dimensional magic cube game.
#  Copyright © 2009-2011  B. Clausius <barcc@gmx.de>
#
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.


# Ported from GNUbik
# Original filename: drwBlock.c
# Original copyright and license: 1998, 2003 John Darrington, GPL3+


# This line makes pyrex happy
global __name__, __file__, __package__

import cython

from debug import debug, DEBUG_DRAW, DEBUG_PICK

debug('Importing module:', __name__)
debug('  from package:', __package__)
debug('  compiled:', cython.compiled)

#pxd from gl cimport *
#pxd from cube_c cimport *

#px-5
from OpenGL.GL import *
from cube import MATRIX_DIM, p_Vector, Matrix, MAX_BLOCKS
from cube import get_cube, get_faces, is_face_visible
from cube import init_vector, init_listuc_range, init_listf_range
from cube import _get_block_transform


#pxd cdef inline p_Vector glGetModelView(p_Vector m):
#pxd    glGetFloatv(GL_MODELVIEW_MATRIX, <float*>m)
#pxd    return m
#px-
def glGetModelView(unused_m):   return glGetFloatv(GL_MODELVIEW_MATRIX)
#pxd cdef inline void glMultMatrixM(p_Vector m):
#pxd    glMultMatrixf(<float*>m)
#px-
glMultMatrixM = glMultMatrixf


# We use a little bit of glut in debug mode 
if DEBUG_DRAW:
    from OpenGL import GLUT as glut
def glutInit(argv):
    if DEBUG_DRAW:
        glut.glutInit(len(argv), *argv[:])
def renderString(string):
    glut.glutStrokeString(glut.GLUT_STROKE_MONO_ROMAN, string)


# this macro produces +1 if i is even. -1 if i is odd 
# We use it to g the faces of the block from zero,  to the
# appropriate place 
#pxd cdef int SHIFT(int i)
def SHIFT(i):
    return (i%2) * 2 - 1


#********************
#* Set the normal vector for block/face.
#******

#pxd cdef void set_normal_vector_vj(int block, int face, int vj, int j)
def set_normal_vector_vj(block, face, vj, j):
    cython.declare(
        view = Matrix,
        view_ = p_Vector,
        dest = cython.p_float)
    #px-
    view = None
    view_ = glGetModelView(view)
    dest = get_faces(block)[face].normal
    transform_vj(dest, view_, vj, j)


#********************
#* Set the vector for block/face/quadrant to v.
#******/

#pxd cdef void set_quadrant_vector_vj(int block, int face, int quadrant, int vj, int j)
def set_quadrant_vector_vj(block, face, quadrant, vj, j):
    cython.declare(
        view = Matrix,
        view_ = p_Vector,
        dest = cython.p_float)
    #px-
    view = None
    view_ = glGetModelView(view)
    dest = get_faces(block)[face].quadrants[quadrant]
    transform_vj(dest, view_, vj, j)

#pxd cdef void transform_vj(float *r, Matrix M, float vj, int j)
def transform_vj(r, M, vj, j):
    i = cython.declare(cython.int)
    for i in range(MATRIX_DIM):
        r[i] = M[j][i] * vj
    

_AnimationState = cython.struct(
    running = cython.int,
    angle = cython.float,
    move_axis = cython.int,
    move_slice = cython.int,
    move_dir = cython.int,
)
anim_state = cython.declare(_AnimationState)
anim_state.running = 0
anim_state.angle = 0
anim_state.move_axis = 0
anim_state.move_slice = 0
anim_state.move_dir = 0

def start_animate(axis, slice_, dir_):
    anim_state.move_axis = axis
    anim_state.move_slice = slice_
    anim_state.move_dir = dir_
    anim_state.angle = 0.0
    anim_state.running = 1

@cython.locals(increment = cython.float)
def step_animate(increment):
    anim_state.angle -= increment
    return abs(anim_state.angle) < 90.0

def end_animate():
    anim_state.running = 0

#pxd cdef void draw_cube()
def draw_cube():
    #px-
    if DEBUG_DRAW:  draw_cube_debug()
    _draw_cube(True)

#pxd cdef void pick_cube()
def pick_cube():
    _draw_cube(False)

# render the cube
#pxd cdef void _draw_cube(bint render_mode)
def _draw_cube(render_mode):
    cython.declare(
        i = cython.int,
        axis_ = cython.int,
        angle = cython.float,
        unity = cython.float,
        M = p_Vector)
    
    for i in range(get_cube().number_blocks):
        glPushMatrix ()
        # Find out if this block is one of those currently being
        # turned.  If so,  j will be < turning_block_qty
        if anim_state.running:
            if get_cube().blocks[i].in_motion:
                # Blocks which are in motion,  need to be animated.
                # so we rotate them according to however much the
                # animation angle is
                angle = anim_state.angle
                
                if anim_state.move_dir:
                    unity = 1
                else:
                    unity = -1

                axis_ = anim_state.move_axis
                if axis_ >= 3:
                    axis_ -= 3
    
                if axis_ == 0:
                    glRotatef (angle,  unity, 0, 0)
                elif axis_ == 1:
                    glRotatef (angle, 0,  unity, 0)
                elif axis_ == 2:
                    glRotatef (angle, 0, 0,  unity)
            
        
        # place the block in its current position and
        # orientation
        M = _get_block_transform (i)
        glPushMatrix()
        glMultMatrixM(M)

        # and draw the block
        draw_block(0, i, render_mode)
        glPopMatrix ()
        
        glPopMatrix ()


##### experimental pick2
#pxd cdef void pick2_cube()
def pick2_cube():
    cython.declare(
        i = cython.int,
        axis = cython.int,
        mask = cython.int)
    
    # Rasterise only the exterior faces,  to speed things up 
    glEnable (GL_CULL_FACE)
    
    for i in range(6):
        glPushMatrix ()
        if i <= 1:
            #glTranslated (0, 0,  SHIFT (i))
            axis = 0
        elif i <= 3:
            #glTranslated (0,  SHIFT (i), 0)
            glRotatef (-90, 1, 0, 0)
            axis = 1
        else:
            #glTranslated (SHIFT (i), 0, 0)
            glRotatef (90, 0, 1, 0)
            axis = 2
        
        # make sure all  the sides are faced with their visible
        # surface pointing to the outside!!  
        if not (i % 2):
            glRotatef (180, 1, 0, 0)
        
        # draw the face,  iff it is visible 
        mask = 0x01 << i
        pick2_face(i, axis)
        
        glPopMatrix()
        break
#pxd cdef void pick2_face(int face, int axis)
def pick2_face(face, axis):
    # This polygon is drawn as four quadrants,  thus:
    #    _______
    #   |\     /|
    #   | \ 1 / |
    #   |  \ /  |
    #   | 0 \ 2 |
    #   |  / \  |
    #   | / 3 \ |
    #   |/____ \|
    #
    # The reason for this is to provide support for an enhanced selection
    # mechanism which can detect which edge of the face is being pointed to.
    cython.declare(
        col = cython.short,
        r = cython.uchar,
        g = cython.uchar,
        dim2 = cython.int)
    r = axis
    dim2 = get_cube().dimension2
    
    glEnableClientState(GL_VERTEX_ARRAY)
    glEnableClientState(GL_COLOR_ARRAY)
    
    glColorPointer(3, GL_UNSIGNED_BYTE, 0, model_data_pick2_colors)
    glVertexPointer(3, GL_FLOAT, 0, model_data_pick2_vertices)
    glDrawArrays(GL_POLYGON, 0, 2*dim2*4*3*3)
    
    glDisableClientState(GL_COLOR_ARRAY)
    glDisableClientState(GL_VERTEX_ARRAY)
#####


# render the block pointed to by attrib.
# if highlight is true,  then the outline is white,
# otherwise it is black 
#pxd cdef void draw_block(int highlight, int block_id, bint)
def draw_block(highlight, block_id, render_mode):
    cython.declare(
        i = cython.int,
        mask = cython.int)
    
    # Rasterise only the exterior faces,  to speed things up 
    glEnable (GL_CULL_FACE)
    
    for i in range(6):
        glPushMatrix ()
        if i <= 1:
            glTranslated (0, 0,  SHIFT (i))
        elif i <= 3:
            glTranslated (0,  SHIFT (i), 0)
            glRotatef (-90, 1, 0, 0)
        else:
            glTranslated (SHIFT (i), 0, 0)
            glRotatef (90, 0, 1, 0)
        
        # make sure all  the sides are faced with their visible
        # surface pointing to the outside!!  
        if not (i % 2):
            glRotatef (180, 1, 0, 0)
        
        # draw the face,  iff it is visible 
        mask = 0x01 << i
        if is_face_visible(block_id, i):
            if render_mode and not DEBUG_PICK:
                draw_face(i, highlight, block_id)
                draw_label(i, block_id)
            else:
                pick_face(i, block_id)
            if render_mode and DEBUG_DRAW:
                draw_face_debug(i, block_id)
        elif anim_state.running:
            if render_mode:
                draw_face(i, highlight, block_id)
        
        glPopMatrix()
    

#pxd ctypedef float Vertex[3]
#pxd ctypedef float FaceTexPos[4*2]
#pxd ctypedef FaceTexPos *p_FaceTexPos
#pxd ctypedef FaceTexPos BlockTexPos[6]
#pxd ctypedef BlockTexPos BlocksTexPos[MAX_BLOCKS]
#pxd ctypedef BlockTexPos *p_BlockTexPos
#px-3
FaceTexPos = lambda: [0] * 4 * 2
BlockTexPos = lambda: [FaceTexPos() for __i in range(6)]
BlocksTexPos = lambda: [BlockTexPos() for __i in range(MAX_BLOCKS)]
#TODO: don't store the texpos of hidden faces
cython.declare(
    model_data_vertex = cython.float[3*4],
    model_data_pick = cython.float[3*12],
    model_data_label = cython.float[3*4],
    model_data_label180 = cython.float[3*4],
    model_data_texpos_mosaic = BlocksTexPos,
    model_data_texpos_tiled = BlocksTexPos,
    model_data_pick2_vertices = cython.float[6*10*10*4*3*3],
    model_data_pick2_colors = cython.uchar[6*10*10*4*3*3])
#px-8
model_data_vertex = [0]*3*4
model_data_pick = [0]*3*12
model_data_label = [0]*3*4
model_data_label180 = [0]*3*4
model_data_texpos_mosaic = BlocksTexPos()
model_data_texpos_tiled = BlocksTexPos()
model_data_pick2_vertices = [0]*(6*10*10*4*3*3)
model_data_pick2_colors = [0]*(6*10*10*4*3*3)
    
def init_model():
    cython.declare(
        dim = cython.int,
        dim2 = cython.int,
        b = cython.int,
        f = cython.int,
        v = cython.int,
        i = cython.int,
        block = cython.int,
        face = cython.int,
        x = cython.int,
        y = cython.int,
        xpos = cython.int,
        ypos = cython.int,
        size = cython.float)
    init_vector(model_data_vertex, [-1, 1, 0,   -1,-1, 0,    1,-1, 0,    1, 1, 0,])
    init_vector(model_data_pick, [   0, 0, 0,   -1, 1, 0,   -1,-1, 0,
                                     0, 0, 0,    1, 1, 0,   -1, 1, 0,
                                     0, 0, 0,    1,-1, 0,    1, 1, 0,
                                     0, 0, 0,   -1,-1, 0,    1,-1, 0])
    
    lratio = cython.declare(cython.float, 0.9)
    init_vector(model_data_label, [-lratio, lratio, 0.01,
                                   -lratio,-lratio, 0.01,
                                    lratio,-lratio, 0.01,
                                    lratio, lratio, 0.01])
    
    init_vector(model_data_label180, [ lratio,-lratio, 0.01,
                                       lratio, lratio, 0.01,
                                      -lratio, lratio, 0.01,
                                      -lratio,-lratio, 0.01])
    
    dim = get_cube().dimension
    dim2 = get_cube().dimension2
    for block in range(get_cube().number_blocks):
        for face in range(6):
            size = 1.0 / dim
            if face & 6 == 0: # 0 or 1
                xpos = block % dim
                ypos = block % dim2 / dim
                ypos = dim - ypos - 1
            elif face & 6 == 2: # 2 or 3
                xpos = block % dim
                ypos = block / dim2
            else:#if face & 6 == 4: # 4 or 5
                xpos = block / dim2
                ypos = block % dim2 / dim
                ypos = dim - ypos - 1
            
            # Invert positions as necessary 
            if face == 0:
                xpos = dim - xpos - 1
            elif face == 2:
                xpos = dim - xpos - 1
            elif face == 5:
                xpos = dim - xpos - 1
            
            init_vector(model_data_texpos_mosaic[block][face], [size*xpos, size*ypos,
                                                                size*xpos, size*(ypos+1),
                                                                size*(xpos+1), size*(ypos+1),
                                                                size*(xpos+1), size*ypos])
            
            init_vector(model_data_texpos_tiled[block][face], [0, 0,  0, 1,  1, 1,  1, 0])
    # face 0
    i = 0
    for y in range(dim):
        for x in range(dim):
            init_listf_range(i, i+36, model_data_pick2_vertices,
            [-dim+1+x,-dim+1+y,-dim,  -dim+1-1+x,-dim+1+1+y,-dim,  -dim+1-1+x,-dim+1-1+y,-dim,
             -dim+1+x,-dim+1+y,-dim,  -dim+1+1+x,-dim+1+1+y,-dim,  -dim+1-1+x,-dim+1+1+y,-dim,
             -dim+1+x,-dim+1+y,-dim,  -dim+1+1+x,-dim+1-1+y,-dim,  -dim+1+1+x,-dim+1+1+y,-dim,
             -dim+1+x,-dim+1+y,-dim,  -dim+1-1+x,-dim+1-1+y,-dim,  -dim+1+1+x,-dim+1-1+y,-dim,])
            init_listuc_range(i, i+36, model_data_pick2_colors,
                            [1,y,1,  1,y,1,  1,y,1,
                             0,x,1,  0,x,1,  0,x,1,
                             1,y,2,  1,y,2,  1,y,2,
                             0,x,2,  0,x,2,  0,x,2])
            i += 36
    # face 1
    for y in range(dim):
        for x in range(dim):
            init_listf_range(i, i+36, model_data_pick2_vertices,
            [-dim+1+2*x,-dim+1+2*y,dim,  -dim+1-1+2*x,-dim+1+1+2*y,dim,  -dim+1-1+2*x,-dim+1-1+2*y,dim,
             -dim+1+2*x,-dim+1+2*y,dim,  -dim+1+1+2*x,-dim+1+1+2*y,dim,  -dim+1-1+2*x,-dim+1+1+2*y,dim,
             -dim+1+2*x,-dim+1+2*y,dim,  -dim+1+1+2*x,-dim+1-1+2*y,dim,  -dim+1+1+2*x,-dim+1+1+2*y,dim,
             -dim+1+2*x,-dim+1+2*y,dim,  -dim+1-1+2*x,-dim+1-1+2*y,dim,  -dim+1+1+2*x,-dim+1-1+2*y,dim,])
            init_listuc_range(i, i+36, model_data_pick2_colors,
                            [1,y,1,  1,y,1,  1,y,1,
                             0,x,1,  0,x,1,  0,x,1,
                             1,y,2,  1,y,2,  1,y,2,
                             0,x,2,  0,x,2,  0,x,2])
            i += 36
    assert i == 2*dim2*36
    
# render the surface of the cube, that is the plastic material the thing is constructed from
#pxd cdef void draw_face(int face, bint highlight, int block_id)
def draw_face(face, highlight, block_id):
    if highlight:
        glColor3fv(_color_white)
    else:
        glColor3fv(_color_black)
    
    set_normal_vector_vj (block_id, face, -1, 2)
    
    glEnableClientState(GL_VERTEX_ARRAY)
    glVertexPointer(3, GL_FLOAT, 0, model_data_vertex)
    glDrawArrays(GL_POLYGON, 0, 4)
    glDisableClientState(GL_VERTEX_ARRAY)
    
# render the colours (ie the little sticky labels)
#pxd cdef void draw_label(int face, int block_id)
def draw_label(face, block_id):
    cython.declare(
        image_segment_size = cython.float,
        xpos = cython.int,
        ypos = cython.int)
    
    glColor3fv (face_rendering[face].color)
    
    if face_rendering[face].texName == -1:
        glDisable (GL_TEXTURE_2D)
    else:
        glEnable (GL_TEXTURE_2D)
        if face_rendering[face].type_ == IMAGED:
            glTexEnvi (GL_TEXTURE_ENV,  GL_TEXTURE_ENV_MODE,  GL_DECAL)
        else:
            glTexEnvi (GL_TEXTURE_ENV,  GL_TEXTURE_ENV_MODE,  GL_MODULATE)
        glBindTexture (GL_TEXTURE_2D,  face_rendering[face].texName)
    
    glEnableClientState(GL_TEXTURE_COORD_ARRAY)
    glEnableClientState(GL_VERTEX_ARRAY)
    if face_rendering[face].type_ == IMAGED and face_rendering[face].distr == _MOSAIC:
        glTexCoordPointer(2, GL_FLOAT, 0, model_data_texpos_mosaic[block_id][face])
    else: # TILED 
        glTexCoordPointer(2, GL_FLOAT, 0, model_data_texpos_tiled[block_id][face])
    if face % 2:
        glVertexPointer(3, GL_FLOAT, 0, model_data_label)
    else:
        glVertexPointer(3, GL_FLOAT, 0, model_data_label180)
    glDrawArrays(GL_POLYGON, 0, 4)
    glDisableClientState(GL_TEXTURE_COORD_ARRAY)
    glDisableClientState(GL_VERTEX_ARRAY)
    
    glDisable (GL_TEXTURE_2D)
    
#pxd cdef void pick_face (int face, int block_id)
def pick_face(face, block_id):
    # This polygon is drawn as four quadrants,  thus:
    #    _______
    #   |\     /|
    #   | \ 1 / |
    #   |  \ /  |
    #   | 0 \ 2 |
    #   |  / \  |
    #   | / 3 \ |
    #   |/____ \|
    #
    # The reason for this is to provide support for an enhanced selection
    # mechanism which can detect which edge of the face is being pointed to.
    cython.declare(
        col = cython.short,
        r = cython.uchar,
        g = cython.uchar)
    col = (block_id << 3) | face
    r = (col >> 8) + 68
    g = col & 0xFF
    
    glEnableClientState(GL_VERTEX_ARRAY)
    
    set_normal_vector_vj (block_id, face, -1, 2)
    
    glColor3ub(r, g, 1)
    glVertexPointer(3, GL_FLOAT, 0, model_data_pick)
    glDrawArrays(GL_POLYGON, 0, 3)
    set_quadrant_vector_vj (block_id, face, 0, 1, 0)
    
    glColor3ub(r, g, 0x55)
    glVertexPointer(3, GL_FLOAT, 0, model_data_pick)
    glDrawArrays(GL_POLYGON, 3, 3)
    set_quadrant_vector_vj (block_id, face, 1, -1, 1)
    
    glColor3ub(r, g, 0xaa)
    glVertexPointer(3, GL_FLOAT, 0, model_data_pick)
    glDrawArrays(GL_POLYGON, 6, 3)
    set_quadrant_vector_vj (block_id, face, 2, -1, 0)
    
    glColor3ub(r, g, 0xff)
    glVertexPointer(3, GL_FLOAT, 0, model_data_pick)
    glDrawArrays(GL_POLYGON, 9, 3)
    set_quadrant_vector_vj (block_id, face, 3, 1, 1)
    
    glDisableClientState(GL_VERTEX_ARRAY)
    

def draw_cube_debug():
    offset = 1.4 * get_cube().dimension
    
    # Show the directions of the axes
    glColor3f(1, 1, 1)
    
    # X axis
    glPushMatrix()
    glTranslatef(-offset, -offset, -get_cube().dimension)
    glBegin(GL_LINES)
    glVertex3f(0, 0, 0)
    glVertex3f(2*get_cube().dimension, 0, 0)
    glEnd()
    
    glRasterPos3d(offset*1.1, 0, 0)
    #px-
    glut.glutBitmapCharacter(glut.GLUT_BITMAP_9_BY_15, ord('X'))
    glPopMatrix()
    
    # Y axis
    glPushMatrix()
    glTranslatef(-offset, -offset, -get_cube().dimension)
    glBegin(GL_LINES)
    glVertex3f(0, 0, 0)
    glVertex3f(0, 2*get_cube().dimension, 0)
    glEnd()
    
    glRasterPos3d(0.1*offset,  offset, 0)
    #px-
    glut.glutBitmapCharacter(glut.GLUT_BITMAP_9_BY_15, ord('Y'))
    glPopMatrix()
    
    # Z axis
    glPushMatrix()
    glTranslatef(-offset, -offset, -get_cube().dimension)
    glBegin(GL_LINES)
    glVertex3f(0, 0, 0)
    glVertex3f(0, 0, 2*get_cube().dimension)
    glEnd()
    
    glRasterPos3d(0.1*offset, 0,  offset)
    #px-
    glut.glutBitmapCharacter(glut.GLUT_BITMAP_9_BY_15, ord('Z'))
    glPopMatrix ()

#pxd cdef void draw_face_debug(int face, int block_id)
def draw_face_debug(face, block_id):
    # render the block number 
    glPushMatrix()
    glColor3f(0, 0, 0)
    
    glTranslatef(-1,  -0.8, 0.02)
    glScalef(0.0075, 0.0075, 0.0075)
    
    renderString("%d" % block_id)
    glPopMatrix()
    
    # render the face number,  a little bit smaller
    # so we can see what's what. 
    glPushMatrix()
    glTranslatef(-0.5, 0.0, 0.02)
    glScalef(0.0075, 0.0075, 0.0075)
    renderString(['U0','D1','L2','R3','F4','B5'][face])
    glPopMatrix()


#pxd cdef enum:
#pxd    _TILED = 0
#pxd    _MOSAIC = 1
#px-
if True: _TILED = 0; _MOSAIC = 1

#pxd cdef enum:
#pxd    UNDEFINED = 0
#pxd    COLORED = 1
#pxd    IMAGED = 2
#px-
if True: UNDEFINED = 0; COLORED = 1; IMAGED = 2


colors = [
            [1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0],
            [0.0, 0.0, 1.0],
            [0.0, 1.0, 1.0],
            [1.0, 0.0, 1.0],
            [1.0, 1.0, 0.0],
        ]
color_black = [0.1, 0.1, 0.1]
color_white = [1.0, 1.0, 1.0]
#pxd ctypedef float color_t[3]
#px-
color_t = list
cython.declare(
        _color_black = color_t,
        _color_white = color_t,
    )
#px-2
_color_black = [0] * 3
_color_white = [0] * 3


#pxd cdef struct _FaceRendering:
#pxd    color_t color
#pxd    int type_
#pxd    int texName
#pxd    int distr
#px-
class _FaceRenderings: pass

#pxd ctypedef _FaceRendering _FaceRenderings[6]
face_rendering = cython.declare(_FaceRenderings)
#px-
face_rendering = [_FaceRenderings() for __i in range(6)]

#pxd cdef void init_face_rendering()
def init_face_rendering():
    cython.declare(
        i = cython.int,
        j = cython.int,)
    for i in range(6):
        #px-
        face_rendering[i].color = [0]*3
        for j in range(3):
            face_rendering[i].color[j] = colors[i][j]
    for j in range(3):
        _color_black[j] = color_black[j]
        _color_white[j] = color_white[j]
init_face_rendering()


def face_rendering_set(swatch, red=-1, green=-1, blue=-1, facetype=UNDEFINED, texname=None, distr=-1):
    if red >= 0:    face_rendering[swatch].color[0] = red
    if green >= 0:  face_rendering[swatch].color[1] = green
    if blue >= 0:   face_rendering[swatch].color[2] = blue
    
    if facetype > 0: face_rendering[swatch].type_ = facetype
    
    if texname is not None:
        face_rendering[swatch].texName = texname
    
    if distr >= 0:  face_rendering[swatch].distr = distr


