package HTMLIO;

# Copyright (c) 2000, FundsXpress Financial Network, Inc.
# This library is free software released under the GNU Lesser General
# Public License, Version 2.1.  Please read the important licensing and
# disclaimer information included below.

# $Id: HTMLIO.pm,v 1.1 2000/11/23 23:36:14 muaddib Exp $

use strict;

#$Id: HTMLIO.pm,v 1.1 2000/11/23 23:36:14 muaddib Exp $

=head1 NAME

HTMLIO - virtual class which handles output and managemant of data

=head1 SYNOPSIS

  ...
  use HTMLIO::Types;
  use AtomicData::Types;

  my $html = init_html_page();
  my $prev_input = new HTMLIO::Text
       ('example1', 'AtomicData::Text', $cgi->param('example1'));
  my ($success,$problems) = $prev_input->verify();
  if (!$success) {
    $html .= 'You entered an invalid name.';
    return;
  }
  my $input = new HTMLIO::Select('example2', 'AtomicData::Integer', 1, 2, 3);
  $input->set_attributes({multiple => 'true',
			    size => '2'});
  $html .= 'You entered '.$prev_input->html_view().
     ' as your name, please select your disrict:<br>';
  $html .= $input->html_input();
  ...

=head1 DESCRIPTION

This class provides much of the functionality needed by HTML aware
data encapsulating objects. To use this package, you must also have
access to and be familiar with Error.pm, available on CPAN. Error.pm
provides exception like error handling functionality. This is a
virtual class, however, since it does not implement
C<html_input>. Most children will only need to implement this
method. Some will have to overide or add methods of their own,
however.

Note that when discussing data here, we mean not only primitives such
as integer, float, character, string, etc., but also data best
distinguished by its function, e.g., social security numbers, routing
and trace numbers, street addresses, U.S. zipcodes, etc. We do not,
however, mean complex data such as programming objects.

HTMLIO allows one to store, retrieve, verify and manage data being
used in HTML pages. The methods are fairly straight forward. The
actual data encapsulating is handled by another object class which has
a 'used' relationship with HTMLIO. HTMLIO serves mostly as a
special I/O go between for the user and this data encapsulating class,
but does, however, do some verification of data and storage of error
messages. The necessary interface required of the data encapsulating
class by HTMLIO is speced out below. For further details on data
verification, see the SEE ALSO section below.

Note that the data returned by retrieval functions is not guaranteed
to be exactly the same as the data entered. It may be canonicalized
(for its type), or reformated in some way, but will always be
equivalent to the original in intended value as per the encapsulating
data type specified. This will occur in one of the following two
cases: 1) some sort of formating request that would affect the value
of the underlying data encapsulator has been performed, or 2) the
underlying data type chooses to return data in its canonical form in
some cases.

=head1 DATA SPEC

HTMLIO relies on atomic data encapsulators. It will internally
maintain a homogenous list of these corresponding to received
values. All data received by a single HTMLIO object should be of the
same general type (be able to be stored in homogeneous encapsulators).

The data encapsulating class must have the following methods:

 ($object) new ($value)
  accepts a value and returns the data encapsulating object

 () set_value ($value)
  sets the value encapsulated by that object

 ($value) value ()
  returns the value stored by that object

 () verify () 
  an exception throwing fuction (using or compatible with Error.pm)
  that will throw an exception of a single type when in error. The
  exception thrown must return a reference to a list detailing the
  errors found when C<text> is called on it.

 () set_format (\%hash)
  accepts a hash of key value pairs that specifies formating
  preferences of the value maintained; the meaningful keys and values
  are specified by the data encapsulating object itself

 () format (bool) 
  causes the data to be formated (see above); if set_format has never
  been called or has been reset, then the behavior is unspecifide
  except that the new data must be equivalent, if not in exactly the
  same form as the old WRT its type; this must affect the return of
  value called on the datatype when appropriate. This function will
  raise an exception if it is unable to cannocalize the current data
  form (see below), unless bool is set to true in which case errors
  will be ignored and the old form will be maintained unchanged..

 () parameters (\%hash)
  will set parameters to constrain what is considered valid data for
  the data type; the key/value pairs are specified by the data type
  itself; this will affect verification of data

 ($value) canonicalize ($value, bool)
  will accept a value proper the data encapsulators type and return it
  in a set canonical form (used in conjuction with test_equality
  below); will raise an exception if it is unable to canonicalize
  unless the bool is set to true, in which case errors will be ignored
  and the old form will be maintanined unchanged.

 ($bool) test_equality ($val1, $val2)
  will return true if $val1 and $val2 are canonically equivalent WRT
  the data types purpose; that is if we have a data type encapsulating
  U.S. dollars, then $1 aned 1.00 would be equivalent

=head1 SEE ALSO

AtomicData: for information on the data encapsulating object.

=head1 FUTURE DEVELOPMENTS

It might be useful to add functions on this type to morph to children
so that users could at runtime morph from one class of HTML I/O
objects to another.

=cut

use strict;

use HTMLIO::Utils qw(:all);

=head1 METHODS

In order to derive a non-virtual subclass, the public fucntion
C<html_input> must be implemented.

=head2 new

 class
 (HTMLIO $obj) new
   (string $name, string $data_class, variable @values)

DESCRIPTION:

Returns an HTMLIO object. This function will populate the internal
list of data with the values passed in via the @values array. If no
values are passed in, then the internal data will be populated with a
single data encapsulator encapsulating a null string. The data
encapsulating object class to be used to encapsulate is specified in
C<$data_class>. C<$name> is the value to be placed in the name
attribute of an input, select, submit, etc., HTML tag.

=cut

sub new {
  my ($class,$attributes) = @_;
  $attributes ||= {};

  my $this = {};
  bless $this, ref($class)||$class;

  $this->{_attributes} = $attributes;

  return $this;
}

=head2 set_attributes

 instance
 () set_attributes (\%attributes);

DESCRIPTION:

Each key/value pair will be placed in the string created by
C<html_input> as attribute/value pairs.

=cut

sub set_attributes {
  my ($this, $hashref) = @_;
  $this->{_attributes} = $hashref;
}

=head2 attributes

 instance
 (\%attributes) attributes ()

DESCRIPTION:

Will return the attribute specifying hash.

=cut

sub attributes {
  my ($this) = @_;
  return $this->{_attributes};
}

=head2 html_view

 instance
 (string $string) html_view ([bool $raw_p, int \@int, string $join])

DESCPRIPTION:

Will return a string of the values stored. If a reference to a list of
indexes is passed in, then it will return us only that data referenced
by the index numbers to form its string. See the note in DESCRIPTION
for ordering of data members. If C<$join> is passed in, then it will
join the list of data members by the string. This function can be used
in conjuction with C<data_count()> to iterate over member
data. C<$raw_p>, if true, indicates that the value of the
C<raw_value()> function (and not the C<value()> function on
encapsulated data should be used. This will have the affect of
returning the data as the user entered it without any sort of
formatting that may have occurred. The default value, if not passed,
is false.q

Example:

Assume the AtomicData::List to be populated with values 1, 2, 3, then:

  print $data->html_view()."\n";
  print $data->html_view(0, undef, ', ')."\n";
  print $data->html_view(0, [1]);

would yield:

 123
 1, 2, 3
 2

=cut

sub html_view {
  my ($this, $data, $raw_p, $slice, $join) = @_;
  $join ||= "\n";

  my @data;
  $raw_p ? (@data = ($data->raw_value())) : (@data = ($data->value()));

  $slice ||= [0...$#data];

  return join $join, map { html_escape($data[$_]) } @$slice;
}

=head2 raw_view

 instance
 (string $raw) raw_view ([int \@slice, string $join])

DESCRIPTION:

Like C<html_view> except that the data is not HTML escaped, it is
passed raw.

=cut

sub raw_view {
  my ($this, $data, $raw_p, $slice, $join) = @_;

  my @data;
  $raw_p ? (@data = ($data->raw_value())) : (@data = ($data->value()));

  $slice ||= [0...$#data];

  return join $join, map { $data[$_] } @$slice;
}

=head2 html_hidden

 instance
 (string $hidden) html_hidden ([int \@slice, string $join])

DESCRIPTION:

Will return the value indicated in hidden fields, otherwise acts like
html_view.

=cut

sub html_hidden {
  my ($this, $name, $data, $slice, $join) = @_;

  return stnd_html_input
    ($name,'hidden',[ $data->value() ],$slice,$join);
}

=head2 html_input

 instance
 (string $html) html_input ([variable \@index, string $join])

DESCRIPTION:

Will return the appropriat HTML necessary to set up user input through
a textfield.  C<@index> specifies which data members (by index, see
C<HTMLIO> for ordering) to set for defualt or which keys to set for
default if this is constrained.*

*: This type does not support constraining here at this time

=cut

sub html_input {
  my ($this, $name, $data, $slice, $join) = @_;
  return stnd_html_input
    ($name,
     'text', 
     [ $data->value() ], 
     $slice, 
     $join, 
     $this->attributes());
}

=head2 set_display

 instance
  () set_display (bool $disp)

DESCRIPTION:

If set to true, then the value of the key/value pair will be shown to
the user when C<html_view> is called. If false, then the key (the HTML
value).

=cut

sub set_display {
  my ($this,$disp) = @_;
  $this->{_display} = $disp;
}

=head2 free_internals

 instance
 () free_internals ()

=cut

sub free_internals {
  my ($this) = @_;

  delete $this->{_attributes};

  return;
}

1;
__END__

=head1 BUGS

No known bugs, but this does not mean no bugs exist.

=head1 SEE ALSO

L<AtomicData>, L<HTMLIO>, L<Field>.

=head1 COPYRIGHT

 PSP - Perl Server Pages
 Copyright (c) 2000, FundsXpress Financial Network, Inc.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2 of the License, or (at your option) any later version.

 BECAUSE THIS LIBRARY IS LICENSED FREE OF CHARGE, THIS LIBRARY IS
 BEING PROVIDED "AS IS WITH ALL FAULTS," WITHOUT ANY WARRANTIES
 OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING, WITHOUT
 LIMITATION, ANY IMPLIED WARRANTIES OF TITLE, NONINFRINGEMENT,
 MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, AND THE
 ENTIRE RISK AS TO SATISFACTORY QUALITY, PERFORMANCE, ACCURACY,
 AND EFFORT IS WITH THE YOU.  See the GNU Lesser General Public
 License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

=cut
