package PSP::Conf;

# Copyright (c) 2000, FundsXpress Financial Network, Inc.
# This library is free software released under the GNU Lesser General
# Public License, Version 2.1.  Please read the important licensing and
# disclaimer information included below.

# $Id: Conf.pm,v 1.5 2001/02/19 10:42:53 muaddie Exp $

use strict;

=head1 NAME

 PSP::Conf - configuration for the PSP driver.

=head1 SYNOPSIS

 #more later

=head1 DESCRIPTION

more later.

=cut

use Exporter;
use PSP::Loader;

# here are the variables we can export.
#
use vars qw(@ISA @VARS @EXPORT_OK);
BEGIN {
  @PSP::Conf::ISA = qw(Exporter);
  @PSP::Conf::EXPORT_OK =
    qw(
       $psp_stuck_seconds
       $psp_driver_log
       $psp_top
       $psp_do_auto_die
       $psp_auto_die_fname
       $psp_loader
       $psp_pile_dir
       $psp_processes_to_spawn
       $psp_requests_per_process
       $psp_save_or_restore_env
       $psp_cfg_fname
       $psp_dump_dir
      );
  $PSP::Conf::EXPORT_TAGS{all} = \@EXPORT_OK;
}
use vars @PSP::Conf::EXPORT_OK;

=head1 GLOBALS

Global variables can be overridden by the following techniques:

 1) by an environment variable.
 2) by exporting that variable from this package and modifying it.

=head2 $psp_stuck_seconds

C<$psp_stuck_seconds> indicates the number of seconds before we will log a
warning and dump debug information to the piledriver log.
This indicates the number of seconds before we will log a warning and
dump debug information to the piledriver log if the LOG_DRIVER 
environmental variable is defined.  

By default, this is 300 seconds (5 minutes).

=head2 $psp_top

    The top of the PSP tree; used to default many filenames
head2 $psp_do_auto_die and $psp_auto_die_fname

Set C<$psp_do_auto_die> to true if to true the driver should die when a
required pile has changed.  C<$psp_auto_die_fname> is the name of the file
whose stat(2) result had changed since it was modified.

By default, this feature is off.

=head2 $psp_loader

C<$psp_loader> is a tree structure that associates branches with classes.  
The default loader is defined below with its auto_populate() method.

=head2 $psp_pile_dir

C<$psp_pile_dir> is the default directory where the loader expects to find
its piles.

=head2 $psp_dump_dir

C<$psp_dump_dir> is a directory where requests and outputs should be dumped.
If undef, nothing is dumped.

=head2 $psp_processes_to_spawn

C<$psp_processes_to_spawn> is the number of processes PSP::ProcManager
should manage.  The default value is 0 to deactivate this service.

=head2 $psp_requests_per_process

C<$psp_requests_per_process> is the number of processes PSP::ProcManager
should manage.  The default is 0 to mean an unlimited number of requests
per process.

=head2 $psp_save_or_restore_env

C<$psp_save_or_restore_env> is the name of the file to which the environment
is dumped in a web served CGI request, and from which the environment is
restored from the command-line.  Default is "" to mean no save or restore
occurs.

=cut

sub import {
  #print STDERR "import called with (@_)\n";

  defined $psp_cfg_fname or
    $psp_cfg_fname = $ENV{PSP_CONF} || "/etc/psp.conf";

  # read the config file.
  get_config($psp_cfg_fname);
  # Make sure we have a $psp_top
  die "Configuration file must set psp_top\n" unless $psp_top;
  my $top = $psp_top;
  # Initialize some configuration variables.
  defined $psp_stuck_seconds or
    $psp_stuck_seconds = $ENV{PILEDRIVER_STUCK_SECONDS} || 5*60;
  defined $psp_driver_log or
    $psp_driver_log = $ENV{PILEDRIVER_LOG} || $top."/logs/piledriver.log";
  defined $psp_pile_dir or
    $psp_pile_dir = $ENV{PILEDRIVER_PILE_DIR} || $top."/share/piles";
  defined $psp_do_auto_die or 
    $psp_do_auto_die = $ENV{PILEDRIVER_DO_AUTO_DIE} || 0;
  defined $psp_processes_to_spawn or
    $psp_processes_to_spawn = $ENV{PILEDRIVER_PROCESSES_TO_SPAWN} || 0;
  defined $psp_requests_per_process or
    $psp_requests_per_process = $ENV{PILEDRIVER_REQUESTS_PER_PROCESS} || 0;
  defined $psp_save_or_restore_env or
    $psp_save_or_restore_env = $ENV{PILEDRIVER_SAVE_OR_RESTORE_ENV} || "";

  # Initialize the loader if we haven't already.
  if (! defined $psp_loader) {
    # We can ssume Loader exists because we are no longer in a BEGIN block.
    $psp_loader = new PSP::Loader('Pile',"",$psp_pile_dir);
    # auto_populate is clean and will not throw status.
    $psp_loader->auto_populate();
  }

  my $pkg = shift;
  my $callpkg = caller($Exporter::ExportLevel);
  Exporter::export $pkg, $callpkg, @_;
}

# load the PSP config file if it exists.

use vars qw(@cfg_stat);
sub get_config {
  my ($fname) = @_;
  $fname ||= $psp_cfg_fname;

  my @stat = stat $fname;
  @stat or warn("$fname: $!"), return;

  if (!@cfg_stat or 
      ($cfg_stat[0] != $stat[9] or
       $cfg_stat[1] != $stat[7])) {
    @cfg_stat = ($stat[9],$stat[7]);

    # read the config file if it exists.
    load_config($fname) or return;
  }
  return 1;
}

sub load_config {
  my ($fname) = @_;

  # if we fail to open file, warn, and return undef.
  #print STDERR "reading $fname\n";
  if (! open FILE, $fname) {
    warn "open: $fname: $!\n";
    return;
  }

  # get the text from that file.
  my $cfg_text = join("",<FILE>);
  close FILE;

  # evaluate that config file in this package.
  my $ret = eval "package PSP::Conf;\n".$cfg_text;

  # report errors in config file.
  if ($@) {
    warn $@;
    return;
  }

  # return true on success.
  return 1;
}

1;
__END__

=head1 BUGS

No known bugs, but this does not mean no bugs exist.

=head1 SEE ALSO

L<PSP::Loader>

=head1 COPYRIGHT

 PSP - Perl Server Pages
 Copyright (c) 2000, FundsXpress Financial Network, Inc.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2 of the License, or (at your option) any later version.

 BECAUSE THIS LIBRARY IS LICENSED FREE OF CHARGE, THIS LIBRARY IS
 BEING PROVIDED "AS IS WITH ALL FAULTS," WITHOUT ANY WARRANTIES
 OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING, WITHOUT
 LIMITATION, ANY IMPLIED WARRANTIES OF TITLE, NONINFRINGEMENT,
 MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, AND THE
 ENTIRE RISK AS TO SATISFACTORY QUALITY, PERFORMANCE, ACCURACY,
 AND EFFORT IS WITH THE YOU.  See the GNU Lesser General Public
 License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

=cut
