#!/bin/bash

# official repository / репозиторий официальной Psi
GIT_REPO_PSI=git://git.psi-im.org/psi.git

# do svn up on patches / обновлять патчи из svn
FORCE_PATCH_UPDATE=${FORCE_PATCH_UPDATE:-1}

# skip patches which applies with errors / пропускать глючные патчи
SKIP_INVALID_PATCH=${SKIP_INVALID_PATCH:-0}

# do git pull on psi git working copy on start
# обновляться с репозитория перед сборкой
FORCE_REPO_UPDATE=${FORCE_REPO_UPDATE:-1}

# log of applying patches / лог применения патчей
PATCH_LOG=${PATCH_LOG:-/dev/null}

# configure options / опции скрипта configure
CONF_OPTS="${CONF_OPTS:---prefix=/usr}"

#######################
# FUNCTIONS / ФУНКЦИИ #
#######################
# Exit with error message
die() { echo; echo " !!!ERROR: ${1}"; exit 1; }

# error status
_epatch_assert() { local _pipestatus=${PIPESTATUS[*]}; [[ ${_pipestatus// /} -eq 0 ]] ; }

#smart patcher
spatch() {
  popts=""
  PATCH_TARGET="${1}"

  if [ ! -f $PATCH_TARGET ]
  then
    echo "patch '${PATCH_TARGET}' doesn't exist"
    exit 1
  fi

  echo -n " * applying ${PATCH_TARGET}..."

  count=0
  while [ "${count}" -lt 5 ]; do
    echo " * trying patch -p${count} ${popts} --dry-run -f" >> $PATCH_LOG
    if (cat ${PATCH_TARGET} | patch -p${count} ${popts} --dry-run -f ; _epatch_assert) >> $PATCH_LOG 2>&1
    then
      cat ${PATCH_TARGET} | patch -p${count} ${popts} >> $PATCH_LOG 2>&1
      _epatch_assert
      if [ "$?" -ne 0 ]
      then
        echo "A dry-run of patch command succeeded, but actually"
        echo "applying the patch failed!"
        count=5
      fi
      break;
    fi
    count=$((count + 1))
  done

  if [ $count -ne 5 ]
  then
     echo "done"
     return 0
  else
     echo "failed"
     echo "see log here ${PATCH_LOG}"
     return 1
  fi
}

check_env() {
  echo -n "testing environment.."
  v=`git --version 2>/dev/null` || die "You should install Git first. / Сначала установите Git"
  v=`svn --version 2>/dev/null` || die "You should install subversion first. / Сначала установите subversion"
  v=`gmake --version 2>/dev/null`
  MAKE="gmake"
  if [ -z "$v" ]; then
    echo "gmake not found! / gmake не найден!"
    echo -n "trying just make / пробуем просто make.."
    v=`make --version 2>/dev/null`
    MAKE="make"
  fi
  [ ! -z "$v" ] || die "You should install GNU Make first / Сначала установите GNU Make"
  v=`qconf --version |grep qconf` || die "You should install qconf(http://delta.affinix.com/qconf/) / Сначала установите qconf"
  echo "OK"
}

prepare_workspace() {
  echo -n "init directories.."
  if [ ! -d "${PSI_DIR}" ]
  then
    mkdir "${PSI_DIR}" || die "can't create work directory ${PSI_DIR}"
  fi
  rm -rf "${PSI_DIR}"/build
  [ -d "${PSI_DIR}"/build ] && die "can't delete old build directory ${PSI_DIR}/build"
  mkdir "${PSI_DIR}"/build || die "can't create build directory ${PSI_DIR}/build"
  echo "OK"
}

fetch_sources() {
  cd "${PSI_DIR}"
  if [ -d "git/.git" ]
  then
    echo "Starting updating.."
    cd git
    if [ $FORCE_REPO_UPDATE != 0 ]; then
      git pull || die "git update failed"
    else
      echo "Update disabled in options"
    fi
  else
    echo "New fresh repo.."
    git clone "${GIT_REPO_PSI}" git || die "git clone failed"
  fi
  git submodule update --init --merge || die "git submodule update failed"

  echo "downloading patches.."
  cd "${PSI_DIR}"
  if [ -d patches ]
  then
    [ $FORCE_PATCH_UPDATE != 0 ] && ( svn up patches || die "patches update failed" )
  else
    svn co http://psi-dev.googlecode.com/svn/trunk/patches/ patches \
  	|| die "patches checkout failed"
  fi

  echo "downloading iconset.."
  cd "${PSI_DIR}"
  if [ -d psiplus_icons ]
  then
    svn up psiplus_icons || die "psiplus icons update failed"
  else
    svn co http://psi-dev.googlecode.com/svn/trunk/iconsets/psiplus/default psiplus_icons \
  	|| die "psiplus icons checkout failed"
  fi
}

prepare_sources() {
  echo "Exporting sources"
  cd "${PSI_DIR}"/git
  git archive --format=tar master | ( cd "${PSI_DIR}/build" ; tar xf - )
  cd iris
  git archive --format=tar master | ( cd "${PSI_DIR}/build/iris" ; tar xf - )
  cd "${PSI_DIR}"/git/src/libpsi
  git archive --format=tar master | ( cd "${PSI_DIR}/build/src/libpsi" ; tar xf - )

  cd "${PSI_DIR}"
  rev=`svnversion patches`
  rm patches/*win32* 2>/dev/null
  rm patches/*psi-dirty-check.diff 2>/dev/null
  PATCHES=`ls -1 patches/*diff 2>/dev/null`
  cd "${PSI_DIR}/build"
  for p in $PATCHES; do
     spatch "${PSI_DIR}/${p}"
     if [ "$?" != 0 ]
     then
       [ $SKIP_INVALID_PATCH != 0 ] && echo "skip invalid patch"
       [ $SKIP_INVALID_PATCH == 0 ] && die "can't continue due to patch failed"
     fi
  done
  sed "s/.xxx/.${rev}/" -i src/applicationinfo.cpp
  sed '/<!--\(.*plugins.*\)-->/a<dep type="plugins"><disabled/></dep>' -i psi.qc

  # prepare icons
  mkdir "${PSI_DIR}/build/iconsets/psiplus"
  svn export "${PSI_DIR}/psiplus_icons" "${PSI_DIR}/build/iconsets/psiplus/default"
  rm -rf "${PSI_DIR}/build/iconsets/psiplus/default/variants"
}

src_compile() {
  qconf
  ./configure ${CONF_OPTS} || die "configure failed"
  $MAKE || die "make failed"
}

pkg_install() {
  echo "Installing. it may require to enter password of owner if install directory.."
  if [ `stat -c "%u" "${INSTALL_ROOT}"` = $UID ]
  then
    $MAKE INSTALL_ROOT="${INSTALL_ROOT}" install || die "make install failed"
  else
    su -c "$MAKE INSTALL_ROOT=\"${INSTALL_ROOT}\" install" `stat -c "%U" "${INSTALL_ROOT}"` || die "make install failed"
  fi
}
